/**
 * @file IWblDbusWifiSetupTestProxyIf.h
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief Wbl DBUS Test Proxy Interface for WifiSetup.
 *
 * @copyright (C) 2016 - 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Interface definition for Wbl DBUS test proxy for WifiSetup.
 */

#ifndef _I_WBL_DBUS_WIFI_SETUP_TEST_PROXY_IF_H_
#define _I_WBL_DBUS_WIFI_SETUP_TEST_PROXY_IF_H_

#include "CcDbusIfTypes.h"
#include <boost/shared_ptr.hpp>

namespace ccdbusif {

// class forward declarations
class IWblDbusWifiSetupCallbackIf;

/**
 * Test proxy interface class for WifiSetup.
 */
class IWblDbusWifiSetupTestProxyIf
{
public:
   /**
    * Destructor.
    */
   virtual ~IWblDbusWifiSetupTestProxyIf() {}

   /**
    * Create proxy.
    *
    * @param[in] callback: callback handler
    * @param[in] port: port
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    */
   virtual void createProxy(IWblDbusWifiSetupCallbackIf& callback, const ::std::string& port, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM) = 0;

   /**
    * Destroy proxy.
    *
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    */
   virtual void destroyProxy(const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM) = 0;

   /**
    * Destroy all proxies.
    */
   virtual void destroyAllProxies(void) = 0;

   /**
    * Send DeAuthenticateStation Request.
    *
    * @param[in] callback: callback handler
    * @param[in] objPath: object path of wifi setup
    * @param[in] macAddress: Mac Address of the station to be deAuthenticated.
    * @param[in] deAuthenticationMode: DeAuthenticate only one associated station as specified by the MacAddress or deAuthenticate all of the associated stations (in such case station MAC address shall "not" be specified).
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendDeAuthenticateStationRequest(IWblDbusWifiSetupCallbackIf& callback, const ::std::string& objPath, const ::std::string& macAddress, const ::std::string& deAuthenticationMode) = 0;

   /**
    * Send ManageBandwidth Request.
    *
    * @param[in] callback: callback handler
    * @param[in] objPath: object path of wifi setup
    * @param[in] allocate: yes/no
    * @param[in] address: address: e.g. IP address or MAC address
    * @param[in] bandwidth: bandwidth
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendManageBandwidthRequest(IWblDbusWifiSetupCallbackIf& callback, const ::std::string& objPath, bool allocate, const ::std::string& address, const unsigned int bandwidth) = 0;

   /**
    * Send SetRestrictions Request.
    *
    * @param[in] callback: callback handler
    * @param[in] objPath: object path of wifi setup
    * @param[in] technology: technology
    * @param[in] restriction: restriction
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendSetRestrictionsRequest(IWblDbusWifiSetupCallbackIf& callback, const ::std::string& objPath, const ::std::string& technology, const ::std::string& restriction) = 0;

   /**
    * Send Mode Get.
    *
    * @param[in] callback: callback handler
    * @param[in] objPath: object path of wifi setup
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendModeGet(IWblDbusWifiSetupCallbackIf& callback, const ::std::string& objPath) = 0;

   /**
    * Send Interface Get.
    *
    * @param[in] callback: callback handler
    * @param[in] objPath: object path of wifi setup
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendInterfaceGet(IWblDbusWifiSetupCallbackIf& callback, const ::std::string& objPath) = 0;

   /**
    * Send MacAddress Get.
    *
    * @param[in] callback: callback handler
    * @param[in] objPath: object path of wifi setup
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendMacAddressGet(IWblDbusWifiSetupCallbackIf& callback, const ::std::string& objPath) = 0;

   /**
    * Send PowerState Get.
    *
    * @param[in] callback: callback handler
    * @param[in] objPath: object path of wifi setup
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendPowerStateGet(IWblDbusWifiSetupCallbackIf& callback, const ::std::string& objPath) = 0;

   /**
    * Send PowerFailureReason Get.
    *
    * @param[in] callback: callback handler
    * @param[in] objPath: object path of wifi setup
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendPowerFailureReasonGet(IWblDbusWifiSetupCallbackIf& callback, const ::std::string& objPath) = 0;

   /**
    * Send Frequency Get.
    *
    * @param[in] callback: callback handler
    * @param[in] objPath: object path of wifi setup
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendFrequencyGet(IWblDbusWifiSetupCallbackIf& callback, const ::std::string& objPath) = 0;

   /**
    * Send CurrentOperatingChannel Get.
    *
    * @param[in] callback: callback handler
    * @param[in] objPath: object path of wifi setup
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendCurrentOperatingChannelGet(IWblDbusWifiSetupCallbackIf& callback, const ::std::string& objPath) = 0;

   /**
    * Send APConfig Get.
    *
    * @param[in] callback: callback handler
    * @param[in] objPath: object path of wifi setup
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendAPConfigGet(IWblDbusWifiSetupCallbackIf& callback, const ::std::string& objPath) = 0;

   /**
    * Send Restrictions Get.
    *
    * @param[in] callback: callback handler
    * @param[in] objPath: object path of wifi setup
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendRestrictionsGet(IWblDbusWifiSetupCallbackIf& callback, const ::std::string& objPath) = 0;

   /**
    * Send SupportedFrequencies Get.
    *
    * @param[in] callback: callback handler
    * @param[in] objPath: object path of wifi setup
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendSupportedFrequenciesGet(IWblDbusWifiSetupCallbackIf& callback, const ::std::string& objPath) = 0;

   /**
    * Send ConnectionStatus Get.
    *
    * @param[in] callback: callback handler
    * @param[in] objPath: object path of wifi setup
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendConnectionStatusGet(IWblDbusWifiSetupCallbackIf& callback, const ::std::string& objPath) = 0;
};

} //ccdbusif

#endif //_I_WBL_DBUS_WIFI_SETUP_TEST_PROXY_IF_H_
