/**
 * @file IWblDbusWifiSetupSendRequestIf.h
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief Wbl DBUS Send Request Interface for WifiSetup.
 *
 * @copyright (C) 2016 - 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Interface definition for Wbl DBUS send request for WifiSetup.
 */

#ifndef _I_WBL_DBUS_WIFI_SETUP_SEND_REQUEST_IF_H_
#define _I_WBL_DBUS_WIFI_SETUP_SEND_REQUEST_IF_H_

#include "CcDbusIfTypes.h"

namespace ccdbusif {

// class forward declarations
class IDestroyAllProxies;
class IWblDbusWifiSetupCallbackIf;

/**
 * Send request interface class for WifiSetup.
 */
class IWblDbusWifiSetupSendRequestIf
{
public:
   /**
    * Destructor.
    */
   virtual ~IWblDbusWifiSetupSendRequestIf() {}

   /**
    * Get destroy interface.
    *
    * @return = destroy interface
    */
   virtual IDestroyAllProxies* getDestroyer(void) = 0;

   /**
    * Set callback handler.
    *
    * @param[in] callbackIf: callback handler
    * @param[in] enableProxy: flag to enable/disable proxy
    * @param[in] callbackId: callback id
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    */
   virtual void setCallbackIf(IWblDbusWifiSetupCallbackIf* callbackIf, const bool enableProxy, const unsigned int callbackId, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM) = 0;

   /**
    * Create proxy interface.
    *
    * @param[in] callbackId: callback id
    * @param[in] createDirectly: flag to trigger direct proxy creation or via work item request
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    */
   virtual void createProxyIf(const unsigned int callbackId, const bool createDirectly = false, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM) = 0;

   /**
    * Destroy proxy interface.
    *
    * @param[in] callbackId: callback id
    * @param[in] destroyDirectly: flag to trigger direct proxy destruction or via work item request
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    */
   virtual void destroyProxyIf(const unsigned int callbackId, const bool destroyDirectly = false, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM) = 0;

   /**
    * Send DeAuthenticateStation Request.
    *
    * @param[in] callbackId: callback id
    * @param[in] objPath: object path of wifi setup
    * @param[in] macAddress: Mac Address of the station to be deAuthenticated.
    * @param[in] deAuthenticationMode: DeAuthenticate only one associated station as specified by the MacAddress or deAuthenticate all of the associated stations (in such case station MAC address shall "not" be specified).
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendDeAuthenticateStationRequest(const unsigned int callbackId, const ::std::string& objPath, const ::std::string& macAddress, const ::std::string& deAuthenticationMode) = 0;

   /**
    * Send ManageBandwidth Request.
    *
    * @param[in] callbackId: callback id
    * @param[in] objPath: object path of wifi setup
    * @param[in] allocate: yes/no
    * @param[in] address: address: e.g. IP address or MAC address
    * @param[in] bandwidth: bandwidth
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendManageBandwidthRequest(const unsigned int callbackId, const ::std::string& objPath, bool allocate, const ::std::string& address, const unsigned int bandwidth) = 0;

   /**
    * Send SetRestrictions Request.
    *
    * @param[in] callbackId: callback id
    * @param[in] objPath: object path of wifi setup
    * @param[in] technology: technology
    * @param[in] restriction: restriction
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendSetRestrictionsRequest(const unsigned int callbackId, const ::std::string& objPath, const ::std::string& technology, const ::std::string& restriction) = 0;

   /**
    * Send Mode Get.
    *
    * @param[in] callbackId: callback id
    * @param[in] objPath: object path of wifi setup
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendModeGet(const unsigned int callbackId, const ::std::string& objPath) = 0;

   /**
    * Send Interface Get.
    *
    * @param[in] callbackId: callback id
    * @param[in] objPath: object path of wifi setup
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendInterfaceGet(const unsigned int callbackId, const ::std::string& objPath) = 0;

   /**
    * Send MacAddress Get.
    *
    * @param[in] callbackId: callback id
    * @param[in] objPath: object path of wifi setup
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendMacAddressGet(const unsigned int callbackId, const ::std::string& objPath) = 0;

   /**
    * Send PowerState Get.
    *
    * @param[in] callbackId: callback id
    * @param[in] objPath: object path of wifi setup
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendPowerStateGet(const unsigned int callbackId, const ::std::string& objPath) = 0;

   /**
    * Send PowerFailureReason Get.
    *
    * @param[in] callbackId: callback id
    * @param[in] objPath: object path of wifi setup
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendPowerFailureReasonGet(const unsigned int callbackId, const ::std::string& objPath) = 0;

   /**
    * Send Frequency Get.
    *
    * @param[in] callbackId: callback id
    * @param[in] objPath: object path of wifi setup
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendFrequencyGet(const unsigned int callbackId, const ::std::string& objPath) = 0;

   /**
    * Send CurrentOperatingChannel Get.
    *
    * @param[in] callbackId: callback id
    * @param[in] objPath: object path of wifi setup
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendCurrentOperatingChannelGet(const unsigned int callbackId, const ::std::string& objPath) = 0;

   /**
    * Send APConfig Get.
    *
    * @param[in] callbackId: callback id
    * @param[in] objPath: object path of wifi setup
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendAPConfigGet(const unsigned int callbackId, const ::std::string& objPath) = 0;

   /**
    * Send Restrictions Get.
    *
    * @param[in] callbackId: callback id
    * @param[in] objPath: object path of wifi setup
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendRestrictionsGet(const unsigned int callbackId, const ::std::string& objPath) = 0;

   /**
    * Send SupportedFrequencies Get.
    *
    * @param[in] callbackId: callback id
    * @param[in] objPath: object path of wifi setup
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendSupportedFrequenciesGet(const unsigned int callbackId, const ::std::string& objPath) = 0;

   /**
    * Send ConnectionStatus Get.
    *
    * @param[in] callbackId: callback id
    * @param[in] objPath: object path of wifi setup
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendConnectionStatusGet(const unsigned int callbackId, const ::std::string& objPath) = 0;
};

} //ccdbusif

#endif //_I_WBL_DBUS_WIFI_SETUP_SEND_REQUEST_IF_H_
