/**
 * @file IWblDbusConflictManagementTestProxyIf.h
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief Wbl DBUS Test Proxy Interface for ConflictManagement.
 *
 * @copyright (C) 2016 - 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Interface definition for Wbl DBUS test proxy for ConflictManagement.
 */

#ifndef _I_WBL_DBUS_CONFLICT_MANAGEMENT_TEST_PROXY_IF_H_
#define _I_WBL_DBUS_CONFLICT_MANAGEMENT_TEST_PROXY_IF_H_

#include "CcDbusIfTypes.h"
#include <boost/shared_ptr.hpp>

namespace ccdbusif {

// class forward declarations
class IWblDbusConflictManagementCallbackIf;

/**
 * Test proxy interface class for ConflictManagement.
 */
class IWblDbusConflictManagementTestProxyIf
{
public:
   /**
    * Destructor.
    */
   virtual ~IWblDbusConflictManagementTestProxyIf() {}

   /**
    * Create proxy.
    *
    * @param[in] callback: callback handler
    * @param[in] port: port
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    */
   virtual void createProxy(IWblDbusConflictManagementCallbackIf& callback, const ::std::string& port, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM) = 0;

   /**
    * Destroy proxy.
    *
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    */
   virtual void destroyProxy(const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM) = 0;

   /**
    * Destroy all proxies.
    */
   virtual void destroyAllProxies(void) = 0;

   /**
    * Send PrepareSetup Request.
    *
    * @param[in] callback: callback handler
    * @param[in] wiFiMode: WiFi mode (NOTE: this is only one of the possible parameters; if interface is needed then extend)
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendPrepareSetupRequest(IWblDbusConflictManagementCallbackIf& callback, const ::std::string& wiFiMode) = 0;

   /**
    * Send DeActivateSetup Request.
    *
    * @param[in] callback: callback handler
    * @param[in] wiFiMode: WiFi mode
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendDeActivateSetupRequest(IWblDbusConflictManagementCallbackIf& callback, const ::std::string& wiFiMode) = 0;

   /**
    * Send WiFiStationModeUsage Request.
    *
    * @param[in] callback: callback handler
    * @param[in] application: application
    * @param[in] usage: usage e.g. download
    * @param[in] holding: started using / finished flag
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendWiFiStationModeUsageRequest(IWblDbusConflictManagementCallbackIf& callback, const ::std::string& application, const ::std::string& usage, bool holding) = 0;

   /**
    * Send ResolveConflict Request.
    *
    * @param[in] callback: callback handler
    * @param[in] conflictid: conflictid
    * @param[in] action: action
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendResolveConflictRequest(IWblDbusConflictManagementCallbackIf& callback, int32 conflictid, int32 action) = 0;

   /**
    * Send ActiveWiFiSetups Get.
    *
    * @param[in] callback: callback handler
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendActiveWiFiSetupsGet(IWblDbusConflictManagementCallbackIf& callback) = 0;

   /**
    * Send WiFiPriorityList Get.
    *
    * @param[in] callback: callback handler
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendWiFiPriorityListGet(IWblDbusConflictManagementCallbackIf& callback) = 0;

   /**
    * Send WiFiPriorityList Set.
    *
    * @param[in] callback: callback handler
    * @param[in] wiFiPriorityList: WiFi priority list
    */
   virtual void sendWiFiPriorityListSet(IWblDbusConflictManagementCallbackIf& callback, const ::std::vector< ::std::string >& wiFiPriorityList) = 0;

   /**
    * Send ConfigurationTimer Get.
    *
    * @param[in] callback: callback handler
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendConfigurationTimerGet(IWblDbusConflictManagementCallbackIf& callback) = 0;

   /**
    * Send ConfigurationTimer Set.
    *
    * @param[in] callback: callback handler
    * @param[in] configurationTimer: configuration timer
    */
   virtual void sendConfigurationTimerSet(IWblDbusConflictManagementCallbackIf& callback, int32 configurationTimer) = 0;
};

} //ccdbusif

#endif //_I_WBL_DBUS_CONFLICT_MANAGEMENT_TEST_PROXY_IF_H_
