/**
 * @file IWblDbusConflictManagementCallbackIf.h
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief Wbl DBUS Callback Interface for ConflictManagement.
 *
 * @copyright (C) 2016 - 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Interface definition for Wbl DBUS callback for ConflictManagement.
 */

#ifndef _I_WBL_DBUS_CONFLICT_MANAGEMENT_CALLBACK_IF_H_
#define _I_WBL_DBUS_CONFLICT_MANAGEMENT_CALLBACK_IF_H_

#include "CcDbusIfTypes.h"

#include "org/bosch/wbl/conflictmanagement.h"

using namespace ::org::bosch::wbl::conflictmanagement;

namespace ccdbusif {

/**
 * Callback interface class for ConflictManagement.
 */
class IWblDbusConflictManagementCallbackIf
{
public:
   /**
    * Destructor.
    */
   virtual ~IWblDbusConflictManagementCallbackIf() {}

   // "ServiceAvailableIF" implementation --- start
   /**
    * Available signal.
    *
    * @param[in] available: available flag
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    */
   virtual void onAvailableCb(const bool available, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM) = 0;

   /**
    * Unavailable signal.
    *
    * @param[in] available: available flag
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    */
   virtual void onUnavailableCb(const bool available, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM) = 0;
   // "ServiceAvailableIF" implementation --- end

   // org/bosch/wbl/conflictmanagementProxy implementation --- start
   /**
    * onPrepareSetupError.
    *
    * @param[in] error: error signal / response
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    * @param[in] interfaceName: interface name
    */
   virtual void onPrepareSetupErrorCb(const ::boost::shared_ptr< PrepareSetupError >& error, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM, const ::std::string& interfaceName = ::std::string()) = 0;

   /**
    * onPrepareSetupResponse.
    *
    * @param[in] response: response signal / response
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    * @param[in] interfaceName: interface name
    */
   virtual void onPrepareSetupResponseCb(const ::boost::shared_ptr< PrepareSetupResponse >& response, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM, const ::std::string& interfaceName = ::std::string()) = 0;

   /**
    * onDeActivateSetupError.
    *
    * @param[in] error: error signal / response
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    * @param[in] interfaceName: interface name
    */
   virtual void onDeActivateSetupErrorCb(const ::boost::shared_ptr< DeActivateSetupError >& error, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM, const ::std::string& interfaceName = ::std::string()) = 0;

   /**
    * onDeActivateSetupResponse.
    *
    * @param[in] response: response signal / response
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    * @param[in] interfaceName: interface name
    */
   virtual void onDeActivateSetupResponseCb(const ::boost::shared_ptr< DeActivateSetupResponse >& response, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM, const ::std::string& interfaceName = ::std::string()) = 0;

   /**
    * onWiFiStationModeUsageError.
    *
    * @param[in] error: error signal / response
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    * @param[in] interfaceName: interface name
    */
   virtual void onWiFiStationModeUsageErrorCb(const ::boost::shared_ptr< WiFiStationModeUsageError >& error, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM, const ::std::string& interfaceName = ::std::string()) = 0;

   /**
    * onWiFiStationModeUsageResponse.
    *
    * @param[in] response: response signal / response
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    * @param[in] interfaceName: interface name
    */
   virtual void onWiFiStationModeUsageResponseCb(const ::boost::shared_ptr< WiFiStationModeUsageResponse >& response, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM, const ::std::string& interfaceName = ::std::string()) = 0;

   /**
    * onResolveConflictError.
    *
    * @param[in] error: error signal / response
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    * @param[in] interfaceName: interface name
    */
   virtual void onResolveConflictErrorCb(const ::boost::shared_ptr< ResolveConflictError >& error, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM, const ::std::string& interfaceName = ::std::string()) = 0;

   /**
    * onResolveConflictResponse.
    *
    * @param[in] response: response signal / response
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    * @param[in] interfaceName: interface name
    */
   virtual void onResolveConflictResponseCb(const ::boost::shared_ptr< ResolveConflictResponse >& response, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM, const ::std::string& interfaceName = ::std::string()) = 0;

   /**
    * onConflictDetectedError.
    *
    * @param[in] error: error signal / response
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    * @param[in] interfaceName: interface name
    */
   virtual void onConflictDetectedErrorCb(const ::boost::shared_ptr< ConflictDetectedError >& error, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM, const ::std::string& interfaceName = ::std::string()) = 0;

   /**
    * onConflictDetectedSignal.
    *
    * @param[in] signal: signal signal / response
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    * @param[in] interfaceName: interface name
    */
   virtual void onConflictDetectedSignalCb(const ::boost::shared_ptr< ConflictDetectedSignal >& signal, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM, const ::std::string& interfaceName = ::std::string()) = 0;

   /**
    * onConflictResolvedError.
    *
    * @param[in] error: error signal / response
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    * @param[in] interfaceName: interface name
    */
   virtual void onConflictResolvedErrorCb(const ::boost::shared_ptr< ConflictResolvedError >& error, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM, const ::std::string& interfaceName = ::std::string()) = 0;

   /**
    * onConflictResolvedSignal.
    *
    * @param[in] signal: signal signal / response
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    * @param[in] interfaceName: interface name
    */
   virtual void onConflictResolvedSignalCb(const ::boost::shared_ptr< ConflictResolvedSignal >& signal, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM, const ::std::string& interfaceName = ::std::string()) = 0;

   /**
    * onActiveWiFiSetupsError.
    *
    * @param[in] error: error signal / response
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    * @param[in] interfaceName: interface name
    */
   virtual void onActiveWiFiSetupsErrorCb(const ::boost::shared_ptr< ActiveWiFiSetupsError >& error, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM, const ::std::string& interfaceName = ::std::string()) = 0;

   /**
    * onActiveWiFiSetupsUpdate.
    *
    * @param[in] update: update signal / response
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    * @param[in] interfaceName: interface name
    */
   virtual void onActiveWiFiSetupsUpdateCb(const ::boost::shared_ptr< ActiveWiFiSetupsUpdate >& update, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM, const ::std::string& interfaceName = ::std::string()) = 0;

   /**
    * onWiFiPriorityListError.
    *
    * @param[in] error: error signal / response
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    * @param[in] interfaceName: interface name
    */
   virtual void onWiFiPriorityListErrorCb(const ::boost::shared_ptr< WiFiPriorityListError >& error, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM, const ::std::string& interfaceName = ::std::string()) = 0;

   /**
    * onWiFiPriorityListUpdate.
    *
    * @param[in] update: update signal / response
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    * @param[in] interfaceName: interface name
    */
   virtual void onWiFiPriorityListUpdateCb(const ::boost::shared_ptr< WiFiPriorityListUpdate >& update, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM, const ::std::string& interfaceName = ::std::string()) = 0;

   /**
    * onConfigurationTimerError.
    *
    * @param[in] error: error signal / response
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    * @param[in] interfaceName: interface name
    */
   virtual void onConfigurationTimerErrorCb(const ::boost::shared_ptr< ConfigurationTimerError >& error, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM, const ::std::string& interfaceName = ::std::string()) = 0;

   /**
    * onConfigurationTimerUpdate.
    *
    * @param[in] update: update signal / response
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    * @param[in] interfaceName: interface name
    */
   virtual void onConfigurationTimerUpdateCb(const ::boost::shared_ptr< ConfigurationTimerUpdate >& update, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM, const ::std::string& interfaceName = ::std::string()) = 0;
   // org/bosch/wbl/conflictmanagementProxy implementation --- end
};

} //ccdbusif

#endif //_I_WBL_DBUS_CONFLICT_MANAGEMENT_CALLBACK_IF_H_
