/**
 * @file PropertiesGetAllDbusProxyIf.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief Properties Get ALL DBUS Proxy.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Properties Get ALL DBUS Proxy.
 */

#include "PropertiesGetAllDbusProxyIf.h"
#include "IPropertiesGetAllDbusCallbackIf.h"
#include "CcDbusIfTypes.h"
#include "CcDbusIfUtils.h"
#include "ICcDbusIfControllerClient.h"
#include "PropertiesGetAllCreateRuntimeProxyWorkItem.h"
#include "PropertiesGetAllDestroyRuntimeProxyWorkItem.h"

#include "TraceClasses.h"
#define FW_S_IMPORT_INTERFACE_ASSERT
#define FW_S_IMPORT_INTERFACE_TRACE
#include "framework_if_if.h"

using namespace ::getall::org::freedesktop::DBus::Properties;

using namespace ::asf::core;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CCDBUSIF_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/PropertiesGetAllDbusProxyIf.cpp.trc.h"
#endif
#endif

namespace ccdbusif {

PropertiesGetAllDbusProxyIf::PropertiesGetAllDbusProxyIf()
{
   // _proxy
}

PropertiesGetAllDbusProxyIf::PropertiesGetAllDbusProxyIf(ICcDbusIfControllerClient* client) : BaseDbusProxyIf(client)
{
   // _proxy
}

PropertiesGetAllDbusProxyIf::~PropertiesGetAllDbusProxyIf()
{
}

void PropertiesGetAllDbusProxyIf::destroyProxy(void)
{
   ETG_TRACE_USR3((" destroyProxy(): Properties"));

   // these proxies have to be destroyed during runtime
   // destroy all proxies now if still available
   _proxy.resetAllProxiesAndClear();
   _proxy.removeAllCallbackIfs();
}

void PropertiesGetAllDbusProxyIf::destroyAllRuntimeProxies(void)
{
   ETG_TRACE_USR3((" destroyAllRuntimeProxies(): Properties"));

   // HINT: feature is currently only used for a static proxy
}

void PropertiesGetAllDbusProxyIf::createRuntimeProxy(const ::std::string& busName, const ::std::string& objPath, const DbusBusType busType, void* user, IPropertiesGetAllDbusCallbackIf* callbackIf)
{
   (void)(user);
   ProxyMetaData metaData;
   metaData.busName = busName;
   metaData.objPath = objPath;
   metaData.busType = busType;

   ETG_TRACE_USR3((" createRuntimeProxy(): Properties: busType=%d busName=%20s objPath=%s", busType, busName.c_str(), objPath.c_str()));

   if(false == _proxy.isInstanceAvailable(metaData))
   {
      ::boost::shared_ptr< PropertiesProxy > propertiesProxy;

      // next check is only necessary because multiple clients can use same proxy
      if(false == _proxy.isProxyAvailable(propertiesProxy, metaData))
      {
         propertiesProxy = PropertiesProxy::createProxy("allPropertiesGetAllPort", busName, objPath, convertLocalBusType(busType), *this);
      }
      _proxy.addProxyInstance(metaData, propertiesProxy);
      _proxy.addCallbackIf(metaData, callbackIf);
   }
}

void PropertiesGetAllDbusProxyIf::destroyRuntimeProxy(const ::std::string& busName, const ::std::string& objPath, const DbusBusType busType, void* user)
{
   (void)(user);
   ProxyMetaData metaData;
   metaData.busName = busName;
   metaData.objPath = objPath;
   metaData.busType = busType;

   ETG_TRACE_USR3((" destroyRuntimeProxy(): Properties: busType=%d busName=%20s objPath=%s", busType, busName.c_str(), objPath.c_str()));

   // nothing to deregister
   _proxy.removeProxyInstance(metaData);
   _proxy.removeCallbackIf(metaData);
}

void PropertiesGetAllDbusProxyIf::setCallbackIf(IPropertiesGetAllDbusCallbackIf* callbackIf, bool enableProxy, const ::std::string& busName, const ::std::string& objPath, const DbusBusType busType, void* user)
{
   (void)(enableProxy);

   // callback interface is now available => create proxy now
   if((NULL != callbackIf) && (NULL != _controllerClient))
   {
      _controllerClient->pushWorkItem(new PropertiesGetAllCreateRuntimeProxyWorkItem(busName, objPath, busType, user, callbackIf, this));
   }
}

// ServiceAvailableIF implementation --- start
void PropertiesGetAllDbusProxyIf::onAvailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3((" onAvailable(): Properties"));

   for(unsigned int i = 0; i < _proxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< PropertiesProxy >& propertiesProxy = _proxy[i];

      if((NULL != propertiesProxy.get()) && (proxy == propertiesProxy))
      {
         const ::std::string& busName = propertiesProxy->getDBusBusName();
         const ::std::string& objectPath = propertiesProxy->getDBusObjectPath();
         DbusBusType busType = convertDBusBusType(propertiesProxy->getBusType());

         IPropertiesGetAllDbusCallbackIf* callback = _proxy.getCallbackIf(busName, objectPath, busType, 0);

         if(NULL != callback)
         {
            // nothing to register

            callback->onPropertiesGetAllProxyAvailable(busName, objectPath, busType, 0, stateChange.getPreviousState(), stateChange.getCurrentState());
         }
         break;
      }
   }
}

void PropertiesGetAllDbusProxyIf::onUnavailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3((" onUnavailable(): Properties"));

   for(unsigned int i = 0; i < _proxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< PropertiesProxy >& propertiesProxy = _proxy[i];

      if((NULL != propertiesProxy.get()) && (proxy == propertiesProxy))
      {
         const ::std::string& busName = propertiesProxy->getDBusBusName();
         const ::std::string& objectPath = propertiesProxy->getDBusObjectPath();
         DbusBusType busType = convertDBusBusType(propertiesProxy->getBusType());

         IPropertiesGetAllDbusCallbackIf* callback = _proxy.getCallbackIf(busName, objectPath, busType, 0);

         // nothing to deregister

         if(NULL != callback)
         {
            callback->onPropertiesGetAllProxyUnavailable(busName, objectPath, busType, 0, stateChange.getPreviousState(), stateChange.getCurrentState());
         }
         break;
      }
   }
}
// ServiceAvailableIF implementation --- end

// HINT: following interfaces are called within context of thread started in CcDbusIfController.cpp ("CC_DBUS_IF_THD")

// IPropertiesGetAllDbusSendRequestIf implementation --- start
act_t PropertiesGetAllDbusProxyIf::sendGetAllRequest(const ::std::string& busName, const ::std::string& objPath, const DbusBusType busType, void* user, const ::std::string& interface)
{
   (void)(user);
   ProxyMetaData metaData;
   metaData.busName = busName;
   metaData.objPath = objPath;
   metaData.busType = busType;

   ::boost::shared_ptr< PropertiesProxy > propertiesProxy;
   if(true == _proxy.isProxyAvailable(propertiesProxy, metaData))
   {
      IPropertiesGetAllDbusCallbackIf* callback = _proxy.getCallbackIf(busName, objPath, busType, user);
      if(NULL != callback)
      {
         return propertiesProxy->sendGetAllRequest(*callback, interface); // TODO: this will not work if callback/entry is removed before answer is received; valid for all dynamic interfaces
      }
   }

   return DEFAULT_ACT;
}
// IPropertiesGetAllDbusSendRequestIf implementation --- end

} //ccdbusif
