/**
 * @file OrgFreedesktopDbusObjectManagerProxyIf.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief OrgFreedesktop DBUS proxy for ObjectManager.
 *
 * @copyright (C) 2016 - 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details OrgFreedesktop DBUS proxy for ObjectManager.
 */

#include "OrgFreedesktopDbusObjectManagerProxyIf.h"
#include "WblDbusParser.h"
#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

using namespace ::asf::core;
using namespace ::objmgr::org::freedesktop::DBus::ObjectManager;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CCDBUSIF_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/OrgFreedesktopDbusObjectManagerProxyIf.cpp.trc.h"
#endif
#endif

namespace ccdbusif {
namespace freedesktop {

OrgFreedesktopDbusObjectManagerProxyIf::OrgFreedesktopDbusObjectManagerProxyIf(ICcDbusIfControllerClient* client) :
DbusBaseProxyIf< IOrgFreedesktopDbusObjectManagerCallbackIf, ObjectManagerProxy, IOrgFreedesktopDbusObjectManagerTestProxyIf >(client),
_portObjectManager("allObjectManagerPort")
{
}

OrgFreedesktopDbusObjectManagerProxyIf::OrgFreedesktopDbusObjectManagerProxyIf(ICcDbusIfControllerClient* client, IOrgFreedesktopDbusObjectManagerTestProxyIf* testProxyIf) :
DbusBaseProxyIf< IOrgFreedesktopDbusObjectManagerCallbackIf, ObjectManagerProxy, IOrgFreedesktopDbusObjectManagerTestProxyIf >(client, testProxyIf),
_portObjectManager("allObjectManagerPort")
{
}

OrgFreedesktopDbusObjectManagerProxyIf::~OrgFreedesktopDbusObjectManagerProxyIf()
{
}

void OrgFreedesktopDbusObjectManagerProxyIf::onAvailable(const boost::shared_ptr< Proxy >& proxy, const ServiceStateChange& stateChange)
{
   for(size_t i = 0; i < getProxyListSize(); i++)
   {
      ::boost::shared_ptr< ObjectManagerProxy >* proxyPtr(getProxy(i));
      if(0 != proxyPtr)
      {
         ::boost::shared_ptr< ObjectManagerProxy >& myProxy = *proxyPtr;
         if((0 != myProxy.get()) && (proxy == myProxy))
         {
            (void)myProxy->sendInterfacesAddedRegister(*this);
            (void)myProxy->sendInterfacesRemovedRegister(*this);

            internalOnAvailable(ServiceState__Available == stateChange.getCurrentState(), myProxy->getDBusObjectPath(), myProxy->getDBusBusName(), convertConnectorOption2BusType(myProxy->getConnectorOptions()));
            break;
         }
      }
   }
}

void OrgFreedesktopDbusObjectManagerProxyIf::onUnavailable(const boost::shared_ptr< Proxy >& proxy, const ServiceStateChange& stateChange)
{
   for(size_t i = 0; i < getProxyListSize(); i++)
   {
      ::boost::shared_ptr< ObjectManagerProxy >* proxyPtr(getProxy(i));
      if(0 != proxyPtr)
      {
         ::boost::shared_ptr< ObjectManagerProxy >& myProxy = *proxyPtr;
         if((0 != myProxy.get()) && (proxy == myProxy))
         {
            myProxy->sendDeregisterAll();

            internalOnUnavailable(ServiceState__Available == stateChange.getCurrentState(), myProxy->getDBusObjectPath(), myProxy->getDBusBusName(), convertConnectorOption2BusType(myProxy->getConnectorOptions()));
            break;
         }
      }
   }
}

void OrgFreedesktopDbusObjectManagerProxyIf::handleSetCallback(IOrgFreedesktopDbusObjectManagerCallbackIf* callbackIf, const bool enableProxy, const unsigned int callbackId, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   storeCallback(callbackIf, enableProxy, callbackId, objPath, busName, busType); // path, name, type shall be given
}

void OrgFreedesktopDbusObjectManagerProxyIf::handleCreateProxy(const unsigned int callbackId, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   internalCreateProxy(callbackId, objPath, busName, busType);
}

void OrgFreedesktopDbusObjectManagerProxyIf::handleDestroyProxy(const unsigned int callbackId, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   internalDestroyProxy(callbackId, objPath, busName, busType);
}

void OrgFreedesktopDbusObjectManagerProxyIf::handleDestroyAllProxies(void)
{
   internalDestroyAllProxies();
}

void OrgFreedesktopDbusObjectManagerProxyIf::handleDestroyAllRuntimeProxies(void)
{
   internalDestroyAllRuntimeProxies();
}

IDestroyAllProxies* OrgFreedesktopDbusObjectManagerProxyIf::getDestroyer(void)
{
   return this;
}

void OrgFreedesktopDbusObjectManagerProxyIf::setCallbackIf(IOrgFreedesktopDbusObjectManagerCallbackIf* callbackIf, const bool enableProxy, const unsigned int callbackId, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   processSetCallback(this, callbackIf, enableProxy, callbackId, objPath, busName, busType);
}

void OrgFreedesktopDbusObjectManagerProxyIf::createProxyIf(const unsigned int callbackId, const bool createDirectly /*= false*/, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   processCreateProxy(this, createDirectly, callbackId, objPath, busName, busType);
}

void OrgFreedesktopDbusObjectManagerProxyIf::destroyProxyIf(const unsigned int callbackId, const bool createDirectly /*= false*/, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   processDestroyProxy(this, createDirectly, callbackId, objPath, busName, busType);
}

act_t OrgFreedesktopDbusObjectManagerProxyIf::sendGetManagedObjectsRequest(const unsigned int callbackId, const ::std::string& objPath, const ::std::string& busName, const DbusBusType busType)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable(objPath, busName, busType)) // path, name, type shall be given
      {
         retAct = _testProxyIf->sendGetManagedObjectsRequest(*this, objPath, busName, busType);
      }
   }
   else
   {
      ::boost::shared_ptr< ObjectManagerProxy > proxy;
      if(true == isProxyAvailable(proxy, objPath, busName, busType)) // path, name, type shall be given
      {
         retAct = proxy->sendGetManagedObjectsRequest(*this);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

void OrgFreedesktopDbusObjectManagerProxyIf::onGetManagedObjectsError(const ::boost::shared_ptr< ObjectManagerProxy >& proxy, const ::boost::shared_ptr< GetManagedObjectsError >& error)
{
   onGetManagedObjectsErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void OrgFreedesktopDbusObjectManagerProxyIf::onGetManagedObjectsResponse(const ::boost::shared_ptr< ObjectManagerProxy >& proxy, const ::boost::shared_ptr< GetManagedObjectsResponse >& response)
{
   onGetManagedObjectsResponseCb(response, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void OrgFreedesktopDbusObjectManagerProxyIf::onInterfacesAddedError(const ::boost::shared_ptr< ObjectManagerProxy >& proxy, const ::boost::shared_ptr< InterfacesAddedError >& error)
{
   onInterfacesAddedErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void OrgFreedesktopDbusObjectManagerProxyIf::onInterfacesAddedSignal(const ::boost::shared_ptr< ObjectManagerProxy >& proxy, const ::boost::shared_ptr< InterfacesAddedSignal >& signal)
{
   onInterfacesAddedSignalCb(signal, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void OrgFreedesktopDbusObjectManagerProxyIf::onInterfacesRemovedError(const ::boost::shared_ptr< ObjectManagerProxy >& proxy, const ::boost::shared_ptr< InterfacesRemovedError >& error)
{
   onInterfacesRemovedErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void OrgFreedesktopDbusObjectManagerProxyIf::onInterfacesRemovedSignal(const ::boost::shared_ptr< ObjectManagerProxy >& proxy, const ::boost::shared_ptr< InterfacesRemovedSignal >& signal)
{
   onInterfacesRemovedSignalCb(signal, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void OrgFreedesktopDbusObjectManagerProxyIf::onAvailableCb(const bool available, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   internalOnAvailable(available, objPath, busName, busType);
}

void OrgFreedesktopDbusObjectManagerProxyIf::onUnavailableCb(const bool available, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   internalOnUnavailable(available, objPath, busName, busType);
}

void OrgFreedesktopDbusObjectManagerProxyIf::onGetManagedObjectsErrorCb(const ::boost::shared_ptr< GetManagedObjectsError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IOrgFreedesktopDbusObjectManagerCallbackIf* callback(removeActAndFindCallback(error->getAct()));
   FW_IF_NULL_PTR_RETURN(callback);
   callback->onGetManagedObjectsErrorCb(error, objPath, busName, busType, interfaceName);
}

void OrgFreedesktopDbusObjectManagerProxyIf::onGetManagedObjectsResponseCb(const ::boost::shared_ptr< GetManagedObjectsResponse >& response, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IOrgFreedesktopDbusObjectManagerCallbackIf* callback(removeActAndFindCallback(response->getAct()));
   FW_IF_NULL_PTR_RETURN(callback);
   callback->onGetManagedObjectsResponseCb(response, objPath, busName, busType, interfaceName);
}

void OrgFreedesktopDbusObjectManagerProxyIf::onInterfacesAddedErrorCb(const ::boost::shared_ptr< InterfacesAddedError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IOrgFreedesktopDbusObjectManagerCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onInterfacesAddedErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onInterfacesAddedErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void OrgFreedesktopDbusObjectManagerProxyIf::onInterfacesAddedSignalCb(const ::boost::shared_ptr< InterfacesAddedSignal >& signal, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IOrgFreedesktopDbusObjectManagerCallbackIf* callback;
   callback = removeActAndFindCallback(signal->getAct());
   if(0 != callback)
   {
      callback->onInterfacesAddedSignalCb(signal, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onInterfacesAddedSignalCb(signal, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void OrgFreedesktopDbusObjectManagerProxyIf::onInterfacesRemovedErrorCb(const ::boost::shared_ptr< InterfacesRemovedError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IOrgFreedesktopDbusObjectManagerCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onInterfacesRemovedErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onInterfacesRemovedErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void OrgFreedesktopDbusObjectManagerProxyIf::onInterfacesRemovedSignalCb(const ::boost::shared_ptr< InterfacesRemovedSignal >& signal, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IOrgFreedesktopDbusObjectManagerCallbackIf* callback;
   callback = removeActAndFindCallback(signal->getAct());
   if(0 != callback)
   {
      callback->onInterfacesRemovedSignalCb(signal, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onInterfacesRemovedSignalCb(signal, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void OrgFreedesktopDbusObjectManagerProxyIf::internalCreateProxy(const unsigned int callbackId, const ::std::string& objPath, const ::std::string& busName, const DbusBusType busType)
{
   const ::std::string& port(_portObjectManager);
   const ::std::string& path(objPath);
   const ::std::string& name(busName);
   const DbusBusType type(busType);

   // check if proxy is available
   if(false == isProxyAvailable(objPath, busName, busType)) // path, name, type shall be given
   {
      ETG_TRACE_USR3((" internalCreateProxy(): ObjectManager: callbackId=%u busType=%d busName=%20s objPath=%s (create)", callbackId, busType, busName.c_str(), objPath.c_str()));

      // create proxy
      if(0 != _testProxyIf)
      {
         createTestProxy(port, path, name, type, *this);
         addTestProxy(objPath, busName, busType); // path, name, type shall be given
      }
      else
      {
         ::boost::shared_ptr< ObjectManagerProxy > proxy = createProxy(port, path, name, type, *this);
         addProxy(proxy, objPath, busName, busType); // path, name, type shall be given
      }
   }
   else
   {
      ETG_TRACE_USR3((" internalCreateProxy(): ObjectManager: callbackId=%u busType=%d busName=%20s objPath=%s (already done)", callbackId, busType, busName.c_str(), objPath.c_str()));

      // proxy creation was already started; either service is available or creation is ongoing
      updateCurrentAvailableStatus(callbackId, getCreationState(objPath, busName, busType) /* path, name, type shall be given */, path, name, type);
   }
}

void OrgFreedesktopDbusObjectManagerProxyIf::internalDestroyProxy(const unsigned int callbackId, const ::std::string& objPath, const ::std::string& busName, const DbusBusType busType)
{
   // HINT: destroying runtime proxy is not possible because ASF is storing a proxy "reference" internally; therefore reset() call will not causing destruction of proxy
   // HINT: due to this we explicitly have to call sendDeregisterAll() to force removing of signal registration (DBUS: member=RemoveMatch)
   // HINT: sendDeregisterAll() is automatically called during destruction of proxy
   // HINT: sendDeregisterAll() is not provided by all proxies

   // check if proxy is available
   if(true == isProxyAvailable(objPath, busName, busType)) // path, name, type shall be given
   {
      ETG_TRACE_USR3((" internalDestroyProxy(): ObjectManager: callbackId=%u busType=%d busName=%20s objPath=%s (destroy)", callbackId, busType, busName.c_str(), objPath.c_str()));

      // destroy proxy
      if(0 != _testProxyIf)
      {
         const ::std::string& path(objPath);
         const ::std::string& name(busName);
         const DbusBusType type(busType);

         destroyTestProxy(path, name, type);
         removeTestProxy(objPath, busName, busType); // path, name, type shall be given
      }
      else
      {
         ::boost::shared_ptr< ObjectManagerProxy >* proxyPtr(getProxy(objPath, busName, busType)); // path, name, type shall be given
         if(0 != proxyPtr)
         {
            ::boost::shared_ptr< ObjectManagerProxy >& proxy = *proxyPtr;
            proxy->sendDeregisterAll();
         }
         removeProxy(objPath, busName, busType); // path, name, type shall be given
      }
   }
   else
   {
      ETG_TRACE_USR3((" internalDestroyProxy(): ObjectManager: callbackId=%u busType=%d busName=%20s objPath=%s (already done)", callbackId, busType, busName.c_str(), objPath.c_str()));
   }

   // NOTE: check following: second callback is registered, anything to do?
}

void OrgFreedesktopDbusObjectManagerProxyIf::internalDestroyAllProxies(void)
{
   // HINT: sendDeregisterAll() is not provided by all proxies

   ETG_TRACE_USR3((" internalDestroyAllProxies(): ObjectManager"));

   // destroy all proxies
   if(0 != _testProxyIf)
   {
      destroyAllTestProxies();
      removeAllTestProxies();
   }
   else
   {
      for(size_t i = 0; i < getProxyListSize(); i++)
      {
         ::boost::shared_ptr< ObjectManagerProxy >* proxyPtr(getProxy(i));
         if(0 != proxyPtr)
         {
            ::boost::shared_ptr< ObjectManagerProxy >& proxy = *proxyPtr;
            proxy->sendDeregisterAll();
         }
      }
      removeAllProxies();
   }

   // remove callback information
   removeCallbacks();
}

void OrgFreedesktopDbusObjectManagerProxyIf::internalDestroyAllRuntimeProxies(void)
{
   // HINT: sendDeregisterAll() is not provided by all proxies

   ETG_TRACE_USR3((" internalDestroyAllRuntimeProxies(): ObjectManager"));

   // destroy all runtime proxies
   // nothing to do because this is a static proxy

   // do not remove callback information
}

void OrgFreedesktopDbusObjectManagerProxyIf::internalOnAvailable(const bool available, const ::std::string& objPath, const ::std::string& busName, const DbusBusType busType)
{
   ETG_TRACE_USR3((" internalOnAvailable(): ObjectManager"));

   // inform all matching callback handler
   setCreationState(available, objPath, busName, busType); // path, name, type shall be given
   updateAvailableStatusToAll(available, objPath, busName, busType, objPath, busName, busType); // path, name, type shall be given
}

void OrgFreedesktopDbusObjectManagerProxyIf::internalOnUnavailable(const bool available, const ::std::string& objPath, const ::std::string& busName, const DbusBusType busType)
{
   ETG_TRACE_USR3((" internalOnUnavailable(): ObjectManager"));

   // inform all matching callback handler
   setCreationState(available, objPath, busName, busType); // path, name, type shall be given
   updateUnavailableStatusToAll(available, objPath, busName, busType, objPath, busName, busType); // path, name, type shall be given
}

} //freedesktop
} //ccdbusif
