/**
 * @file IntrospectableProxyManager.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief Proxy management for introspectable interface.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Proxy management for introspectable interface.
 */

#include "IntrospectableProxyManager.h"
#include "IIntrospectableDbusCallbackIf.h"

#define FW_S_IMPORT_INTERFACE_ASSERT
#include "framework_if_if.h"

namespace ccdbusif {

IntrospectableProxyManager::IntrospectableProxyManager()
{
   // _proxyMetaData2CallbackIf
}

IntrospectableProxyManager::IntrospectableProxyManager(const IntrospectableProxyManager& ref)
: ProxyManager< IntrospectableProxy >(ref)
{
   _proxyMetaData2CallbackIf = ref._proxyMetaData2CallbackIf;
}

IntrospectableProxyManager& IntrospectableProxyManager::operator=(const IntrospectableProxyManager& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   ProxyManager< IntrospectableProxy >::operator=(ref);

   _proxyMetaData2CallbackIf = ref._proxyMetaData2CallbackIf;

   return *this;
}

bool IntrospectableProxyManager::operator==(const IntrospectableProxyManager& ref) const
{
   bool result = ProxyManager< IntrospectableProxy >::operator==(ref);

   result = (true == result) && (_proxyMetaData2CallbackIf == ref._proxyMetaData2CallbackIf);

   return result;
}

bool IntrospectableProxyManager::operator!=(const IntrospectableProxyManager& ref) const
{
   return !(operator==(ref));
}

IntrospectableProxyManager::~IntrospectableProxyManager()
{
}

void IntrospectableProxyManager::addCallbackIf(const ProxyMetaData& metaData, IIntrospectableDbusCallbackIf* callback)
{
   ::std::map<ProxyMetaData, IIntrospectableDbusCallbackIf*>::iterator it = _proxyMetaData2CallbackIf.find(metaData);

   if(_proxyMetaData2CallbackIf.end() != it)
   {
      // entry is already stored => can happen because function is called after reception of different signals
   }
   else
   {
      // entry is not stored
      _proxyMetaData2CallbackIf[metaData] = callback;

      // check size
      if(0 < _maxNumberOfProxyInstances)
      {
         if(_proxyMetaData2CallbackIf.size() > _maxNumberOfProxyInstances)
         {
            // should never happen - forgot to remove instance?
            // #error_indication
            FW_NORMAL_ASSERT_ALWAYS();
         }
      }
   }
}

void IntrospectableProxyManager::addCallbackIf(const ::std::string& busName, const ::std::string& objPath, const DbusBusType busType, void* user, IIntrospectableDbusCallbackIf* callback)
{
   (void)(user);
   ProxyMetaData metaData;
   metaData.busName = busName;
   metaData.objPath = objPath;
   metaData.busType = busType;

   addCallbackIf(metaData, callback);
}

void IntrospectableProxyManager::removeCallbackIf(const ProxyMetaData& metaData)
{
   ::std::map<ProxyMetaData, IIntrospectableDbusCallbackIf*>::iterator it = _proxyMetaData2CallbackIf.find(metaData);

   if(_proxyMetaData2CallbackIf.end() == it)
   {
      // entry is already erased => can happen
   }
   else
   {
      // entry is not erased
      _proxyMetaData2CallbackIf.erase(it);
   }
}

void IntrospectableProxyManager::removeCallbackIf(const ::std::string& busName, const ::std::string& objPath, const DbusBusType busType, void* user)
{
   (void)(user);
   ProxyMetaData metaData;
   metaData.busName = busName;
   metaData.objPath = objPath;
   metaData.busType = busType;

   removeCallbackIf(metaData);
}

IIntrospectableDbusCallbackIf* IntrospectableProxyManager::getCallbackIf(const ProxyMetaData& metaData)
{
   ::std::map<ProxyMetaData, IIntrospectableDbusCallbackIf*>::iterator it = _proxyMetaData2CallbackIf.find(metaData);

   if(_proxyMetaData2CallbackIf.end() != it)
   {
      return it->second;
   }

   return NULL;
}

IIntrospectableDbusCallbackIf* IntrospectableProxyManager::getCallbackIf(const ::std::string& busName, const ::std::string& objPath, const DbusBusType busType, void* user)
{
   (void)(user);
   ProxyMetaData metaData;
   metaData.busName = busName;
   metaData.objPath = objPath;
   metaData.busType = busType;

   return getCallbackIf(metaData);
}

void IntrospectableProxyManager::removeAllCallbackIfs(void)
{
   _proxyMetaData2CallbackIf.clear();
}

} //ccdbusif
