/**
 * @file EvolutionGeniviDbusObexSessionProxyIf.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Proxy for ObexSession.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details EvolutionGenivi DBUS proxy for ObexSession.
 */

#include "EvolutionGeniviDbusObexSessionProxyIf.h"
#include "IEvolutionGeniviDbusObexSessionCallbackIf.h"
#include "CcDbusIfTypes.h"
#include "CcDbusIfUtils.h"
#include "ICcDbusIfControllerClient.h"
#include "CcDbusIfCreateProxyWorkItem.h"
#include "CcDbusIfCreateRuntimeProxyWorkItem.h"
#include "CcDbusIfDestroyRuntimeProxyWorkItem.h"
#include "EvolutionGeniviDbusParser.h"

#include "TraceClasses.h"
#define FW_S_IMPORT_INTERFACE_TRACE
#include "framework_if_if.h"

using namespace ::org::bluez::obex::Session;
using namespace ::asf::core;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CCDBUSIF_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/EvolutionGeniviDbusObexSessionProxyIf.cpp.trc.h"
#endif
#endif

namespace ccdbusif {

EvolutionGeniviDbusObexSessionProxyIf::EvolutionGeniviDbusObexSessionProxyIf()
{
   // _obexSessionProxy is set later
   _callbackIf = NULL;
}

EvolutionGeniviDbusObexSessionProxyIf::EvolutionGeniviDbusObexSessionProxyIf(ICcDbusIfControllerClient* client) : BaseDbusProxyIf(client)
{
   // _obexSessionProxy is set later
   _callbackIf = NULL;
}

EvolutionGeniviDbusObexSessionProxyIf::~EvolutionGeniviDbusObexSessionProxyIf()
{
   _callbackIf = NULL;
}

void EvolutionGeniviDbusObexSessionProxyIf::createProxy(void)
{
   // these proxies have to be created during runtime
}

void EvolutionGeniviDbusObexSessionProxyIf::destroyProxy(void)
{
   ETG_TRACE_USR3((" destroyProxy(): Session"));

   // these proxies have to be destroyed during runtime
   // destroy all proxies now if still available
   _obexSessionProxy.resetAllProxiesAndClear();
}

void EvolutionGeniviDbusObexSessionProxyIf::destroyAllRuntimeProxies(void)
{
   ETG_TRACE_USR3((" destroyAllRuntimeProxies(): Session"));

   for(unsigned int i = 0; i < _obexSessionProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< SessionProxy >& obexSessionProxy = _obexSessionProxy[i];
      if(NULL != obexSessionProxy.get())
      {
         obexSessionProxy->sendDeregisterAll();
      }
   }

   _obexSessionProxy.resetAllProxiesAndClear();
}

void EvolutionGeniviDbusObexSessionProxyIf::createRuntimeProxy(const ::std::string& objPath)
{
   ETG_TRACE_USR3((" createRuntimeProxy(): Session: objPath=%s", objPath.c_str()));

   if((NULL != _callbackIf) && (true == _enableProxy))
   {
      if(false == _obexSessionProxy.isInstanceAvailable(objPath))
      {
         evolution::EvolutionGeniviDbusParser parser;
         ::boost::shared_ptr< SessionProxy > obexSessionProxy = SessionProxy::createProxy("evoObexSessionPort", parser.getInterface2BusName(evolution::IF_SESSION), objPath, convertLocalBusType(parser.getInterface2BusType(evolution::IF_SESSION)), *this);
         _obexSessionProxy.addProxyInstance(objPath, obexSessionProxy);
      }
   }
}

void EvolutionGeniviDbusObexSessionProxyIf::destroyRuntimeProxy(const ::std::string& objPath)
{
   ETG_TRACE_USR3((" destroyRuntimeProxy(): Session: objPath=%s", objPath.c_str()));

   // HINT: destroying runtime proxy is not possible because ASF is storing a proxy "reference" internally; therefore reset() call will not causing destruction of proxy
   // HINT: due to this we explicitly have to call sendDeregisterAll() to force removing of signal registration (DBUS: member=RemoveMatch)
   // HINT: sendDeregisterAll() is automatically called during destruction of proxy

   ::boost::shared_ptr< SessionProxy > obexSessionProxy;
   if(true == _obexSessionProxy.isProxyAvailable(obexSessionProxy, objPath))
   {
      obexSessionProxy->sendDeregisterAll();
   }
   _obexSessionProxy.removeProxyInstance(objPath);
}

void EvolutionGeniviDbusObexSessionProxyIf::setCallbackIf(IEvolutionGeniviDbusObexSessionCallbackIf* callbackIf, bool enableProxy)
{
   _callbackIf = callbackIf;

   if(NULL != _callbackIf)
   {
      _enableProxy = enableProxy;
   }

   // callback interface is now available => but proxy will be created later
}

void EvolutionGeniviDbusObexSessionProxyIf::sessionCreated(const ::std::string& objPath)
{
   // create runtime proxy
   if((NULL != _callbackIf) && (NULL != _controllerClient))
   {
      _controllerClient->pushWorkItem(new CcDbusIfCreateRuntimeProxyWorkItem(objPath, this));
   }
}

void EvolutionGeniviDbusObexSessionProxyIf::sessionRemoved(const ::std::string& objPath)
{
   // destroy runtime proxy
   if((NULL != _callbackIf) && (NULL != _controllerClient))
   {
      _controllerClient->pushWorkItem(new CcDbusIfDestroyRuntimeProxyWorkItem(objPath, this));
   }
}

// ServiceAvailableIF implementation --- start
void EvolutionGeniviDbusObexSessionProxyIf::onAvailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3((" onAvailable(): Session"));

   for(unsigned int i = 0; i < _obexSessionProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< SessionProxy >& obexSessionProxy = _obexSessionProxy[i];
      if((NULL != obexSessionProxy.get()) && (proxy == obexSessionProxy))
      {
         if(NULL != _callbackIf)
         {
            obexSessionProxy->sendSourceRegister(*_callbackIf);
            obexSessionProxy->sendDestinationRegister(*_callbackIf);
            obexSessionProxy->sendChannelRegister(*_callbackIf);
            obexSessionProxy->sendTargetRegister(*_callbackIf);

            _callbackIf->onProxyAvailable(stateChange.getPreviousState(), stateChange.getCurrentState(), obexSessionProxy->getDBusObjectPath());
         }
         break;
      }
   }
}

void EvolutionGeniviDbusObexSessionProxyIf::onUnavailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3((" onUnavailable(): Session"));

   for(unsigned int i = 0; i < _obexSessionProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< SessionProxy >& obexSessionProxy = _obexSessionProxy[i];
      if((NULL != obexSessionProxy.get()) && (proxy == obexSessionProxy))
      {
         obexSessionProxy->sendDeregisterAll();

         if(NULL != _callbackIf)
         {
            _callbackIf->onProxyUnavailable(stateChange.getPreviousState(), stateChange.getCurrentState(), obexSessionProxy->getDBusObjectPath());
         }
         break;
      }
   }
}
// ServiceAvailableIF implementation --- end

// HINT: following interfaces are called within context of thread started in CcDbusIfController.cpp ("CC_DBUS_IF_THD")

// IEvolutionGeniviDbusObexSessionProxyIf implementation --- start
act_t EvolutionGeniviDbusObexSessionProxyIf::sendSourceGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< SessionProxy > proxy;
   if(true == _obexSessionProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendSourceGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusObexSessionProxyIf::sendDestinationGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< SessionProxy > proxy;
   if(true == _obexSessionProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendDestinationGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusObexSessionProxyIf::sendChannelGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< SessionProxy > proxy;
   if(true == _obexSessionProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendChannelGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusObexSessionProxyIf::sendTargetGet(const ::std::string& objPath)
{
   ::boost::shared_ptr< SessionProxy > proxy;
   if(true == _obexSessionProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendTargetGet(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}
// IEvolutionGeniviDbusObexSessionProxyIf implementation --- end

} //ccdbusif
