/**
 * @file EvolutionGeniviDbusObexClientProxyIf.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Proxy for ObexClient.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details EvolutionGenivi DBUS proxy for ObexClient.
 */

#include "EvolutionGeniviDbusObexClientProxyIf.h"
#include "IEvolutionGeniviDbusObexClientCallbackIf.h"
#include "CcDbusIfTypes.h"
#include "ICcDbusIfControllerClient.h"
#include "CcDbusIfCreateProxyWorkItem.h"
#include "EvolutionGeniviDbusParser.h"

#include "TraceClasses.h"
#define FW_S_IMPORT_INTERFACE_TRACE
#include "framework_if_if.h"

using namespace ::org::bluez::obex::Client;
using namespace ::asf::core;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CCDBUSIF_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/EvolutionGeniviDbusObexClientProxyIf.cpp.trc.h"
#endif
#endif

namespace ccdbusif {

EvolutionGeniviDbusObexClientProxyIf::EvolutionGeniviDbusObexClientProxyIf()
{
   // _obexClientProxy is set later
   _callbackIf = NULL;
}

EvolutionGeniviDbusObexClientProxyIf::EvolutionGeniviDbusObexClientProxyIf(ICcDbusIfControllerClient* client) : BaseDbusProxyIf(client)
{
   // _obexClientProxy is set later
   _callbackIf = NULL;
}

EvolutionGeniviDbusObexClientProxyIf::~EvolutionGeniviDbusObexClientProxyIf()
{
   _callbackIf = NULL;
}

void EvolutionGeniviDbusObexClientProxyIf::createProxy(void)
{
   ETG_TRACE_USR3((" createProxy(): Client"));

   if(NULL != _callbackIf)
   {
      if(true == _enableProxy)
      {
         _obexClientProxy = ClientProxy::createProxy("evoObexClientPort", *this);
      }
   }
}

void EvolutionGeniviDbusObexClientProxyIf::destroyProxy(void)
{
   ETG_TRACE_USR3((" destroyProxy(): Client"));

   if(NULL != _obexClientProxy.get())
   {
      _obexClientProxy->sendDeregisterAll();
      _obexClientProxy.reset();
   }
}

void EvolutionGeniviDbusObexClientProxyIf::destroyAllRuntimeProxies(void)
{
   ETG_TRACE_USR3((" destroyAllRuntimeProxies(): Client"));

   // nothing to do because this is a static proxy
}

void EvolutionGeniviDbusObexClientProxyIf::setCallbackIf(IEvolutionGeniviDbusObexClientCallbackIf* callbackIf, bool enableProxy)
{
   _callbackIf = callbackIf;

   if(NULL != _callbackIf)
   {
      _enableProxy = enableProxy;
   }

   // callback interface is now available => create proxy now
   if((NULL != _callbackIf) && (NULL != _controllerClient))
   {
      _controllerClient->pushWorkItem(new CcDbusIfCreateProxyWorkItem(this));
   }
}

// ServiceAvailableIF implementation --- start
void EvolutionGeniviDbusObexClientProxyIf::onAvailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3((" onAvailable(): Client"));

   if((NULL != _obexClientProxy.get()) && (proxy == _obexClientProxy))
   {
      if(NULL != _callbackIf)
      {
         _obexClientProxy->sendSessionCreatedRegister(*_callbackIf);
         _obexClientProxy->sendSessionRemovedRegister(*_callbackIf);

         _callbackIf->onProxyAvailable(stateChange.getPreviousState(), stateChange.getCurrentState(), _obexClientProxy->getDBusObjectPath());
      }
   }
}

void EvolutionGeniviDbusObexClientProxyIf::onUnavailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3((" onUnavailable(): Client"));

   if((NULL != _obexClientProxy.get()) && (proxy == _obexClientProxy))
   {
      _obexClientProxy->sendDeregisterAll();

      if(NULL != _callbackIf)
      {
         _callbackIf->onProxyUnavailable(stateChange.getPreviousState(), stateChange.getCurrentState(), _obexClientProxy->getDBusObjectPath());
      }
   }
}
// ServiceAvailableIF implementation --- end

// IEvolutionGeniviDbusObexClientProxyIf implementation --- start
act_t EvolutionGeniviDbusObexClientProxyIf::sendCreateSessionRequest(const ::std::string& destination, const ::std::string& target, const ::std::string& source, const uint8 channel, const int32 instance)
{
   (void)(channel); // according HLI: The parameter will be ignored by module, host needn't set it.

   if((NULL != _callbackIf) && (NULL != _obexClientProxy.get()))
   {
      evolution::EvolutionGeniviDbusParser parser;
      DbusVariant value;
      ::std::map< ::std::string, ::asf::dbus::DBusVariant > args;

      value.reset();
      value.setString(target);
      parser.addDictionaryItem(args, parser.getObexClientType2String(evolution::OBEX_CLIENT_ARGS_TARGET), value);
      value.reset();
      value.setString(source);
      parser.addDictionaryItem(args, parser.getObexClientType2String(evolution::OBEX_CLIENT_ARGS_SOURCE), value);
      value.reset();
      value.setInt32(instance);
      parser.addDictionaryItem(args, parser.getObexClientType2String(evolution::OBEX_CLIENT_ARGS_INSTANCE_ID), value);

      return _obexClientProxy->sendCreateSessionRequest(*_callbackIf, destination, args);
   }
   else
   {
      return DEFAULT_ACT;
   }
}

act_t EvolutionGeniviDbusObexClientProxyIf::sendCreateSessionRequest(const ::std::string& destination, const ::std::string& target, const ::std::string& source, const uint8 channel)
{
   (void)(channel); // according HLI: The parameter will be ignored by module, host needn't set it.

   if((NULL != _callbackIf) && (NULL != _obexClientProxy.get()))
   {
      evolution::EvolutionGeniviDbusParser parser;
      DbusVariant value;
      ::std::map< ::std::string, ::asf::dbus::DBusVariant > args;

      value.reset();
      value.setString(target);
      parser.addDictionaryItem(args, parser.getObexClientType2String(evolution::OBEX_CLIENT_ARGS_TARGET), value);
      value.reset();
      value.setString(source);
      //::std::vector<std::string>& tmp = value.getStringArrayMutable();
      //tmp.push_back(source);
      //tmp.push_back("katze");
      //tmp.push_back("Hund");
      parser.addDictionaryItem(args, parser.getObexClientType2String(evolution::OBEX_CLIENT_ARGS_SOURCE), value);

      return _obexClientProxy->sendCreateSessionRequest(*_callbackIf, destination, args);
   }
   else
   {
      return DEFAULT_ACT;
   }
}

act_t EvolutionGeniviDbusObexClientProxyIf::sendRemoveSessionRequest(const ::std::string& session)
{
   if((NULL != _callbackIf) && (NULL != _obexClientProxy.get()))
   {
      return _obexClientProxy->sendRemoveSessionRequest(*_callbackIf, session);
   }
   else
   {
      return DEFAULT_ACT;
   }
}

act_t EvolutionGeniviDbusObexClientProxyIf::sendCancelSessionRequest(const ::std::string& destination, const ::std::string& target, const ::std::string& source, const uint8 channel, const int32 instance)
{
   (void)(channel); // according HLI: The parameter will be ignored by module, host needn't set it.

   if((NULL != _callbackIf) && (NULL != _obexClientProxy.get()))
   {
      evolution::EvolutionGeniviDbusParser parser;
      DbusVariant value;
      ::std::map< ::std::string, ::asf::dbus::DBusVariant > args;

      value.reset();
      value.setString(target);
      parser.addDictionaryItem(args, parser.getObexClientType2String(evolution::OBEX_CLIENT_ARGS_TARGET), value);
      value.reset();
      value.setString(source);
      parser.addDictionaryItem(args, parser.getObexClientType2String(evolution::OBEX_CLIENT_ARGS_SOURCE), value);
      value.reset();
      value.setInt32(instance);
      parser.addDictionaryItem(args, parser.getObexClientType2String(evolution::OBEX_CLIENT_ARGS_INSTANCE_ID), value);

      return _obexClientProxy->sendCancelSessionRequest(*_callbackIf, destination, args);
   }
   else
   {
      return DEFAULT_ACT;
   }
}

act_t EvolutionGeniviDbusObexClientProxyIf::sendCancelSessionRequest(const ::std::string& destination, const ::std::string& target, const ::std::string& source, const uint8 channel)
{
   (void)(channel); // according HLI: The parameter will be ignored by module, host needn't set it.

   if((NULL != _callbackIf) && (NULL != _obexClientProxy.get()))
   {
      evolution::EvolutionGeniviDbusParser parser;
      DbusVariant value;
      ::std::map< ::std::string, ::asf::dbus::DBusVariant > args;

      value.reset();
      value.setString(target);
      parser.addDictionaryItem(args, parser.getObexClientType2String(evolution::OBEX_CLIENT_ARGS_TARGET), value);
      value.reset();
      value.setString(source);
      //::std::vector<std::string>& tmp = value.getStringArrayMutable();
      //tmp.push_back(source);
      //tmp.push_back("katze");
      //tmp.push_back("Hund");
      parser.addDictionaryItem(args, parser.getObexClientType2String(evolution::OBEX_CLIENT_ARGS_SOURCE), value);

      return _obexClientProxy->sendCancelSessionRequest(*_callbackIf, destination, args);
   }
   else
   {
      return DEFAULT_ACT;
   }
}
// IEvolutionGeniviDbusObexClientProxyIf implementation --- end

} //ccdbusif
