/**
 * @file EvolutionGeniviDbusLEAdapterProxyIf.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS proxy for LEAdapter.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details EvolutionGenivi DBUS proxy for LEAdapter.
 */

#include "EvolutionGeniviDbusLEAdapterProxyIf.h"
#include "IEvolutionGeniviDbusLEAdapterCallbackIf.h"
#include "CcDbusIfTypes.h"
#include "CcDbusIfUtils.h"
#include "ICcDbusIfControllerClient.h"
#include "CcDbusIfCreateProxyWorkItem.h"
#include "EvolutionGeniviDbusParser.h"

#include "TraceClasses.h"
#define FW_S_IMPORT_INTERFACE_TRACE
#include "framework_if_if.h"

using namespace ::org::bluez::LE_Adapter1;
using namespace ::asf::core;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CCDBUSIF_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/EvolutionGeniviDbusLEAdapterProxyIf.cpp.trc.h"
#endif
#endif

namespace ccdbusif {

EvolutionGeniviDbusLEAdapterProxyIf::EvolutionGeniviDbusLEAdapterProxyIf()
{
   // _lEAdapterProxy is set later
   _callbackIf = NULL;
}

EvolutionGeniviDbusLEAdapterProxyIf::EvolutionGeniviDbusLEAdapterProxyIf(ICcDbusIfControllerClient* client) : BaseDbusProxyIf(client)
{
   // _lEAdapterProxy is set later
   _callbackIf = NULL;
}

EvolutionGeniviDbusLEAdapterProxyIf::~EvolutionGeniviDbusLEAdapterProxyIf()
{
   _callbackIf = NULL;
}

void EvolutionGeniviDbusLEAdapterProxyIf::createProxy(void)
{
   ETG_TRACE_USR3((" createProxy(): LE_Adapter1"));

   if(NULL != _callbackIf)
   {
      if(true == _enableProxy)
      {
         evolution::EvolutionGeniviDbusParser parser;
         _lEAdapterProxy = LE_Adapter1Proxy::createProxy("evoLEAdapterPort", parser.getInterface2BusName(evolution::IF_LE_ADAPTER), parser.getInterface2ObjectPath(evolution::IF_LE_ADAPTER), convertLocalBusType(parser.getInterface2BusType(evolution::IF_LE_ADAPTER)), *this);
      }
   }
}

void EvolutionGeniviDbusLEAdapterProxyIf::destroyProxy(void)
{
   ETG_TRACE_USR3((" destroyProxy(): LE_Adapter1"));

   if(NULL != _lEAdapterProxy.get())
   {
      _lEAdapterProxy->sendDeregisterAll();
      _lEAdapterProxy.reset();
   }
}

void EvolutionGeniviDbusLEAdapterProxyIf::destroyAllRuntimeProxies(void)
{
   ETG_TRACE_USR3((" destroyAllRuntimeProxies(): LE_Adapter1"));

   // nothing to do because this is a static proxy
}

void EvolutionGeniviDbusLEAdapterProxyIf::setCallbackIf(IEvolutionGeniviDbusLEAdapterCallbackIf* callbackIf, bool enableProxy)
{
   _callbackIf = callbackIf;

   if(NULL != _callbackIf)
   {
      _enableProxy = enableProxy;
   }

   // callback interface is now available => create proxy now
   if((NULL != _callbackIf) && (NULL != _controllerClient))
   {
      _controllerClient->pushWorkItem(new CcDbusIfCreateProxyWorkItem(this));
   }
}

// ServiceAvailableIF implementation --- start
void EvolutionGeniviDbusLEAdapterProxyIf::onAvailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3((" onAvailable(): LE_Adapter1"));

   if((NULL != _lEAdapterProxy.get()) && (proxy == _lEAdapterProxy))
   {
      if(NULL != _callbackIf)
      {
         _lEAdapterProxy->sendLeAdapterInitializationRegister(*_callbackIf);
         _lEAdapterProxy->sendLeDeviceCreatedRegister(*_callbackIf);
         _lEAdapterProxy->sendLeDeviceRemovedRegister(*_callbackIf);
         _lEAdapterProxy->sendAddressTypeRegister(*_callbackIf);
         _lEAdapterProxy->sendAddressRegister(*_callbackIf);
         _lEAdapterProxy->sendAliasRegister(*_callbackIf);
         _lEAdapterProxy->sendAdvertisingTimeoutRegister(*_callbackIf);
         _lEAdapterProxy->sendDiscoverableModeRegister(*_callbackIf);
         _lEAdapterProxy->sendAdvertisingDataRegister(*_callbackIf);
         _lEAdapterProxy->sendAdvertiseEnableRegister(*_callbackIf);

         _callbackIf->onProxyAvailable(stateChange.getPreviousState(), stateChange.getCurrentState(), _lEAdapterProxy->getDBusObjectPath());
      }
   }
}

void EvolutionGeniviDbusLEAdapterProxyIf::onUnavailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3((" onUnavailable(): LE_Adapter1"));

   if((NULL != _lEAdapterProxy.get()) && (proxy == _lEAdapterProxy))
   {
      _lEAdapterProxy->sendDeregisterAll();

      if(NULL != _callbackIf)
      {
         _callbackIf->onProxyUnavailable(stateChange.getPreviousState(), stateChange.getCurrentState(), _lEAdapterProxy->getDBusObjectPath());
      }
   }
}
// ServiceAvailableIF implementation --- end

// IEvolutionGeniviDbusLEAdapterProxyIf implementation --- start
act_t EvolutionGeniviDbusLEAdapterProxyIf::sendCreateLeDeviceRequest(const ::std::string& address)
{
   if((NULL != _callbackIf) && (NULL != _lEAdapterProxy.get()))
   {
      return _lEAdapterProxy->sendCreateLeDeviceRequest(*_callbackIf, address);
   }
   else
   {
      return DEFAULT_ACT;
   }
}

act_t EvolutionGeniviDbusLEAdapterProxyIf::sendRemoveLeDeviceRequest(const ::std::string& device)
{
   if((NULL != _callbackIf) && (NULL != _lEAdapterProxy.get()))
   {
      return _lEAdapterProxy->sendRemoveLeDeviceRequest(*_callbackIf, device);
   }
   else
   {
      return DEFAULT_ACT;
   }
}

act_t EvolutionGeniviDbusLEAdapterProxyIf::sendAddressTypeGet(void)
{
   if((NULL != _callbackIf) && (NULL != _lEAdapterProxy.get()))
   {
      return _lEAdapterProxy->sendAddressTypeGet(*_callbackIf);
   }
   else
   {
      return DEFAULT_ACT;
   }
}

act_t EvolutionGeniviDbusLEAdapterProxyIf::sendAddressGet(void)
{
   if((NULL != _callbackIf) && (NULL != _lEAdapterProxy.get()))
   {
      return _lEAdapterProxy->sendAddressGet(*_callbackIf);
   }
   else
   {
      return DEFAULT_ACT;
   }
}

act_t EvolutionGeniviDbusLEAdapterProxyIf::sendAliasGet(void)
{
   if((NULL != _callbackIf) && (NULL != _lEAdapterProxy.get()))
   {
      return _lEAdapterProxy->sendAliasGet(*_callbackIf);
   }
   else
   {
      return DEFAULT_ACT;
   }
}

void EvolutionGeniviDbusLEAdapterProxyIf::sendAliasSet(const ::std::string& alias)
{
   if((NULL != _callbackIf) && (NULL != _lEAdapterProxy.get()))
   {
      _lEAdapterProxy->sendAliasSet(alias);
   }
}

act_t EvolutionGeniviDbusLEAdapterProxyIf::sendAdvertisingTimeoutGet(void)
{
   if((NULL != _callbackIf) && (NULL != _lEAdapterProxy.get()))
   {
      return _lEAdapterProxy->sendAdvertisingTimeoutGet(*_callbackIf);
   }
   else
   {
      return DEFAULT_ACT;
   }
}

void EvolutionGeniviDbusLEAdapterProxyIf::sendAdvertisingTimeoutSet(uint8 advertisingTimeout)
{
   if((NULL != _callbackIf) && (NULL != _lEAdapterProxy.get()))
   {
      _lEAdapterProxy->sendAdvertisingTimeoutSet(advertisingTimeout);
   }
}

act_t EvolutionGeniviDbusLEAdapterProxyIf::sendDiscoverableModeGet(void)
{
   if((NULL != _callbackIf) && (NULL != _lEAdapterProxy.get()))
   {
      return _lEAdapterProxy->sendDiscoverableModeGet(*_callbackIf);
   }
   else
   {
      return DEFAULT_ACT;
   }
}

void EvolutionGeniviDbusLEAdapterProxyIf::sendDiscoverableModeSet(uint8 discoverableMode)
{
   if((NULL != _callbackIf) && (NULL != _lEAdapterProxy.get()))
   {
      _lEAdapterProxy->sendDiscoverableModeSet(discoverableMode);
   }
}

act_t EvolutionGeniviDbusLEAdapterProxyIf::sendAdvertisingDataGet(void)
{
   if((NULL != _callbackIf) && (NULL != _lEAdapterProxy.get()))
   {
      return _lEAdapterProxy->sendAdvertisingDataGet(*_callbackIf);
   }
   else
   {
      return DEFAULT_ACT;
   }
}

void EvolutionGeniviDbusLEAdapterProxyIf::sendAdvertisingDataSet(const ::std::vector< uint8 >& advertisingData)
{
   if((NULL != _callbackIf) && (NULL != _lEAdapterProxy.get()))
   {
      _lEAdapterProxy->sendAdvertisingDataSet(advertisingData);
   }
}

act_t EvolutionGeniviDbusLEAdapterProxyIf::sendAdvertiseEnableGet(void)
{
   if((NULL != _callbackIf) && (NULL != _lEAdapterProxy.get()))
   {
      return _lEAdapterProxy->sendAdvertiseEnableGet(*_callbackIf);
   }
   else
   {
      return DEFAULT_ACT;
   }
}

void EvolutionGeniviDbusLEAdapterProxyIf::sendAdvertiseEnableSet(bool advertiseEnable)
{
   if((NULL != _callbackIf) && (NULL != _lEAdapterProxy.get()))
   {
      _lEAdapterProxy->sendAdvertiseEnableSet(advertiseEnable);
   }
}
// IEvolutionGeniviDbusLEAdapterProxyIf implementation --- end

} //ccdbusif
