/**
 * @file EvolutionGeniviDbusHfpModemProxyIf.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Proxy for HfpModem.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details EvolutionGenivi DBUS proxy for HfpModem.
 */

#include "EvolutionGeniviDbusHfpModemProxyIf.h"
#include "IEvolutionGeniviDbusHfpModemCallbackIf.h"
#include "CcDbusIfTypes.h"
#include "CcDbusIfUtils.h"
#include "ICcDbusIfControllerClient.h"
#include "CcDbusIfCreateProxyWorkItem.h"
#include "CcDbusIfCreateRuntimeProxyWorkItem.h"
#include "CcDbusIfDestroyRuntimeProxyWorkItem.h"
#include "EvolutionGeniviDbusParser.h"

#include "TraceClasses.h"
#define FW_S_IMPORT_INTERFACE_TRACE
#include "framework_if_if.h"

using namespace ::org::ofono::Modem;
using namespace ::asf::core;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CCDBUSIF_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/EvolutionGeniviDbusHfpModemProxyIf.cpp.trc.h"
#endif
#endif

namespace ccdbusif {

EvolutionGeniviDbusHfpModemProxyIf::EvolutionGeniviDbusHfpModemProxyIf()
{
   // _hfpModemProxy is set later
   _callbackIf = NULL;
}

EvolutionGeniviDbusHfpModemProxyIf::EvolutionGeniviDbusHfpModemProxyIf(ICcDbusIfControllerClient* client) : BaseDbusProxyIf(client)
{
   // _hfpModemProxy is set later
   _callbackIf = NULL;
}

EvolutionGeniviDbusHfpModemProxyIf::~EvolutionGeniviDbusHfpModemProxyIf()
{
   _callbackIf = NULL;
}

void EvolutionGeniviDbusHfpModemProxyIf::createProxy(void)
{
   // these proxies have to be created during runtime
}

void EvolutionGeniviDbusHfpModemProxyIf::destroyProxy(void)
{
   ETG_TRACE_USR3((" destroyProxy(): Modem"));

   // these proxies have to be destroyed during runtime
   // destroy all proxies now if still available
   _hfpModemProxy.resetAllProxiesAndClear();
}

void EvolutionGeniviDbusHfpModemProxyIf::destroyAllRuntimeProxies(void)
{
   ETG_TRACE_USR3((" destroyAllRuntimeProxies(): Modem"));

   for(unsigned int i = 0; i < _hfpModemProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< ModemProxy >& hfpModemProxy = _hfpModemProxy[i];
      if(NULL != hfpModemProxy.get())
      {
         hfpModemProxy->sendDeregisterAll();
      }
   }

   _hfpModemProxy.resetAllProxiesAndClear();
}

void EvolutionGeniviDbusHfpModemProxyIf::createRuntimeProxy(const ::std::string& objPath)
{
   ETG_TRACE_USR3((" createRuntimeProxy(): Modem: objPath=%s", objPath.c_str()));

   if((NULL != _callbackIf) && (true == _enableProxy))
   {
      if(false == _hfpModemProxy.isInstanceAvailable(objPath))
      {
         evolution::EvolutionGeniviDbusParser parser;
         ::boost::shared_ptr< ModemProxy > hfpModemProxy = ModemProxy::createProxy("evoHfpModemPort", parser.getInterface2BusName(evolution::IF_MODEM), objPath, convertLocalBusType(parser.getInterface2BusType(evolution::IF_MODEM)), *this);
         _hfpModemProxy.addProxyInstance(objPath, hfpModemProxy);
      }
   }
}

void EvolutionGeniviDbusHfpModemProxyIf::destroyRuntimeProxy(const ::std::string& objPath)
{
   ETG_TRACE_USR3((" destroyRuntimeProxy(): Modem: objPath=%s", objPath.c_str()));

   // HINT: destroying runtime proxy is not possible because ASF is storing a proxy "reference" internally; therefore reset() call will not causing destruction of proxy
   // HINT: due to this we explicitly have to call sendDeregisterAll() to force removing of signal registration (DBUS: member=RemoveMatch)
   // HINT: sendDeregisterAll() is automatically called during destruction of proxy

   ::boost::shared_ptr< ModemProxy > hfpModemProxy;
   if(true == _hfpModemProxy.isProxyAvailable(hfpModemProxy, objPath))
   {
      hfpModemProxy->sendDeregisterAll();
   }
   _hfpModemProxy.removeProxyInstance(objPath);
}

void EvolutionGeniviDbusHfpModemProxyIf::setCallbackIf(IEvolutionGeniviDbusHfpModemCallbackIf* callbackIf, bool enableProxy)
{
   _callbackIf = callbackIf;

   if(NULL != _callbackIf)
   {
      _enableProxy = enableProxy;
   }

   // callback interface is now available => but proxy will be created later
}

void EvolutionGeniviDbusHfpModemProxyIf::modemCreated(const ::std::string& objPath)
{
   // create runtime proxy
   if((NULL != _callbackIf) && (NULL != _controllerClient))
   {
      _controllerClient->pushWorkItem(new CcDbusIfCreateRuntimeProxyWorkItem(objPath, this));
   }
}

void EvolutionGeniviDbusHfpModemProxyIf::modemRemoved(const ::std::string& objPath, const bool destroyDirectly /*= false*/)
{
   // destroy runtime proxy
   if (true == destroyDirectly)
   {
      // Called within context of ASF component thread, we can directly destroy the proxy.
      destroyRuntimeProxy(objPath);
   }
   else
   {
      if ((NULL != _callbackIf) && (NULL != _controllerClient))
      {
         _controllerClient->pushWorkItem(new CcDbusIfDestroyRuntimeProxyWorkItem(objPath, this));
      }
   }
}

// ServiceAvailableIF implementation --- start
void EvolutionGeniviDbusHfpModemProxyIf::onAvailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3((" onAvailable(): Modem"));

   for(unsigned int i = 0; i < _hfpModemProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< ModemProxy >& hfpModemProxy = _hfpModemProxy[i];
      if((NULL != hfpModemProxy.get()) && (proxy == hfpModemProxy))
      {
         if(NULL != _callbackIf)
         {
            hfpModemProxy->sendPropertyChangedRegister(*_callbackIf);
            hfpModemProxy->sendSCOConnectRequestRegister(*_callbackIf);

            _callbackIf->onProxyAvailable(stateChange.getPreviousState(), stateChange.getCurrentState(), hfpModemProxy->getDBusObjectPath());
         }
         break;
      }
   }
}

void EvolutionGeniviDbusHfpModemProxyIf::onUnavailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3((" onUnavailable(): Modem"));

   for(unsigned int i = 0; i < _hfpModemProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< ModemProxy >& hfpModemProxy = _hfpModemProxy[i];
      if((NULL != hfpModemProxy.get()) && (proxy == hfpModemProxy))
      {
         hfpModemProxy->sendDeregisterAll();

         if(NULL != _callbackIf)
         {
            _callbackIf->onProxyUnavailable(stateChange.getPreviousState(), stateChange.getCurrentState(), hfpModemProxy->getDBusObjectPath());
         }
         break;
      }
   }
}
// ServiceAvailableIF implementation --- end

// IEvolutionGeniviDbusHfpModemProxyIf implementation --- start
act_t EvolutionGeniviDbusHfpModemProxyIf::sendGetPropertiesRequest(const ::std::string& objPath)
{
   ::boost::shared_ptr< ModemProxy > proxy;
   if(true == _hfpModemProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendGetPropertiesRequest(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusHfpModemProxyIf::sendSetPropertyRequest(const ::std::string& objPath, const ::std::string& property, const bool value)
{
   ::boost::shared_ptr< ModemProxy > proxy;
   if(true == _hfpModemProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         ::asf::dbus::DBusVariant variant;
         DbusVariant vValue;
         vValue.setBool(value);
         evolution::EvolutionGeniviDbusParser parser;
         parser.addVariantValue(variant, vValue);
         return proxy->sendSetPropertyRequest(*_callbackIf, property, variant);
      }
   }
   return DEFAULT_ACT;
}

act_t EvolutionGeniviDbusHfpModemProxyIf::sendAcceptSCOConnectRequest(const ::std::string& objPath)
{
   ::boost::shared_ptr< ModemProxy > proxy;
   if(true == _hfpModemProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendAcceptSCOConnectRequest(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}
// IEvolutionGeniviDbusHfpModemProxyIf implementation --- end

} //ccdbusif
