/**
 * @file EvolutionGeniviDbusDunProxyIf.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Proxy for Dun.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details EvolutionGenivi DBUS proxy for Dun.
 */

#include "EvolutionGeniviDbusDunProxyIf.h"
#include "IEvolutionGeniviDbusDunCallbackIf.h"
#include "CcDbusIfTypes.h"
#include "ICcDbusIfControllerClient.h"
#include "CcDbusIfCreateProxyWorkItem.h"

using namespace ::org::alps::evo::DUN;
using namespace ::asf::core;

namespace ccdbusif {

EvolutionGeniviDbusDunProxyIf::EvolutionGeniviDbusDunProxyIf()
{
   // _dunProxy is set later
   _callbackIf = NULL;
}

EvolutionGeniviDbusDunProxyIf::EvolutionGeniviDbusDunProxyIf(ICcDbusIfControllerClient* client) : BaseDbusProxyIf(client)
{
   // _dunProxy is set later
   _callbackIf = NULL;
}

EvolutionGeniviDbusDunProxyIf::~EvolutionGeniviDbusDunProxyIf()
{
   _callbackIf = NULL;
}

void EvolutionGeniviDbusDunProxyIf::createProxy(void)
{
   if(NULL != _callbackIf)
   {
      if(true == _enableProxy)
      {
         _dunProxy = DUNProxy::createProxy("evoDunPort", *this);
      }
   }
}

void EvolutionGeniviDbusDunProxyIf::destroyProxy(void)
{
   if(NULL != _dunProxy.get())
   {
      _dunProxy->sendDeregisterAll();
      _dunProxy.reset();
   }
}

void EvolutionGeniviDbusDunProxyIf::destroyAllRuntimeProxies(void)
{
   // nothing to do because this is a static proxy
}

void EvolutionGeniviDbusDunProxyIf::setCallbackIf(IEvolutionGeniviDbusDunCallbackIf* callbackIf, bool enableProxy)
{
   _callbackIf = callbackIf;

   if(NULL != _callbackIf)
   {
      _enableProxy = enableProxy;
   }

   // callback interface is now available => create proxy now
   if((NULL != _callbackIf) && (NULL != _controllerClient))
   {
      _controllerClient->pushWorkItem(new CcDbusIfCreateProxyWorkItem(this));
   }
}

// ServiceAvailableIF implementation --- start
void EvolutionGeniviDbusDunProxyIf::onAvailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   if((NULL != _dunProxy.get()) && (proxy == _dunProxy))
   {
      if(NULL != _callbackIf)
      {
         _dunProxy->sendBtApplDunSendDataCfmRegister(*_callbackIf);
         _dunProxy->sendBtApplDunSendIndRegister(*_callbackIf);
         _dunProxy->sendBtApplDunDataIndRegister(*_callbackIf);
         _dunProxy->sendBtApplDunDataSendFcIndRegister(*_callbackIf);
         _dunProxy->sendBtApplDunFlowCtrlCfmRegister(*_callbackIf);

         _callbackIf->onProxyAvailable(stateChange.getPreviousState(), stateChange.getCurrentState(), _dunProxy->getDBusObjectPath());
      }
   }
}

void EvolutionGeniviDbusDunProxyIf::onUnavailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   if((NULL != _dunProxy.get()) && (proxy == _dunProxy))
   {
      _dunProxy->sendDeregisterAll();

      if(NULL != _callbackIf)
      {
         _callbackIf->onProxyUnavailable(stateChange.getPreviousState(), stateChange.getCurrentState(), _dunProxy->getDBusObjectPath());
      }
   }
}
// ServiceAvailableIF implementation --- end

// IEvolutionGeniviDbusDunProxyIf implementation --- start
act_t EvolutionGeniviDbusDunProxyIf::sendBtApplDunSendDataReqRequest(const ::std::vector< uint8 >& data)
{
   if((NULL != _callbackIf) && (NULL != _dunProxy.get()))
   {
      return _dunProxy->sendBtApplDunSendDataReqRequest(*_callbackIf, data);
   }
   else
   {
      return DEFAULT_ACT;
   }
}

act_t EvolutionGeniviDbusDunProxyIf::sendBtApplDunFlowCtrlReqRequest(uint8 action)
{
   if((NULL != _callbackIf) && (NULL != _dunProxy.get()))
   {
      return _dunProxy->sendBtApplDunFlowCtrlReqRequest(*_callbackIf, action);
   }
   else
   {
      return DEFAULT_ACT;
   }
}
// IEvolutionGeniviDbusDunProxyIf implementation --- end

} //ccdbusif
