/**
 * @file EvolutionGeniviDbusDMProxyIf.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Proxy for DM.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details EvolutionGenivi DBUS proxy for DM.
 */

#include "EvolutionGeniviDbusDMProxyIf.h"
#include "IEvolutionGeniviDbusDMCallbackIf.h"
#include "CcDbusIfTypes.h"
#include "ICcDbusIfControllerClient.h"
#include "CcDbusIfCreateProxyWorkItem.h"

#include "TraceClasses.h"
#define FW_S_IMPORT_INTERFACE_TRACE
#include "framework_if_if.h"

using namespace ::org::alps::evo::DM;
using namespace ::asf::core;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CCDBUSIF_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/EvolutionGeniviDbusDMProxyIf.cpp.trc.h"
#endif
#endif

namespace ccdbusif {

EvolutionGeniviDbusDMProxyIf::EvolutionGeniviDbusDMProxyIf()
{
   // _dMProxy is set later
   _callbackIf = NULL;
}

EvolutionGeniviDbusDMProxyIf::EvolutionGeniviDbusDMProxyIf(ICcDbusIfControllerClient* client) : BaseDbusProxyIf(client)
{
   // _dMProxy is set later
   _callbackIf = NULL;
}

EvolutionGeniviDbusDMProxyIf::~EvolutionGeniviDbusDMProxyIf()
{
   _callbackIf = NULL;
}

void EvolutionGeniviDbusDMProxyIf::createProxy(void)
{
   ETG_TRACE_USR3((" createProxy(): DM"));

   if(NULL != _callbackIf)
   {
      if(true == _enableProxy)
      {
         _dMProxy = DMProxy::createProxy("evoDmPort", *this);
      }
   }
}

void EvolutionGeniviDbusDMProxyIf::destroyProxy(void)
{
   ETG_TRACE_USR3((" destroyProxy(): DM"));

   if(NULL != _dMProxy.get())
   {
      _dMProxy->sendDeregisterAll();
      _dMProxy.reset();
   }
}

void EvolutionGeniviDbusDMProxyIf::destroyAllRuntimeProxies(void)
{
   ETG_TRACE_USR3((" destroyAllRuntimeProxies(): DM"));

   // nothing to do because this is a static proxy
}

void EvolutionGeniviDbusDMProxyIf::setCallbackIf(IEvolutionGeniviDbusDMCallbackIf* callbackIf, bool enableProxy)
{
   _callbackIf = callbackIf;

   if(NULL != _callbackIf)
   {
      _enableProxy = enableProxy;
   }

   // callback interface is now available => create proxy now
   if((NULL != _callbackIf) && (NULL != _controllerClient))
   {
      _controllerClient->pushWorkItem(new CcDbusIfCreateProxyWorkItem(this));
   }
}

// ServiceAvailableIF implementation --- start
void EvolutionGeniviDbusDMProxyIf::onAvailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3((" onAvailable(): DM"));

   if((NULL != _dMProxy.get()) && (proxy == _dMProxy))
   {
      if(NULL != _callbackIf)
      {
         _dMProxy->sendBtApplDeviceConnectCfmRegister(*_callbackIf);
         _dMProxy->sendBtApplDeviceConnectIndRegister(*_callbackIf);
         _dMProxy->sendBtApplDeviceDisconnectCfmRegister(*_callbackIf);
         _dMProxy->sendBtApplDeviceDisconnectIndRegister(*_callbackIf);
         _dMProxy->sendBtApplDeviceDisconnectCompIndRegister(*_callbackIf);
         _dMProxy->sendBtApplDeviceConnectCompIndRegister(*_callbackIf);
         _dMProxy->sendBtApplCancelConnectCfmRegister(*_callbackIf);
         _dMProxy->sendBtApplStatusNotificationIndRegister(*_callbackIf);
         _dMProxy->sendBtApplGetHwVersionCfmRegister(*_callbackIf);
         _dMProxy->sendBtApplSetConfigurationCfmRegister(*_callbackIf);
         _dMProxy->sendBtApplReadConfigurationCfmRegister(*_callbackIf);
         _dMProxy->sendBtApplTestModeCfmRegister(*_callbackIf);
         _dMProxy->sendBtApplTestModeCompIndRegister(*_callbackIf);
         _dMProxy->sendBtApplTestModeLinkQualityCfmRegister(*_callbackIf);

         _callbackIf->onProxyAvailable(stateChange.getPreviousState(), stateChange.getCurrentState(), _dMProxy->getDBusObjectPath());
      }
   }
}

void EvolutionGeniviDbusDMProxyIf::onUnavailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3((" onUnavailable(): DM"));

   if((NULL != _dMProxy.get()) && (proxy == _dMProxy))
   {
      _dMProxy->sendDeregisterAll();

      if(NULL != _callbackIf)
      {
         _callbackIf->onProxyUnavailable(stateChange.getPreviousState(), stateChange.getCurrentState(), _dMProxy->getDBusObjectPath());
      }
   }
}
// ServiceAvailableIF implementation --- end

// IEvolutionGeniviDbusDMProxyIf implementation --- start
act_t EvolutionGeniviDbusDMProxyIf::sendBtApplDeviceConnectReqRequest(const ::std::vector< uint8 >& address, uint64 function, uint8 instance)
{
   if((NULL != _callbackIf) && (NULL != _dMProxy.get()))
   {
      return _dMProxy->sendBtApplDeviceConnectReqRequest(*_callbackIf, address, function, instance);
   }
   else
   {
      return DEFAULT_ACT;
   }
}

act_t EvolutionGeniviDbusDMProxyIf::sendBtApplDeviceDisconnectReqRequest(uint64 function, uint8 instance)
{
   if((NULL != _callbackIf) && (NULL != _dMProxy.get()))
   {
      return _dMProxy->sendBtApplDeviceDisconnectReqRequest(*_callbackIf, function, instance);
   }
   else
   {
      return DEFAULT_ACT;
   }
}

act_t EvolutionGeniviDbusDMProxyIf::sendBtApplCancelConnectReqRequest(void)
{
   if((NULL != _callbackIf) && (NULL != _dMProxy.get()))
   {
      return _dMProxy->sendBtApplCancelConnectReqRequest(*_callbackIf);
   }
   else
   {
      return DEFAULT_ACT;
   }
}

act_t EvolutionGeniviDbusDMProxyIf::sendBtApplGetHwVersionReqRequest(void)
{
   if((NULL != _callbackIf) && (NULL != _dMProxy.get()))
   {
      return _dMProxy->sendBtApplGetHwVersionReqRequest(*_callbackIf);
   }
   else
   {
      return DEFAULT_ACT;
   }
}

act_t EvolutionGeniviDbusDMProxyIf::sendBtApplSetConfigurationReqRequest(uint8 instance, uint16 config_id, const ::std::vector< uint8 >& data)
{
   if((NULL != _callbackIf) && (NULL != _dMProxy.get()))
   {
      return _dMProxy->sendBtApplSetConfigurationReqRequest(*_callbackIf, instance, config_id, data);
   }
   else
   {
      return DEFAULT_ACT;
   }
}

act_t EvolutionGeniviDbusDMProxyIf::sendBtApplReadConfigurationReqRequest(uint8 instance, uint16 config_id)
{
   if((NULL != _callbackIf) && (NULL != _dMProxy.get()))
   {
      return _dMProxy->sendBtApplReadConfigurationReqRequest(*_callbackIf, instance, config_id);
   }
   else
   {
      return DEFAULT_ACT;
   }
}

act_t EvolutionGeniviDbusDMProxyIf::sendBtApplSspDebugModeReqRequest(uint8 mode)
{
   if((NULL != _callbackIf) && (NULL != _dMProxy.get()))
   {
      return _dMProxy->sendBtApplSspDebugModeReqRequest(*_callbackIf, mode);
   }
   else
   {
      return DEFAULT_ACT;
   }
}

act_t EvolutionGeniviDbusDMProxyIf::sendBtApplTestModeReqRequest(uint8 enable, uint8 mode, uint8 role, const ::std::vector< uint8 >& addr)
{
   if((NULL != _callbackIf) && (NULL != _dMProxy.get()))
   {
      return _dMProxy->sendBtApplTestModeReqRequest(*_callbackIf, enable, mode, role, addr);
   }
   else
   {
      return DEFAULT_ACT;
   }
}

act_t EvolutionGeniviDbusDMProxyIf::sendBtApplTestModeLinkQualityReqRequest(uint8 mode)
{
   if((NULL != _callbackIf) && (NULL != _dMProxy.get()))
   {
      return _dMProxy->sendBtApplTestModeLinkQualityReqRequest(*_callbackIf, mode);
   }
   else
   {
      return DEFAULT_ACT;
   }
}
// IEvolutionGeniviDbusDMProxyIf implementation --- end

} //ccdbusif
