/**
 * @file EvolutionGeniviDbusAgentManagerProxyIf.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Proxy for AgentManager.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details EvolutionGenivi DBUS proxy for AgentManager.
 */

#include "EvolutionGeniviDbusAgentManagerProxyIf.h"
#include "IEvolutionGeniviDbusAgentManagerCallbackIf.h"
#include "CcDbusIfTypes.h"
#include "ICcDbusIfControllerClient.h"
#include "CcDbusIfCreateProxyWorkItem.h"

#include "TraceClasses.h"
#define FW_S_IMPORT_INTERFACE_TRACE
#include "framework_if_if.h"

using namespace ::org::bluez::AgentManager;
using namespace ::asf::core;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CCDBUSIF_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/EvolutionGeniviDbusAgentManagerProxyIf.cpp.trc.h"
#endif
#endif

namespace ccdbusif {

EvolutionGeniviDbusAgentManagerProxyIf::EvolutionGeniviDbusAgentManagerProxyIf()
{
   // _agentManagerProxy is set later
   _callbackIf = NULL;
}

EvolutionGeniviDbusAgentManagerProxyIf::EvolutionGeniviDbusAgentManagerProxyIf(ICcDbusIfControllerClient* client) : BaseDbusProxyIf(client)
{
   // _agentManagerProxy is set later
   _callbackIf = NULL;
}

EvolutionGeniviDbusAgentManagerProxyIf::~EvolutionGeniviDbusAgentManagerProxyIf()
{
   _callbackIf = NULL;
}

void EvolutionGeniviDbusAgentManagerProxyIf::createProxy(void)
{
   ETG_TRACE_USR3((" createProxy(): AgentManager"));

   // HINT: this interface is called within context of thread started in CcDbusIfController.cpp ("CC_DBUS_IF_THD")

   // HINT: without/before proxy creation _agentManagerProxy.get() returns 0

   if(NULL != _callbackIf)
   {
      if(true == _enableProxy)
      {
         _agentManagerProxy = AgentManagerProxy::createProxy("evoAgentManagerPort", *this);
      }
   }

   // HINT: with/after proxy creation _agentManagerProxy.get() returns non-0
}

void EvolutionGeniviDbusAgentManagerProxyIf::destroyProxy(void)
{
   ETG_TRACE_USR3((" destroyProxy(): AgentManager"));

   // HINT: this interface is called within context of thread started in CcDbusIfController.cpp ("CC_DBUS_IF_THD")

   // HINT: without/before proxy deletion _agentManagerProxy.get() returns non-0

   if(NULL != _agentManagerProxy.get())
   {
      // nothing to deregister
      _agentManagerProxy.reset();
   }

   // HINT: with/after proxy deletion _agentManagerProxy.get() returns 0
}

void EvolutionGeniviDbusAgentManagerProxyIf::destroyAllRuntimeProxies(void)
{
   ETG_TRACE_USR3((" destroyAllRuntimeProxies(): AgentManager"));

   // nothing to do because this is a static proxy
}

void EvolutionGeniviDbusAgentManagerProxyIf::setCallbackIf(IEvolutionGeniviDbusAgentManagerCallbackIf* callbackIf, bool enableProxy)
{
   _callbackIf = callbackIf;

   // HINT: act is provided via response->getAct() / error->getAct() in callback message
   // HINT: act shall be given to receiver and can be compared with act value returned by send function

   if(NULL != _callbackIf)
   {
      _enableProxy = enableProxy;
   }

   // callback interface is now available => create proxy now
   if((NULL != _callbackIf) && (NULL != _controllerClient))
   {
      _controllerClient->pushWorkItem(new CcDbusIfCreateProxyWorkItem(this));
   }
}

// ServiceAvailableIF implementation --- start
void EvolutionGeniviDbusAgentManagerProxyIf::onAvailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3((" onAvailable(): AgentManager"));

   if((NULL != _agentManagerProxy.get()) && (proxy == _agentManagerProxy))
   {
      if(NULL != _callbackIf)
      {
         // nothing to register

         _callbackIf->onProxyAvailable(stateChange.getPreviousState(), stateChange.getCurrentState(), _agentManagerProxy->getDBusObjectPath());
      }
   }
}

void EvolutionGeniviDbusAgentManagerProxyIf::onUnavailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3((" onUnavailable(): AgentManager"));

   if((NULL != _agentManagerProxy.get()) && (proxy == _agentManagerProxy))
   {
      // nothing to deregister

      if(NULL != _callbackIf)
      {
         _callbackIf->onProxyUnavailable(stateChange.getPreviousState(), stateChange.getCurrentState(), _agentManagerProxy->getDBusObjectPath());
      }
   }
}
// ServiceAvailableIF implementation --- end

// HINT: following interfaces are called within context of thread started in CcDbusIfController.cpp ("CC_DBUS_IF_THD")

// IEvolutionGeniviDbusAgentManagerProxyIf implementation --- start
act_t EvolutionGeniviDbusAgentManagerProxyIf::sendRegisterAgentRequest(const ::std::string& agent, const ::std::string& capability)
{
   if((NULL != _callbackIf) && (NULL != _agentManagerProxy.get()))
   {
      return _agentManagerProxy->sendRegisterAgentRequest(*_callbackIf, agent, capability);
   }
   else
   {
      return DEFAULT_ACT;
   }
}

act_t EvolutionGeniviDbusAgentManagerProxyIf::sendUnregisterAgentRequest(const ::std::string& agent)
{
   if((NULL != _callbackIf) && (NULL != _agentManagerProxy.get()))
   {
      return _agentManagerProxy->sendUnregisterAgentRequest(*_callbackIf, agent);
   }
   else
   {
      return DEFAULT_ACT;
   }
}

act_t EvolutionGeniviDbusAgentManagerProxyIf::sendRequestDefaultAgentRequest(const ::std::string& agent)
{
   if((NULL != _callbackIf) && (NULL != _agentManagerProxy.get()))
   {
      return _agentManagerProxy->sendRequestDefaultAgentRequest(*_callbackIf, agent);
   }
   else
   {
      return DEFAULT_ACT;
   }
}
// IEvolutionGeniviDbusAgentManagerProxyIf implementation --- end

} //ccdbusif
