/**
 * @file EvolutionGeniviDbusAdapterProxyIf.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS proxy for Adapter.
 *
 * @copyright (C) 2016 - 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details EvolutionGenivi DBUS proxy for Adapter.
 */

#include "EvolutionGeniviDbusAdapterProxyIf.h"
#include "EvolutionGeniviDbusParser.h"
#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

using namespace ::asf::core;
using namespace ::org::bluez::Adapter1;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CCDBUSIF_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/EvolutionGeniviDbusAdapterProxyIf.cpp.trc.h"
#endif
#endif

namespace ccdbusif {
namespace evolution {

EvolutionGeniviDbusAdapterProxyIf::EvolutionGeniviDbusAdapterProxyIf(ICcDbusIfControllerClient* client) :
DbusBaseProxyIf< IEvolutionGeniviDbusAdapterCallbackIf, Adapter1Proxy, IEvolutionGeniviDbusAdapterTestProxyIf >(client)
{
}

EvolutionGeniviDbusAdapterProxyIf::EvolutionGeniviDbusAdapterProxyIf(ICcDbusIfControllerClient* client, IEvolutionGeniviDbusAdapterTestProxyIf* testProxyIf) :
DbusBaseProxyIf< IEvolutionGeniviDbusAdapterCallbackIf, Adapter1Proxy, IEvolutionGeniviDbusAdapterTestProxyIf >(client, testProxyIf)
{
}

EvolutionGeniviDbusAdapterProxyIf::~EvolutionGeniviDbusAdapterProxyIf()
{
}

void EvolutionGeniviDbusAdapterProxyIf::onAvailable(const boost::shared_ptr< Proxy >& proxy, const ServiceStateChange& stateChange)
{
   for(size_t i = 0; i < getProxyListSize(); i++)
   {
      ::boost::shared_ptr< Adapter1Proxy >* proxyPtr(getProxy(i));
      if(0 != proxyPtr)
      {
         ::boost::shared_ptr< Adapter1Proxy >& myProxy = *proxyPtr;
         if((0 != myProxy.get()) && (proxy == myProxy))
         {
            (void)myProxy->sendAdapterInitializationRegister(*this);
            (void)myProxy->sendDeviceCreatedRegister(*this);
            (void)myProxy->sendDeviceRemovedRegister(*this);
            (void)myProxy->sendBtStackLogRegister(*this);
            (void)myProxy->sendHCIEventRegister(*this);
            (void)myProxy->sendDeviceFoundRegister(*this);
            (void)myProxy->sendAddressRegister(*this);
            (void)myProxy->sendClassRegister(*this);
            (void)myProxy->sendAliasRegister(*this);
            (void)myProxy->sendPoweredRegister(*this);
            (void)myProxy->sendDiscoverableRegister(*this);
            (void)myProxy->sendDiscoverableTimeoutRegister(*this);
            (void)myProxy->sendPairableRegister(*this);
            (void)myProxy->sendPairableTimeoutRegister(*this);
            (void)myProxy->sendDiscoveringRegister(*this);
            (void)myProxy->sendUUIDsRegister(*this);
            (void)myProxy->sendWBSModeRegister(*this);
            (void)myProxy->sendConnectableRegister(*this);
            (void)myProxy->sendHCIModeRegister(*this);
            (void)myProxy->sendVersionRegister(*this);
            (void)myProxy->sendTraceLevelRegister(*this);
            (void)myProxy->sendTraceCategoryRegister(*this);
            (void)myProxy->sendAvailableSPPServersRegister(*this);

            internalOnAvailable(ServiceState__Available == stateChange.getCurrentState(), myProxy->getDBusObjectPath(), myProxy->getDBusBusName(), convertConnectorOption2BusType(myProxy->getConnectorOptions()));
            break;
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onUnavailable(const boost::shared_ptr< Proxy >& proxy, const ServiceStateChange& stateChange)
{
   for(size_t i = 0; i < getProxyListSize(); i++)
   {
      ::boost::shared_ptr< Adapter1Proxy >* proxyPtr(getProxy(i));
      if(0 != proxyPtr)
      {
         ::boost::shared_ptr< Adapter1Proxy >& myProxy = *proxyPtr;
         if((0 != myProxy.get()) && (proxy == myProxy))
         {
            myProxy->sendDeregisterAll();

            internalOnUnavailable(ServiceState__Available == stateChange.getCurrentState(), myProxy->getDBusObjectPath(), myProxy->getDBusBusName(), convertConnectorOption2BusType(myProxy->getConnectorOptions()));
            break;
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::handleSetCallback(IEvolutionGeniviDbusAdapterCallbackIf* callbackIf, const bool enableProxy, const unsigned int callbackId, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   storeCallback(callbackIf, enableProxy, callbackId); // independent of path, name, type => update status to all
}

void EvolutionGeniviDbusAdapterProxyIf::handleCreateProxy(const unsigned int callbackId, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   internalCreateProxy(callbackId, objPath, busName, busType);
}

void EvolutionGeniviDbusAdapterProxyIf::handleDestroyProxy(const unsigned int callbackId, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   internalDestroyProxy(callbackId, objPath, busName, busType);
}

void EvolutionGeniviDbusAdapterProxyIf::handleDestroyAllProxies(void)
{
   internalDestroyAllProxies();
}

void EvolutionGeniviDbusAdapterProxyIf::handleDestroyAllRuntimeProxies(void)
{
   internalDestroyAllRuntimeProxies();
}

IDestroyAllProxies* EvolutionGeniviDbusAdapterProxyIf::getDestroyer(void)
{
   return this;
}

void EvolutionGeniviDbusAdapterProxyIf::setCallbackIf(IEvolutionGeniviDbusAdapterCallbackIf* callbackIf, const bool enableProxy, const unsigned int callbackId, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   processSetCallback(this, callbackIf, enableProxy, callbackId, objPath, busName, busType);
}

void EvolutionGeniviDbusAdapterProxyIf::createProxyIf(const unsigned int callbackId, const bool createDirectly /*= false*/, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   processCreateProxy(this, createDirectly, callbackId, objPath, busName, busType);
}

void EvolutionGeniviDbusAdapterProxyIf::destroyProxyIf(const unsigned int callbackId, const bool createDirectly /*= false*/, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   processDestroyProxy(this, createDirectly, callbackId, objPath, busName, busType);
}

act_t EvolutionGeniviDbusAdapterProxyIf::sendStartDiscoveryRequest(const unsigned int callbackId)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         retAct = _testProxyIf->sendStartDiscoveryRequest(*this);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         retAct = proxy->sendStartDiscoveryRequest(*this);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

act_t EvolutionGeniviDbusAdapterProxyIf::sendStopDiscoveryRequest(const unsigned int callbackId)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         retAct = _testProxyIf->sendStopDiscoveryRequest(*this);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         retAct = proxy->sendStopDiscoveryRequest(*this);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

act_t EvolutionGeniviDbusAdapterProxyIf::sendRemoveDeviceRequest(const unsigned int callbackId, const ::std::string& device)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         retAct = _testProxyIf->sendRemoveDeviceRequest(*this, device);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         retAct = proxy->sendRemoveDeviceRequest(*this, device);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

act_t EvolutionGeniviDbusAdapterProxyIf::sendCreateDeviceRequest(const unsigned int callbackId, const ::std::string& address)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         retAct = _testProxyIf->sendCreateDeviceRequest(*this, address);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         retAct = proxy->sendCreateDeviceRequest(*this, address);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

act_t EvolutionGeniviDbusAdapterProxyIf::sendSendHCICmdRequest(const unsigned int callbackId, const ::std::vector< uint8 >& data)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         retAct = _testProxyIf->sendSendHCICmdRequest(*this, data);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         retAct = proxy->sendSendHCICmdRequest(*this, data);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

act_t EvolutionGeniviDbusAdapterProxyIf::sendAddressGet(const unsigned int callbackId)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         retAct = _testProxyIf->sendAddressGet(*this);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         retAct = proxy->sendAddressGet(*this);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

act_t EvolutionGeniviDbusAdapterProxyIf::sendClassGet(const unsigned int callbackId)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         retAct = _testProxyIf->sendClassGet(*this);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         retAct = proxy->sendClassGet(*this);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

act_t EvolutionGeniviDbusAdapterProxyIf::sendAliasGet(const unsigned int callbackId)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         retAct = _testProxyIf->sendAliasGet(*this);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         retAct = proxy->sendAliasGet(*this);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

void EvolutionGeniviDbusAdapterProxyIf::sendAliasSet(const unsigned int callbackId, const ::std::string& alias)
{
   (void)(callbackId);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         _testProxyIf->sendAliasSet(*this, alias);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         proxy->sendAliasSet(alias);
      }
   }
}

act_t EvolutionGeniviDbusAdapterProxyIf::sendPoweredGet(const unsigned int callbackId)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         retAct = _testProxyIf->sendPoweredGet(*this);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         retAct = proxy->sendPoweredGet(*this);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

void EvolutionGeniviDbusAdapterProxyIf::sendPoweredSet(const unsigned int callbackId, bool powered)
{
   (void)(callbackId);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         _testProxyIf->sendPoweredSet(*this, powered);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         proxy->sendPoweredSet(powered);
      }
   }
}

act_t EvolutionGeniviDbusAdapterProxyIf::sendDiscoverableGet(const unsigned int callbackId)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         retAct = _testProxyIf->sendDiscoverableGet(*this);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         retAct = proxy->sendDiscoverableGet(*this);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

void EvolutionGeniviDbusAdapterProxyIf::sendDiscoverableSet(const unsigned int callbackId, bool discoverable)
{
   (void)(callbackId);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         _testProxyIf->sendDiscoverableSet(*this, discoverable);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         proxy->sendDiscoverableSet(discoverable);
      }
   }
}

act_t EvolutionGeniviDbusAdapterProxyIf::sendDiscoverableTimeoutGet(const unsigned int callbackId)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         retAct = _testProxyIf->sendDiscoverableTimeoutGet(*this);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         retAct = proxy->sendDiscoverableTimeoutGet(*this);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

void EvolutionGeniviDbusAdapterProxyIf::sendDiscoverableTimeoutSet(const unsigned int callbackId, uint32 discoverableTimeout)
{
   (void)(callbackId);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         _testProxyIf->sendDiscoverableTimeoutSet(*this, discoverableTimeout);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         proxy->sendDiscoverableTimeoutSet(discoverableTimeout);
      }
   }
}

act_t EvolutionGeniviDbusAdapterProxyIf::sendPairableGet(const unsigned int callbackId)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         retAct = _testProxyIf->sendPairableGet(*this);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         retAct = proxy->sendPairableGet(*this);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

void EvolutionGeniviDbusAdapterProxyIf::sendPairableSet(const unsigned int callbackId, bool pairable)
{
   (void)(callbackId);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         _testProxyIf->sendPairableSet(*this, pairable);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         proxy->sendPairableSet(pairable);
      }
   }
}

act_t EvolutionGeniviDbusAdapterProxyIf::sendPairableTimeoutGet(const unsigned int callbackId)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         retAct = _testProxyIf->sendPairableTimeoutGet(*this);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         retAct = proxy->sendPairableTimeoutGet(*this);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

void EvolutionGeniviDbusAdapterProxyIf::sendPairableTimeoutSet(const unsigned int callbackId, uint32 pairableTimeout)
{
   (void)(callbackId);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         _testProxyIf->sendPairableTimeoutSet(*this, pairableTimeout);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         proxy->sendPairableTimeoutSet(pairableTimeout);
      }
   }
}

act_t EvolutionGeniviDbusAdapterProxyIf::sendDiscoveringGet(const unsigned int callbackId)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         retAct = _testProxyIf->sendDiscoveringGet(*this);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         retAct = proxy->sendDiscoveringGet(*this);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

act_t EvolutionGeniviDbusAdapterProxyIf::sendUUIDsGet(const unsigned int callbackId)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         retAct = _testProxyIf->sendUUIDsGet(*this);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         retAct = proxy->sendUUIDsGet(*this);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

void EvolutionGeniviDbusAdapterProxyIf::sendUUIDsSet(const unsigned int callbackId, const ::std::vector< ::std::string >& uUIDs)
{
   (void)(callbackId);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         _testProxyIf->sendUUIDsSet(*this, uUIDs);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         proxy->sendUUIDsSet(uUIDs);
      }
   }
}

act_t EvolutionGeniviDbusAdapterProxyIf::sendWBSModeGet(const unsigned int callbackId)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         retAct = _testProxyIf->sendWBSModeGet(*this);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         retAct = proxy->sendWBSModeGet(*this);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

void EvolutionGeniviDbusAdapterProxyIf::sendWBSModeSet(const unsigned int callbackId, bool wBSMode)
{
   (void)(callbackId);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         _testProxyIf->sendWBSModeSet(*this, wBSMode);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         proxy->sendWBSModeSet(wBSMode);
      }
   }
}

act_t EvolutionGeniviDbusAdapterProxyIf::sendConnectableGet(const unsigned int callbackId)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         retAct = _testProxyIf->sendConnectableGet(*this);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         retAct = proxy->sendConnectableGet(*this);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

void EvolutionGeniviDbusAdapterProxyIf::sendConnectableSet(const unsigned int callbackId, bool connectable)
{
   (void)(callbackId);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         _testProxyIf->sendConnectableSet(*this, connectable);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         proxy->sendConnectableSet(connectable);
      }
   }
}

act_t EvolutionGeniviDbusAdapterProxyIf::sendHCIModeGet(const unsigned int callbackId)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         retAct = _testProxyIf->sendHCIModeGet(*this);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         retAct = proxy->sendHCIModeGet(*this);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

void EvolutionGeniviDbusAdapterProxyIf::sendHCIModeSet(const unsigned int callbackId, bool hCIMode)
{
   (void)(callbackId);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         _testProxyIf->sendHCIModeSet(*this, hCIMode);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         proxy->sendHCIModeSet(hCIMode);
      }
   }
}

act_t EvolutionGeniviDbusAdapterProxyIf::sendVersionGet(const unsigned int callbackId)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         retAct = _testProxyIf->sendVersionGet(*this);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         retAct = proxy->sendVersionGet(*this);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

act_t EvolutionGeniviDbusAdapterProxyIf::sendTraceLevelGet(const unsigned int callbackId)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         retAct = _testProxyIf->sendTraceLevelGet(*this);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         retAct = proxy->sendTraceLevelGet(*this);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

void EvolutionGeniviDbusAdapterProxyIf::sendTraceLevelSet(const unsigned int callbackId, const ::std::vector< ::std::string >& traceLevel)
{
   (void)(callbackId);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         _testProxyIf->sendTraceLevelSet(*this, traceLevel);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         proxy->sendTraceLevelSet(traceLevel);
      }
   }
}

act_t EvolutionGeniviDbusAdapterProxyIf::sendTraceCategoryGet(const unsigned int callbackId)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         retAct = _testProxyIf->sendTraceCategoryGet(*this);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         retAct = proxy->sendTraceCategoryGet(*this);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

void EvolutionGeniviDbusAdapterProxyIf::sendTraceCategorySet(const unsigned int callbackId, const ::std::vector< ::std::string >& traceCategory)
{
   (void)(callbackId);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         _testProxyIf->sendTraceCategorySet(*this, traceCategory);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         proxy->sendTraceCategorySet(traceCategory);
      }
   }
}

act_t EvolutionGeniviDbusAdapterProxyIf::sendAvailableSPPServersGet(const unsigned int callbackId)
{
   act_t retAct(DEFAULT_ACT);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         retAct = _testProxyIf->sendAvailableSPPServersGet(*this);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         retAct = proxy->sendAvailableSPPServersGet(*this);
      }
   }

   addAct(callbackId, retAct);

   return retAct;
}

void EvolutionGeniviDbusAdapterProxyIf::sendAvailableSPPServersSet(const unsigned int callbackId, const ::std::vector< ::std::string >& availableSPPServers)
{
   (void)(callbackId);

   if(0 != _testProxyIf)
   {
      if(true == isTestProxyAvailable()) // independent of path, name, type
      {
         _testProxyIf->sendAvailableSPPServersSet(*this, availableSPPServers);
      }
   }
   else
   {
      ::boost::shared_ptr< Adapter1Proxy > proxy;
      if(true == isProxyAvailable(proxy)) // independent of path, name, type
      {
         proxy->sendAvailableSPPServersSet(availableSPPServers);
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onStartDiscoveryError(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< StartDiscoveryError >& error)
{
   onStartDiscoveryErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onStartDiscoveryResponse(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< StartDiscoveryResponse >& response)
{
   onStartDiscoveryResponseCb(response, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onStopDiscoveryError(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< StopDiscoveryError >& error)
{
   onStopDiscoveryErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onStopDiscoveryResponse(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< StopDiscoveryResponse >& response)
{
   onStopDiscoveryResponseCb(response, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onRemoveDeviceError(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< RemoveDeviceError >& error)
{
   onRemoveDeviceErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onRemoveDeviceResponse(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< RemoveDeviceResponse >& response)
{
   onRemoveDeviceResponseCb(response, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onCreateDeviceError(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< CreateDeviceError >& error)
{
   onCreateDeviceErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onCreateDeviceResponse(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< CreateDeviceResponse >& response)
{
   onCreateDeviceResponseCb(response, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onSendHCICmdError(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< SendHCICmdError >& error)
{
   onSendHCICmdErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onSendHCICmdResponse(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< SendHCICmdResponse >& response)
{
   onSendHCICmdResponseCb(response, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onAdapterInitializationError(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< AdapterInitializationError >& error)
{
   onAdapterInitializationErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onAdapterInitializationSignal(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< AdapterInitializationSignal >& signal)
{
   onAdapterInitializationSignalCb(signal, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onDeviceCreatedError(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< DeviceCreatedError >& error)
{
   onDeviceCreatedErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onDeviceCreatedSignal(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< DeviceCreatedSignal >& signal)
{
   onDeviceCreatedSignalCb(signal, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onDeviceRemovedError(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< DeviceRemovedError >& error)
{
   onDeviceRemovedErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onDeviceRemovedSignal(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< DeviceRemovedSignal >& signal)
{
   onDeviceRemovedSignalCb(signal, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onBtStackLogError(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< BtStackLogError >& error)
{
   onBtStackLogErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onBtStackLogSignal(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< BtStackLogSignal >& signal)
{
   onBtStackLogSignalCb(signal, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onHCIEventError(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< HCIEventError >& error)
{
   onHCIEventErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onHCIEventSignal(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< HCIEventSignal >& signal)
{
   onHCIEventSignalCb(signal, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onDeviceFoundError(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< DeviceFoundError >& error)
{
   onDeviceFoundErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onDeviceFoundSignal(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< DeviceFoundSignal >& signal)
{
   onDeviceFoundSignalCb(signal, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onAddressError(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< AddressError >& error)
{
   onAddressErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onAddressUpdate(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< AddressUpdate >& update)
{
   onAddressUpdateCb(update, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onClassError(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< ClassError >& error)
{
   onClassErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onClassUpdate(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< ClassUpdate >& update)
{
   onClassUpdateCb(update, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onAliasError(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< AliasError >& error)
{
   onAliasErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onAliasUpdate(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< AliasUpdate >& update)
{
   onAliasUpdateCb(update, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onPoweredError(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< PoweredError >& error)
{
   onPoweredErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onPoweredUpdate(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< PoweredUpdate >& update)
{
   onPoweredUpdateCb(update, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onDiscoverableError(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< DiscoverableError >& error)
{
   onDiscoverableErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onDiscoverableUpdate(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< DiscoverableUpdate >& update)
{
   onDiscoverableUpdateCb(update, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onDiscoverableTimeoutError(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< DiscoverableTimeoutError >& error)
{
   onDiscoverableTimeoutErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onDiscoverableTimeoutUpdate(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< DiscoverableTimeoutUpdate >& update)
{
   onDiscoverableTimeoutUpdateCb(update, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onPairableError(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< PairableError >& error)
{
   onPairableErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onPairableUpdate(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< PairableUpdate >& update)
{
   onPairableUpdateCb(update, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onPairableTimeoutError(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< PairableTimeoutError >& error)
{
   onPairableTimeoutErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onPairableTimeoutUpdate(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< PairableTimeoutUpdate >& update)
{
   onPairableTimeoutUpdateCb(update, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onDiscoveringError(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< DiscoveringError >& error)
{
   onDiscoveringErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onDiscoveringUpdate(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< DiscoveringUpdate >& update)
{
   onDiscoveringUpdateCb(update, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onUUIDsError(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< UUIDsError >& error)
{
   onUUIDsErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onUUIDsUpdate(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< UUIDsUpdate >& update)
{
   onUUIDsUpdateCb(update, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onWBSModeError(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< WBSModeError >& error)
{
   onWBSModeErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onWBSModeUpdate(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< WBSModeUpdate >& update)
{
   onWBSModeUpdateCb(update, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onConnectableError(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< ConnectableError >& error)
{
   onConnectableErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onConnectableUpdate(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< ConnectableUpdate >& update)
{
   onConnectableUpdateCb(update, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onHCIModeError(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< HCIModeError >& error)
{
   onHCIModeErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onHCIModeUpdate(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< HCIModeUpdate >& update)
{
   onHCIModeUpdateCb(update, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onVersionError(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< VersionError >& error)
{
   onVersionErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onVersionUpdate(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< VersionUpdate >& update)
{
   onVersionUpdateCb(update, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onTraceLevelError(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< TraceLevelError >& error)
{
   onTraceLevelErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onTraceLevelUpdate(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< TraceLevelUpdate >& update)
{
   onTraceLevelUpdateCb(update, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onTraceCategoryError(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< TraceCategoryError >& error)
{
   onTraceCategoryErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onTraceCategoryUpdate(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< TraceCategoryUpdate >& update)
{
   onTraceCategoryUpdateCb(update, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onAvailableSPPServersError(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< AvailableSPPServersError >& error)
{
   onAvailableSPPServersErrorCb(error, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onAvailableSPPServersUpdate(const ::boost::shared_ptr< Adapter1Proxy >& proxy, const ::boost::shared_ptr< AvailableSPPServersUpdate >& update)
{
   onAvailableSPPServersUpdateCb(update, proxy->getDBusObjectPath(), proxy->getDBusBusName(), convertConnectorOption2BusType(proxy->getConnectorOptions()), proxy->getInterfaceName());
}

void EvolutionGeniviDbusAdapterProxyIf::onAvailableCb(const bool available, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   internalOnAvailable(available, objPath, busName, busType);
}

void EvolutionGeniviDbusAdapterProxyIf::onUnavailableCb(const bool available, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/)
{
   internalOnUnavailable(available, objPath, busName, busType);
}

void EvolutionGeniviDbusAdapterProxyIf::onStartDiscoveryErrorCb(const ::boost::shared_ptr< StartDiscoveryError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback(removeActAndFindCallback(error->getAct()));
   FW_IF_NULL_PTR_RETURN(callback);
   callback->onStartDiscoveryErrorCb(error, objPath, busName, busType, interfaceName);
}

void EvolutionGeniviDbusAdapterProxyIf::onStartDiscoveryResponseCb(const ::boost::shared_ptr< StartDiscoveryResponse >& response, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback(removeActAndFindCallback(response->getAct()));
   FW_IF_NULL_PTR_RETURN(callback);
   callback->onStartDiscoveryResponseCb(response, objPath, busName, busType, interfaceName);
}

void EvolutionGeniviDbusAdapterProxyIf::onStopDiscoveryErrorCb(const ::boost::shared_ptr< StopDiscoveryError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback(removeActAndFindCallback(error->getAct()));
   FW_IF_NULL_PTR_RETURN(callback);
   callback->onStopDiscoveryErrorCb(error, objPath, busName, busType, interfaceName);
}

void EvolutionGeniviDbusAdapterProxyIf::onStopDiscoveryResponseCb(const ::boost::shared_ptr< StopDiscoveryResponse >& response, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback(removeActAndFindCallback(response->getAct()));
   FW_IF_NULL_PTR_RETURN(callback);
   callback->onStopDiscoveryResponseCb(response, objPath, busName, busType, interfaceName);
}

void EvolutionGeniviDbusAdapterProxyIf::onRemoveDeviceErrorCb(const ::boost::shared_ptr< RemoveDeviceError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback(removeActAndFindCallback(error->getAct()));
   FW_IF_NULL_PTR_RETURN(callback);
   callback->onRemoveDeviceErrorCb(error, objPath, busName, busType, interfaceName);
}

void EvolutionGeniviDbusAdapterProxyIf::onRemoveDeviceResponseCb(const ::boost::shared_ptr< RemoveDeviceResponse >& response, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback(removeActAndFindCallback(response->getAct()));
   FW_IF_NULL_PTR_RETURN(callback);
   callback->onRemoveDeviceResponseCb(response, objPath, busName, busType, interfaceName);
}

void EvolutionGeniviDbusAdapterProxyIf::onCreateDeviceErrorCb(const ::boost::shared_ptr< CreateDeviceError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback(removeActAndFindCallback(error->getAct()));
   FW_IF_NULL_PTR_RETURN(callback);
   callback->onCreateDeviceErrorCb(error, objPath, busName, busType, interfaceName);
}

void EvolutionGeniviDbusAdapterProxyIf::onCreateDeviceResponseCb(const ::boost::shared_ptr< CreateDeviceResponse >& response, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback(removeActAndFindCallback(response->getAct()));
   FW_IF_NULL_PTR_RETURN(callback);
   callback->onCreateDeviceResponseCb(response, objPath, busName, busType, interfaceName);
}

void EvolutionGeniviDbusAdapterProxyIf::onSendHCICmdErrorCb(const ::boost::shared_ptr< SendHCICmdError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback(removeActAndFindCallback(error->getAct()));
   FW_IF_NULL_PTR_RETURN(callback);
   callback->onSendHCICmdErrorCb(error, objPath, busName, busType, interfaceName);
}

void EvolutionGeniviDbusAdapterProxyIf::onSendHCICmdResponseCb(const ::boost::shared_ptr< SendHCICmdResponse >& response, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback(removeActAndFindCallback(response->getAct()));
   FW_IF_NULL_PTR_RETURN(callback);
   callback->onSendHCICmdResponseCb(response, objPath, busName, busType, interfaceName);
}

void EvolutionGeniviDbusAdapterProxyIf::onAdapterInitializationErrorCb(const ::boost::shared_ptr< AdapterInitializationError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onAdapterInitializationErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onAdapterInitializationErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onAdapterInitializationSignalCb(const ::boost::shared_ptr< AdapterInitializationSignal >& signal, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(signal->getAct());
   if(0 != callback)
   {
      callback->onAdapterInitializationSignalCb(signal, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onAdapterInitializationSignalCb(signal, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onDeviceCreatedErrorCb(const ::boost::shared_ptr< DeviceCreatedError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onDeviceCreatedErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onDeviceCreatedErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onDeviceCreatedSignalCb(const ::boost::shared_ptr< DeviceCreatedSignal >& signal, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(signal->getAct());
   if(0 != callback)
   {
      callback->onDeviceCreatedSignalCb(signal, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onDeviceCreatedSignalCb(signal, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onDeviceRemovedErrorCb(const ::boost::shared_ptr< DeviceRemovedError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onDeviceRemovedErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onDeviceRemovedErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onDeviceRemovedSignalCb(const ::boost::shared_ptr< DeviceRemovedSignal >& signal, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(signal->getAct());
   if(0 != callback)
   {
      callback->onDeviceRemovedSignalCb(signal, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onDeviceRemovedSignalCb(signal, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onBtStackLogErrorCb(const ::boost::shared_ptr< BtStackLogError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onBtStackLogErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onBtStackLogErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onBtStackLogSignalCb(const ::boost::shared_ptr< BtStackLogSignal >& signal, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(signal->getAct());
   if(0 != callback)
   {
      callback->onBtStackLogSignalCb(signal, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onBtStackLogSignalCb(signal, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onHCIEventErrorCb(const ::boost::shared_ptr< HCIEventError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onHCIEventErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onHCIEventErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onHCIEventSignalCb(const ::boost::shared_ptr< HCIEventSignal >& signal, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(signal->getAct());
   if(0 != callback)
   {
      callback->onHCIEventSignalCb(signal, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onHCIEventSignalCb(signal, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onDeviceFoundErrorCb(const ::boost::shared_ptr< DeviceFoundError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onDeviceFoundErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onDeviceFoundErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onDeviceFoundSignalCb(const ::boost::shared_ptr< DeviceFoundSignal >& signal, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(signal->getAct());
   if(0 != callback)
   {
      callback->onDeviceFoundSignalCb(signal, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onDeviceFoundSignalCb(signal, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onAddressErrorCb(const ::boost::shared_ptr< AddressError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onAddressErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onAddressErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onAddressUpdateCb(const ::boost::shared_ptr< AddressUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(update->getAct());
   if(0 != callback)
   {
      callback->onAddressUpdateCb(update, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onAddressUpdateCb(update, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onClassErrorCb(const ::boost::shared_ptr< ClassError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onClassErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onClassErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onClassUpdateCb(const ::boost::shared_ptr< ClassUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(update->getAct());
   if(0 != callback)
   {
      callback->onClassUpdateCb(update, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onClassUpdateCb(update, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onAliasErrorCb(const ::boost::shared_ptr< AliasError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onAliasErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onAliasErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onAliasUpdateCb(const ::boost::shared_ptr< AliasUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(update->getAct());
   if(0 != callback)
   {
      callback->onAliasUpdateCb(update, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onAliasUpdateCb(update, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onPoweredErrorCb(const ::boost::shared_ptr< PoweredError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onPoweredErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onPoweredErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onPoweredUpdateCb(const ::boost::shared_ptr< PoweredUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(update->getAct());
   if(0 != callback)
   {
      callback->onPoweredUpdateCb(update, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onPoweredUpdateCb(update, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onDiscoverableErrorCb(const ::boost::shared_ptr< DiscoverableError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onDiscoverableErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onDiscoverableErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onDiscoverableUpdateCb(const ::boost::shared_ptr< DiscoverableUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(update->getAct());
   if(0 != callback)
   {
      callback->onDiscoverableUpdateCb(update, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onDiscoverableUpdateCb(update, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onDiscoverableTimeoutErrorCb(const ::boost::shared_ptr< DiscoverableTimeoutError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onDiscoverableTimeoutErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onDiscoverableTimeoutErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onDiscoverableTimeoutUpdateCb(const ::boost::shared_ptr< DiscoverableTimeoutUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(update->getAct());
   if(0 != callback)
   {
      callback->onDiscoverableTimeoutUpdateCb(update, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onDiscoverableTimeoutUpdateCb(update, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onPairableErrorCb(const ::boost::shared_ptr< PairableError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onPairableErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onPairableErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onPairableUpdateCb(const ::boost::shared_ptr< PairableUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(update->getAct());
   if(0 != callback)
   {
      callback->onPairableUpdateCb(update, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onPairableUpdateCb(update, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onPairableTimeoutErrorCb(const ::boost::shared_ptr< PairableTimeoutError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onPairableTimeoutErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onPairableTimeoutErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onPairableTimeoutUpdateCb(const ::boost::shared_ptr< PairableTimeoutUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(update->getAct());
   if(0 != callback)
   {
      callback->onPairableTimeoutUpdateCb(update, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onPairableTimeoutUpdateCb(update, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onDiscoveringErrorCb(const ::boost::shared_ptr< DiscoveringError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onDiscoveringErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onDiscoveringErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onDiscoveringUpdateCb(const ::boost::shared_ptr< DiscoveringUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(update->getAct());
   if(0 != callback)
   {
      callback->onDiscoveringUpdateCb(update, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onDiscoveringUpdateCb(update, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onUUIDsErrorCb(const ::boost::shared_ptr< UUIDsError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onUUIDsErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onUUIDsErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onUUIDsUpdateCb(const ::boost::shared_ptr< UUIDsUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(update->getAct());
   if(0 != callback)
   {
      callback->onUUIDsUpdateCb(update, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onUUIDsUpdateCb(update, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onWBSModeErrorCb(const ::boost::shared_ptr< WBSModeError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onWBSModeErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onWBSModeErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onWBSModeUpdateCb(const ::boost::shared_ptr< WBSModeUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(update->getAct());
   if(0 != callback)
   {
      callback->onWBSModeUpdateCb(update, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onWBSModeUpdateCb(update, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onConnectableErrorCb(const ::boost::shared_ptr< ConnectableError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onConnectableErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onConnectableErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onConnectableUpdateCb(const ::boost::shared_ptr< ConnectableUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(update->getAct());
   if(0 != callback)
   {
      callback->onConnectableUpdateCb(update, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onConnectableUpdateCb(update, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onHCIModeErrorCb(const ::boost::shared_ptr< HCIModeError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onHCIModeErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onHCIModeErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onHCIModeUpdateCb(const ::boost::shared_ptr< HCIModeUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(update->getAct());
   if(0 != callback)
   {
      callback->onHCIModeUpdateCb(update, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onHCIModeUpdateCb(update, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onVersionErrorCb(const ::boost::shared_ptr< VersionError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onVersionErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onVersionErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onVersionUpdateCb(const ::boost::shared_ptr< VersionUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(update->getAct());
   if(0 != callback)
   {
      callback->onVersionUpdateCb(update, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onVersionUpdateCb(update, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onTraceLevelErrorCb(const ::boost::shared_ptr< TraceLevelError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onTraceLevelErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onTraceLevelErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onTraceLevelUpdateCb(const ::boost::shared_ptr< TraceLevelUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(update->getAct());
   if(0 != callback)
   {
      callback->onTraceLevelUpdateCb(update, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onTraceLevelUpdateCb(update, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onTraceCategoryErrorCb(const ::boost::shared_ptr< TraceCategoryError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onTraceCategoryErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onTraceCategoryErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onTraceCategoryUpdateCb(const ::boost::shared_ptr< TraceCategoryUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(update->getAct());
   if(0 != callback)
   {
      callback->onTraceCategoryUpdateCb(update, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onTraceCategoryUpdateCb(update, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onAvailableSPPServersErrorCb(const ::boost::shared_ptr< AvailableSPPServersError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(error->getAct());
   if(0 != callback)
   {
      callback->onAvailableSPPServersErrorCb(error, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onAvailableSPPServersErrorCb(error, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::onAvailableSPPServersUpdateCb(const ::boost::shared_ptr< AvailableSPPServersUpdate >& update, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const DbusBusType busType /*= BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   IEvolutionGeniviDbusAdapterCallbackIf* callback;
   callback = removeActAndFindCallback(update->getAct());
   if(0 != callback)
   {
      callback->onAvailableSPPServersUpdateCb(update, objPath, busName, busType, interfaceName);
   }
   else
   {
      for(size_t i = 0; i < getCallbackListSize(); i++)
      {
         callback = getCallbackEntry(i);
         if(0 != callback)
         {
            callback->onAvailableSPPServersUpdateCb(update, objPath, busName, busType, interfaceName);
         }
      }
   }
}

void EvolutionGeniviDbusAdapterProxyIf::internalCreateProxy(const unsigned int callbackId, const ::std::string& objPath, const ::std::string& busName, const DbusBusType busType)
{
   evolution::EvolutionGeniviDbusParser parser;
   const ::std::string& port(parser.getPort(evolution::IF_ADAPTER));
   const ::std::string& path(parser.getObjectPath(evolution::IF_ADAPTER));
   const ::std::string& name(parser.getBusName(evolution::IF_ADAPTER));
   const DbusBusType type(parser.getBusType(evolution::IF_ADAPTER));

   // check if proxy is available
   if(false == isProxyAvailable()) // independent of path, name, type
   {
      ETG_TRACE_USR3((" internalCreateProxy(): Adapter: callbackId=%u busType=%d busName=%20s objPath=%s (create)", callbackId, busType, busName.c_str(), objPath.c_str()));

      // create proxy
      if(0 != _testProxyIf)
      {
         createTestProxy(port, path, name, type, *this);
         addTestProxy(); // independent of path, name, type
      }
      else
      {
         ::boost::shared_ptr< Adapter1Proxy > proxy = createProxy(port, path, name, type, *this);
         addProxy(proxy); // independent of path, name, type
      }
   }
   else
   {
      ETG_TRACE_USR3((" internalCreateProxy(): Adapter: callbackId=%u busType=%d busName=%20s objPath=%s (already done)", callbackId, busType, busName.c_str(), objPath.c_str()));

      // proxy creation was already started; either service is available or creation is ongoing
      updateCurrentAvailableStatus(callbackId, getCreationState() /* independent of path, name, type */, path, name, type);
   }
}

void EvolutionGeniviDbusAdapterProxyIf::internalDestroyProxy(const unsigned int callbackId, const ::std::string& objPath, const ::std::string& busName, const DbusBusType busType)
{
   // HINT: destroying runtime proxy is not possible because ASF is storing a proxy "reference" internally; therefore reset() call will not causing destruction of proxy
   // HINT: due to this we explicitly have to call sendDeregisterAll() to force removing of signal registration (DBUS: member=RemoveMatch)
   // HINT: sendDeregisterAll() is automatically called during destruction of proxy
   // HINT: sendDeregisterAll() is not provided by all proxies

   // check if proxy is available
   if(true == isProxyAvailable()) // independent of path, name, type
   {
      ETG_TRACE_USR3((" internalDestroyProxy(): Adapter: callbackId=%u busType=%d busName=%20s objPath=%s (destroy)", callbackId, busType, busName.c_str(), objPath.c_str()));

      // destroy proxy
      if(0 != _testProxyIf)
      {
         evolution::EvolutionGeniviDbusParser parser;
         const ::std::string& path(parser.getObjectPath(evolution::IF_ADAPTER));
         const ::std::string& name(parser.getBusName(evolution::IF_ADAPTER));
         const DbusBusType type(parser.getBusType(evolution::IF_ADAPTER));

         destroyTestProxy(path, name, type);
         removeTestProxy(); // independent of path, name, type
      }
      else
      {
         ::boost::shared_ptr< Adapter1Proxy >* proxyPtr(getProxy()); // independent of path, name, type
         if(0 != proxyPtr)
         {
            ::boost::shared_ptr< Adapter1Proxy >& proxy = *proxyPtr;
            proxy->sendDeregisterAll();
         }
         removeProxy(); // independent of path, name, type
      }
   }
   else
   {
      ETG_TRACE_USR3((" internalDestroyProxy(): Adapter: callbackId=%u busType=%d busName=%20s objPath=%s (already done)", callbackId, busType, busName.c_str(), objPath.c_str()));
   }

   // NOTE: check following: second callback is registered, anything to do?
}

void EvolutionGeniviDbusAdapterProxyIf::internalDestroyAllProxies(void)
{
   // HINT: sendDeregisterAll() is not provided by all proxies

   ETG_TRACE_USR3((" internalDestroyAllProxies(): Adapter"));

   // destroy all proxies
   if(0 != _testProxyIf)
   {
      destroyAllTestProxies();
      removeAllTestProxies();
   }
   else
   {
      for(size_t i = 0; i < getProxyListSize(); i++)
      {
         ::boost::shared_ptr< Adapter1Proxy >* proxyPtr(getProxy(i));
         if(0 != proxyPtr)
         {
            ::boost::shared_ptr< Adapter1Proxy >& proxy = *proxyPtr;
            proxy->sendDeregisterAll();
         }
      }
      removeAllProxies();
   }

   // remove callback information
   removeCallbacks();
}

void EvolutionGeniviDbusAdapterProxyIf::internalDestroyAllRuntimeProxies(void)
{
   // HINT: sendDeregisterAll() is not provided by all proxies

   ETG_TRACE_USR3((" internalDestroyAllRuntimeProxies(): Adapter"));

   // destroy all runtime proxies
   // nothing to do because this is a static proxy

   // do not remove callback information
}

void EvolutionGeniviDbusAdapterProxyIf::internalOnAvailable(const bool available, const ::std::string& objPath, const ::std::string& busName, const DbusBusType busType)
{
   ETG_TRACE_USR3((" internalOnAvailable(): Adapter"));

   // inform all matching callback handler
   setCreationState(available); // independent of path, name, type
   updateAvailableStatusToAll(available, objPath, busName, busType); // independent of path, name, type => update status to all
}

void EvolutionGeniviDbusAdapterProxyIf::internalOnUnavailable(const bool available, const ::std::string& objPath, const ::std::string& busName, const DbusBusType busType)
{
   ETG_TRACE_USR3((" internalOnUnavailable(): Adapter"));

   // inform all matching callback handler
   setCreationState(available); // independent of path, name, type
   updateUnavailableStatusToAll(available, objPath, busName, busType); // independent of path, name, type => update status to all
}

} //evolution
} //ccdbusif
