/**
 * @file IEvolutionGeniviDbusTraceSendRequestIf.h
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Send Request Interface for Trace.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Interface definition for EvolutionGenivi DBUS send request for Trace.
 */

#ifndef _I_EVOLUTION_GENIVI_DBUS_TRACE_SEND_REQUEST_IF_H_
#define _I_EVOLUTION_GENIVI_DBUS_TRACE_SEND_REQUEST_IF_H_

#include <stdbool.h> // <cstdbool> is marked with C++11

#include "asf/core/Types.h"

namespace ccdbusif {

// class forward declarations
class IEvolutionGeniviDbusTraceCallbackIf;

/**
 * Send request interface class for Trace.
 */
class IEvolutionGeniviDbusTraceSendRequestIf
{
public:
   /**
    * Destructor.
    */
   virtual ~IEvolutionGeniviDbusTraceSendRequestIf() {}

   /**
    * Set callback handler.
    *
    * @param[in] callbackIf: callback handler
    * @param[in] enableProxy: flag to enable/disable proxy
    */
   virtual void setCallbackIf(IEvolutionGeniviDbusTraceCallbackIf* callbackIf, bool enableProxy) = 0;

   /**
    * Send DebugTraceConfig Request.
    *
    * @param[in] level: Debug trace output level.
    *                   Bit0 – FATAL: system is unusable
    *                   Bit1 – ERROR: error conditions
    *                   Bit2 – WARN: warning conditions
    *                   Bit3 – INFO: informational
    *                   Bit4 – DEBUG: debug-level messages
    *                   Bit5 – LOG: normal messages
    *                   Bit6 – 0: reserved for future use
    *                   Bit7 – TIME: need time information
    * @param[in] current_time: Current time form 0 to 23H59m59s999ms
    * @param[in] mode: Mode of debug trace.
    *                  0x00 – Disable: do not output log
    *                  0x01 – Auto: to specify log categories by module
    *                  0x02 – Manual: to specify log category by host
    * @param[in] category1: To specify category while in manual mode. Set bit as 0/1 to enable/disable the corresponding trace.
    *                       Categories of Bluetooth application.
    *                       Bit0 – GAP
    *                       Bit1 – CM_DM
    *                       Bit2 – SPP
    *                       Bit3 – HFP
    *                       Bit4 – AVP
    *                       Bit5 – AVP Reserved
    *                       Bit6 – PBDL_HFP
    *                       Bit7 – PBDL_SPP
    *                       Bit8 – PBDL_PBAP
    *                       Bit9 – PBDL_SYNCML
    *                       Bit10 – MSG_HFP
    *                       Bit11 – MSG_MAP
    *                       Bit12 – MSG_SPP
    *                       Bit13 – DUN
    *                       Bit14 – OPPS
    *                       Bit15 – OPPC
    *                       Bit16 – BIPR
    *                       Bit17 – BIPI
    *                       Bit18 – PAN
    *                       Bit19 – DI
    *                       Bit20 – HS
    *                       Bit21 – MISC
    *                       Bit22 - SAP
    *                       Bit23 ~ Bit31 – Reserved
    * @param[in] category2: Categories of Bluetooth stack.
    *                       Bit0 – HCI
    *                       Bit1 – L2CAP
    *                       Bit2 – RFCOMM
    *                       Bit3 – OBEX
    *                       Bit4 – SDP
    *                       Bit5 – GAP
    *                       Bit6 – SPP
    *                       Bit7 – BTMNG
    *                       Bit8 – SEC
    *                       Bit9 – AVRCP
    *                       Bit10 – A2DP
    *                       Bit11 – DUN
    *                       Bit12 – OPP
    *                       Bit13 – HANDSFREE
    *                       Bit14 – BIP
    *                       Bit15 – PBAP
    *                       Bit16 – 3GPP
    *                       Bit17 – SYNCML
    *                       Bit18 – MAP
    *                       Bit19 – BNEP
    *                       Bit20 – PAN
    *                       Bit21 – DI
    *                       Bit22 – A2MP
    *                       Bit23 – LAYER_SPP_TRANS
    *                       Bit24 – BTCOMMON
    *                       Bit25 – SAP
    *                       Bit26 ~ Bit31 – Reserved
    * @param[in] category3: reserved
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendDebugTraceConfigRequest(uint8 level, uint32 current_time, uint8 mode, uint32 category1, uint32 category2, uint32 category3) = 0;

   /**
    * Send ProtocolTraceConfig Request.
    *
    * @param[in] filter: The filter of the trace information. Default is not report any trace information to host.
    *                    Set each bit as 1 to enable the corresponding packet sent to host, 0 to disable it.
    *                    Bit0 – HCI command packet
    *                    Bit1 – ACL packet, AV music data is controlled by bit10
    *                    Bit2 – SCO data packet(TBD)
    *                    Bit3 – HCI event packet
    *                    Bit4 – L2CAP signaling information
    *                    Bit5 – RFCOMM SABM/DISC/UA/DM/UIH(DLCI0)
    *                    Bit6 – RFCOMM UIH(except DLCI0)
    *                    Bit7 – AVDTP Signaling
    *                    Bit8 – AVCTP
    *                    Bit9 – BNEP control packet.
    *                    Bit10 – AV music data in ACL packet.
    *                    Bit11 ~ 15 – Reserved
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendProtocolTraceConfigRequest(uint16 filter) = 0;
};

} //ccdbusif

#endif //_I_EVOLUTION_GENIVI_DBUS_TRACE_SEND_REQUEST_IF_H_
