/**
 * @file IEvolutionGeniviDbusServiceSendRequestIf.h
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Send Request Interface for Service.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Interface definition for EvolutionGenivi DBUS send request for Service.
 */

#ifndef _I_EVOLUTION_GENIVI_DBUS_SERVICE_SEND_REQUEST_IF_H_
#define _I_EVOLUTION_GENIVI_DBUS_SERVICE_SEND_REQUEST_IF_H_

#include <stdbool.h> // <cstdbool> is marked with C++11
#include <string>

#include "asf/core/Types.h"

namespace ccdbusif {

// class forward declarations
class IEvolutionGeniviDbusServiceCallbackIf;

/**
 * Send request interface class for Service.
 */
class IEvolutionGeniviDbusServiceSendRequestIf
{
public:
   /**
    * Destructor.
    */
   virtual ~IEvolutionGeniviDbusServiceSendRequestIf() {}

   /**
    * Set callback handler.
    *
    * @param[in] callbackIf: callback handler
    * @param[in] enableProxy: flag to enable/disable proxy
    */
   virtual void setCallbackIf(IEvolutionGeniviDbusServiceCallbackIf* callbackIf, bool enableProxy) = 0;

   /**
    * Handle signal for created service.
    *
    * @param[in] objPath: object path of created service
    */
   virtual void serviceCreated(const ::std::string& objPath) = 0;

   /**
    * Handle signal for removed service.
    *
    * @param[in] objPath: object path of removed service
    * @param[in] destroyDirectly: flag to trigger direct proxy destruction or via work item request
    */
   virtual void serviceRemoved(const ::std::string& objPath, const bool destroyDirectly = false) = 0;

   /**
    * Send Connect Request.
    *
    * @param[in] objPath: object path of service
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendConnectRequest(const ::std::string& objPath) = 0;

   /**
    * Send Disconnect Request.
    *
    * @param[in] objPath: object path of service
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendDisconnectRequest(const ::std::string& objPath) = 0;

   /**
    * Send RemoteUUID Get.
    *
    * @param[in] objPath: object path of service
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendRemoteUUIDGet(const ::std::string& objPath) = 0;

   /**
    * Send State Get.
    *
    * @param[in] objPath: object path of service
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendStateGet(const ::std::string& objPath) = 0;

   /**
    * Send Device Get.
    *
    * @param[in] objPath: object path of service
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendDeviceGet(const ::std::string& objPath) = 0;

   /**
    * Send Version Get.
    *
    * @param[in] objPath: object path of service
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendVersionGet(const ::std::string& objPath) = 0;
};

} //ccdbusif

#endif //_I_EVOLUTION_GENIVI_DBUS_SERVICE_SEND_REQUEST_IF_H_
