/**
 * @file IEvolutionGeniviDbusSerialSendRequestIf.h
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Send Request Interface for Serial.
 *
 * @copyright (C) 2016 - 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Interface definition for EvolutionGenivi DBUS send request for Serial.
 */

#ifndef _I_EVOLUTION_GENIVI_DBUS_SERIAL_SEND_REQUEST_IF_H_
#define _I_EVOLUTION_GENIVI_DBUS_SERIAL_SEND_REQUEST_IF_H_

#include "CcDbusIfTypes.h"

namespace ccdbusif {

// class forward declarations
class IDestroyAllProxies;
class IEvolutionGeniviDbusSerialCallbackIf;

/**
 * Send request interface class for Serial.
 */
class IEvolutionGeniviDbusSerialSendRequestIf
{
public:
   /**
    * Destructor.
    */
   virtual ~IEvolutionGeniviDbusSerialSendRequestIf() {}

   /**
    * Get destroy interface.
    *
    * @return = destroy interface
    */
   virtual IDestroyAllProxies* getDestroyer(void) = 0;

   /**
    * Set callback handler.
    *
    * @param[in] callbackIf: callback handler
    * @param[in] enableProxy: flag to enable/disable proxy
    * @param[in] callbackId: callback id
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    */
   virtual void setCallbackIf(IEvolutionGeniviDbusSerialCallbackIf* callbackIf, const bool enableProxy, const unsigned int callbackId, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM) = 0;

   /**
    * Create proxy interface.
    *
    * @param[in] callbackId: callback id
    * @param[in] createDirectly: flag to trigger direct proxy creation or via work item request
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    */
   virtual void createProxyIf(const unsigned int callbackId, const bool createDirectly = false, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM) = 0;

   /**
    * Destroy proxy interface.
    *
    * @param[in] callbackId: callback id
    * @param[in] destroyDirectly: flag to trigger direct proxy destruction or via work item request
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    */
   virtual void destroyProxyIf(const unsigned int callbackId, const bool destroyDirectly = false, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM) = 0;

   /**
    * Send SendData Request.
    *
    * @param[in] callbackId: callback id
    * @param[in] objPath: object path of service
    * @param[in] data: SPP data
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendSendDataRequest(const unsigned int callbackId, const ::std::string& objPath, const ::std::vector< uint8 >& data) = 0;

   /**
    * Send FlowCtrlRecv Get.
    *
    * @param[in] callbackId: callback id
    * @param[in] objPath: object path of service
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendFlowCtrlRecvGet(const unsigned int callbackId, const ::std::string& objPath) = 0;

   /**
    * Send FlowCtrlRecv Set.
    *
    * @param[in] callbackId: callback id
    * @param[in] objPath: object path of service
    * @param[in] flowCtrlRecv: flowCtrlRecv command
    */
   virtual void sendFlowCtrlRecvSet(const unsigned int callbackId, const ::std::string& objPath, const ::std::string& flowCtrlRecv) = 0;
};

} //ccdbusif

#endif //_I_EVOLUTION_GENIVI_DBUS_SERIAL_SEND_REQUEST_IF_H_
