/**
 * @file IEvolutionGeniviDbusPanSendRequestIf.h
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Send Request Interface for Pan.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Interface definition for EvolutionGenivi DBUS send request for Pan.
 */

#ifndef _I_EVOLUTION_GENIVI_DBUS_PAN_SEND_REQUEST_IF_H_
#define _I_EVOLUTION_GENIVI_DBUS_PAN_SEND_REQUEST_IF_H_

#include <stdbool.h> // <cstdbool> is marked with C++11
#include <vector>

#include "asf/core/Types.h"

namespace ccdbusif {

// class forward declarations
class IEvolutionGeniviDbusPanCallbackIf;

/**
 * Send request interface class for Pan.
 */
class IEvolutionGeniviDbusPanSendRequestIf
{
public:
   /**
    * Destructor.
    */
   virtual ~IEvolutionGeniviDbusPanSendRequestIf() {}

   /**
    * Set callback handler.
    *
    * @param[in] callbackIf: callback handler
    * @param[in] enableProxy: flag to enable/disable proxy
    */
   virtual void setCallbackIf(IEvolutionGeniviDbusPanCallbackIf* callbackIf, bool enableProxy) = 0;

   /**
    * Send BtApplPanSendDataReq Request.
    * Request to send PAN packet to the remote device via the PAN service
    *
    * @param[in]  dest_addr - destination address
    * @param[in]  src_addr - source address
    * @param[in]  pkt_type - packet type
    * @param[in]  data - Length of filter list
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendBtApplPanSendDataReqRequest(const ::std::vector< uint8 >& dest_addr, const ::std::vector< uint8 >& src_addr, uint16 pkt_type, const ::std::vector< uint8 >& data) = 0;

   /**
    * Send BtApplPanFilterNettypeReq Request.
    * This message is used to control which network protocol types shall not be filtered and transmitted over PAN connection
    *
    * @param[in]  nettype_num - count of network protocol types in list
    * @param[in]  nettype_filter_list - The filter list of network protocol types.
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendBtApplPanFilterNettypeReqRequest(uint16 nettype_num, const ::std::vector< uint8 >& nettype_filter_list) = 0;

   /**
    * Send BtApplPanFilterMuladdrReq Request.
    * This message is used to control which multicast destination addresses shall
    * not be filtered and transmitted over PAN connection.
    *
    * @param[in] addr_num - count of multicast destination addresses in list
    * @param[in] addr_filter_list - The filter list of multicast destination addresses
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendBtApplPanFilterMuladdrReqRequest(uint16 addr_num, const ::std::vector< uint8 >& addr_filter_list) = 0;
};

} //ccdbusif

#endif //_I_EVOLUTION_GENIVI_DBUS_PAN_SEND_REQUEST_IF_H_
