/**
 * @file IEvolutionGeniviDbusObexClientSendRequestIf.h
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Send Request Interface for ObexClient.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Interface definition for EvolutionGenivi DBUS send request for ObexClient.
 */

#ifndef _I_EVOLUTION_GENIVI_DBUS_OBEX_CLIENT_SEND_REQUEST_IF_H_
#define _I_EVOLUTION_GENIVI_DBUS_OBEX_CLIENT_SEND_REQUEST_IF_H_

#include <stdbool.h> // <cstdbool> is marked with C++11
#include <string>

#include "asf/core/Types.h"

namespace ccdbusif {

// class forward declarations
class IEvolutionGeniviDbusObexClientCallbackIf;

/**
 * Send request interface class for ObexClient.
 */
class IEvolutionGeniviDbusObexClientSendRequestIf
{
public:
   /**
    * Destructor.
    */
   virtual ~IEvolutionGeniviDbusObexClientSendRequestIf() {}

   /**
    * Set callback handler.
    *
    * @param[in] callbackIf: callback handler
    * @param[in] enableProxy: flag to enable/disable proxy
    */
   virtual void setCallbackIf(IEvolutionGeniviDbusObexClientCallbackIf* callbackIf, bool enableProxy) = 0;

   /**
    * Send CreateSession Request.
    * Create a new OBEX session for the given remote address
    *
    * Possible errors:
    *  - org.bluez.obex.Error.InvalidArguments
    *  - org.bluez.obex.Error.Failed
    *
    * @param[in] destination: Bluetooth address of the remote device
    * @param[in] target: type of session to be created
    * @param[in] source: local address to be used
    * @param[in] channel: The parameter will be ignored by module, host needn’t set it.
    * @param[in] instance: Remote instance_id to be connected
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendCreateSessionRequest(const ::std::string& destination, const ::std::string& target, const ::std::string& source, const uint8 channel, const int32 instance) = 0;

   virtual act_t sendCreateSessionRequest(const ::std::string& destination, const ::std::string& target, const ::std::string& source, const uint8 channel) = 0;

   /**
    * Send RemoveSession Request.
    * Unregister session and abort pending transfers.
    *
    * Possible errors:
    *  - org.bluez.obex.Error.InvalidArguments
    *  - org.bluez.obex.Error.NotAuthorized
    *
    * @param[in] session: session
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendRemoveSessionRequest(const ::std::string& session) = 0;

   /**
    * Send CancelSession Request.
    * Cancel creation a new OBEX session for the given remote address
    *
    * Possible errors:
    *  - org.bluez.obex.Error.InvalidArguments
    *  - org.bluez.obex.Error.Failed
    *
    * @param[in] destination: Bluetooth address of the remote device
    * @param[in] target: type of session to be created
    * @param[in] source: local address to be used
    * @param[in] channel: The parameter will be ignored by module, host needn’t set it.
    * @param[in] instance: Remote instance_id to be connected
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendCancelSessionRequest(const ::std::string& destination, const ::std::string& target, const ::std::string& source, const uint8 channel, const int32 instance) = 0;

   virtual act_t sendCancelSessionRequest(const ::std::string& destination, const ::std::string& target, const ::std::string& source, const uint8 channel) = 0;
};

} //ccdbusif

#endif //_I_EVOLUTION_GENIVI_DBUS_OBEX_CLIENT_SEND_REQUEST_IF_H_
