/**
 * @file IEvolutionGeniviDbusLEDeviceSendRequestIf.h
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief Interface definition for EvolutionGenivi DBUS callback for LEDevice.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Interface definition for EvolutionGenivi DBUS callback for LEDevice.
 */

#ifndef _I_EVOLUTION_GENIVI_DBUS_LE_DEVICE_SEND_REQUEST_IF_H_
#define _I_EVOLUTION_GENIVI_DBUS_LE_DEVICE_SEND_REQUEST_IF_H_

#include <stdbool.h> // <cstdbool> is marked with C++11
#include <string>

#include "asf/core/Types.h"

namespace ccdbusif {

// class forward declarations
class IEvolutionGeniviDbusLEDeviceCallbackIf;

/**
 * Send request interface class for LEDevice.
 */
class IEvolutionGeniviDbusLEDeviceSendRequestIf
{
public:
   /**
    * Destructor.
    */
   virtual ~IEvolutionGeniviDbusLEDeviceSendRequestIf() {}

   /**
    * Set callback handler.
    *
    * @param[in] callbackIf: callback handler
    * @param[in] enableProxy: flag to enable/disable proxy
    */
   virtual void setCallbackIf(IEvolutionGeniviDbusLEDeviceCallbackIf* callbackIf, bool enableProxy) = 0;

   /**
    * Handle signal for created device.
    *
    * @param[in] objPath: object path of created device
    */
   virtual void deviceCreated(const ::std::string& objPath) = 0;

   /**
    * Handle signal for removed device.
    *
    * @param[in] objPath: object path of removed device
    */
   virtual void deviceRemoved(const ::std::string& objPath) = 0;

   /**
    * Send Disconnect Request.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendDisconnectRequest(const ::std::string& objPath) = 0;

   /**
    * Send AddressType Get.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendAddressTypeGet(const ::std::string& objPath) = 0;

   /**
    * Send Address Get.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendAddressGet(const ::std::string& objPath) = 0;

   /**
    * Send Connected Get.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendConnectedGet(const ::std::string& objPath) = 0;
};

} //ccdbusif

#endif //_I_EVOLUTION_GENIVI_DBUS_LE_DEVICE_SEND_REQUEST_IF_H_
