/**
 * @file IEvolutionGeniviDbusLEAdapterSendRequestIf.h
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief Interface definition for EvolutionGenivi DBUS callback for LEAdapter.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Interface definition for EvolutionGenivi DBUS callback for LEAdapter.
 */

#ifndef _I_EVOLUTION_GENIVI_DBUS_LE_ADAPTER_SEND_REQUEST_IF_H_
#define _I_EVOLUTION_GENIVI_DBUS_LE_ADAPTER_SEND_REQUEST_IF_H_

#include <stdbool.h> // <cstdbool> is marked with C++11
#include <string>
#include <vector>

#include "asf/core/Types.h"

namespace ccdbusif {

// class forward declarations
class IEvolutionGeniviDbusLEAdapterCallbackIf;

/**
 * Send request interface class for LEAdapter.
 */
class IEvolutionGeniviDbusLEAdapterSendRequestIf
{
public:
   /**
    * Destructor.
    */
   virtual ~IEvolutionGeniviDbusLEAdapterSendRequestIf() {}

   /**
    * Set callback handler.
    *
    * @param[in] callbackIf: callback handler
    * @param[in] enableProxy: flag to enable/disable proxy
    */
   virtual void setCallbackIf(IEvolutionGeniviDbusLEAdapterCallbackIf* callbackIf, bool enableProxy) = 0;

   /**
    * Send CreateLeDevice Request.
    *
    * @param[in] address: arguments details
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendCreateLeDeviceRequest(const ::std::string& address) = 0;

   /**
    * Send RemoveLeDevice Request.
    *
    * @param[in] device: arguments details
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendRemoveLeDeviceRequest(const ::std::string& device) = 0;

   /**
    * Send AddressType Get.
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendAddressTypeGet(void) = 0;

   /**
    * Send Address Get.
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendAddressGet(void) = 0;

   /**
    * Send Alias Get.
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendAliasGet(void) = 0;

   /**
    * Send Alias Set.
    *
    * @param[in] alias: arguments details
    */
   virtual void sendAliasSet(const ::std::string& alias) = 0;

   /**
    * Send AdvertisingTimeout Get.
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendAdvertisingTimeoutGet(void) = 0;

   /**
    * Send AdvertisingTimeout Set.
    *
    * @param[in] advertisingTimeout: arguments details
    */
   virtual void sendAdvertisingTimeoutSet(uint8 advertisingTimeout) = 0;

   /**
    * Send DiscoverableMode Get.
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendDiscoverableModeGet(void) = 0;

   /**
    * Send DiscoverableMode Set.
    *
    * @param[in] discoverableMode: arguments details
    */
   virtual void sendDiscoverableModeSet(uint8 discoverableMode) = 0;

   /**
    * Send AdvertisingData Get.
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendAdvertisingDataGet(void) = 0;

   /**
    * Send AdvertisingData Set.
    *
    * @param[in] advertisingData: arguments details
    */
   virtual void sendAdvertisingDataSet(const ::std::vector< uint8 >& advertisingData) = 0;

   /**
    * Send AdvertiseEnable Get.
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendAdvertiseEnableGet(void) = 0;

   /**
    * Send AdvertiseEnable Set.
    *
    * @param[in] advertiseEnable: arguments details
    */
   virtual void sendAdvertiseEnableSet(bool advertiseEnable) = 0;
};

} //ccdbusif

#endif //_I_EVOLUTION_GENIVI_DBUS_LE_ADAPTER_SEND_REQUEST_IF_H_
