/**
 * @file IEvolutionGeniviDbusHfpModemSendRequestIf.h
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Send Request Interface for HfpModem.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Interface definition for EvolutionGenivi DBUS send request for HfpModem.
 */

#ifndef _I_EVOLUTION_GENIVI_DBUS_HFP_MODEM_SEND_REQUEST_IF_H_
#define _I_EVOLUTION_GENIVI_DBUS_HFP_MODEM_SEND_REQUEST_IF_H_

#include <stdbool.h> // <cstdbool> is marked with C++11
#include <string>

#include "asf/core/Types.h"

namespace ccdbusif {

// class forward declarations
class IEvolutionGeniviDbusHfpModemCallbackIf;

/**
 * Send request interface class for HfpModem.
 */
class IEvolutionGeniviDbusHfpModemSendRequestIf
{
public:
   /**
    * Destructor.
    */
   virtual ~IEvolutionGeniviDbusHfpModemSendRequestIf() {}

   /**
    * Set callback handler.
    *
    * @param[in] callbackIf: callback handler
    * @param[in] enableProxy: flag to enable/disable proxy
    */
   virtual void setCallbackIf(IEvolutionGeniviDbusHfpModemCallbackIf* callbackIf, bool enableProxy) = 0;

   /**
    * Handle signal for created modem.
    *
    * @param[in] objPath: object path of created modem
    */
   virtual void modemCreated(const ::std::string& objPath) = 0;

   /**
    * Handle signal for removed modem.
    *
    * @param[in] objPath: object path of removed modem
    * @param[in] destroyDirectly: flag to trigger direct proxy destruction or via work item request
    */
   virtual void modemRemoved(const ::std::string& objPath, const bool destroyDirectly = false) = 0;

   /**
    * Send GetProperties Request.
    *
    * @param[in] objPath: object path of modem
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendGetPropertiesRequest(const ::std::string& objPath) = 0;

   /**
    * Send SetProperty Request.
    *
    * @param[in] objPath: object path of modem
    * @param[in] property: property string to be updated
    * @param[in] value: property value to be updated
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendSetPropertyRequest(const ::std::string& objPath, const ::std::string& property, const bool value) = 0;

   /**
    * Send AcceptSCOConnect Request.
    *
    * @param[in] objPath: object path of modem
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendAcceptSCOConnectRequest(const ::std::string& objPath) = 0;
};

} //ccdbusif

#endif //_I_EVOLUTION_GENIVI_DBUS_HFP_MODEM_SEND_REQUEST_IF_H_
