/**
 * @file IEvolutionGeniviDbusDeviceSendRequestIf.h
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Send Request Interface for Device.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Interface definition for EvolutionGenivi DBUS send request for Device.
 */

#ifndef _I_EVOLUTION_GENIVI_DBUS_DEVICE_SEND_REQUEST_IF_H_
#define _I_EVOLUTION_GENIVI_DBUS_DEVICE_SEND_REQUEST_IF_H_

#include <stdbool.h> // <cstdbool> is marked with C++11
#include <string>

#include "asf/core/Types.h"

namespace ccdbusif {

// class forward declarations
class IEvolutionGeniviDbusDeviceCallbackIf;

/**
 * Send request interface class for Device.
 */
class IEvolutionGeniviDbusDeviceSendRequestIf
{
public:
   /**
    * Destructor.
    */
   virtual ~IEvolutionGeniviDbusDeviceSendRequestIf() {}

   /**
    * Set callback handler.
    *
    * @param[in] callbackIf: callback handler
    * @param[in] enableProxy: flag to enable/disable proxy
    */
   virtual void setCallbackIf(IEvolutionGeniviDbusDeviceCallbackIf* callbackIf, bool enableProxy) = 0;

   /**
    * Handle signal for created device.
    *
    * @param[in] objPath: object path of created device
    */
   virtual void deviceCreated(const ::std::string& objPath) = 0;

   /**
    * Handle signal for removed device.
    *
    * @param[in] objPath: object path of removed device
    * @param[in] destroyDirectly: flag to trigger direct proxy destruction or via work item request
    */
   virtual void deviceRemoved(const ::std::string& objPath, const bool destroyDirectly = false) = 0;

   /**
    * Send Connect Request.
    * This is a generic method to connect any profiles the remote device supports that can be connected to and have been
    * flagged as auto-connectable on our side. If only subset of profiles is already connected it will try to connect currently
    * disconnected ones.
    * If at least one profile was connected successfully this method will indicate success.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendConnectRequest(const ::std::string& objPath) = 0;

   /**
    * Send Disconnect Request.
    * This method gracefully disconnects all connected profiles and then
    * terminates the low-level ACL connection.The ACL connection will be
    * terminated even if some profiles were not disconnected properly e.g. due
    * to misbehaving device. This method can be also used to cancel a preceding
    * Connect call before a reply to it has been received.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendDisconnectRequest(const ::std::string& objPath) = 0;

   /**
    * Send Pair Request.
    * This method will connect to the remote device, initiate pairing and then
    * retrieve all SDP records (or GATT primary services). If the application has
    * registered its own agent, then that specific agent will be used. Otherwise
    * it will use the default agent. Only for applications like a pairing wizard
    * would it make sense to replace the default agent. In almost all other cases
    * the default agent will be sufficient. In the case where there is no registered
    * agent, application specific or default, this method will fail.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendPairRequest(const ::std::string& objPath) = 0;

   /**
    * Send CancelPairing Request.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendCancelPairingRequest(const ::std::string& objPath) = 0;

   /**
    * Send DiscoverServices Request.
    *
    * @param[in] objPath: object path of device
    * @param[in] pattern: search pattern
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendDiscoverServicesRequest(const ::std::string& objPath, const ::std::string& pattern) = 0;

   /**
    * Send CancelDiscovery Request.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendCancelDiscoveryRequest(const ::std::string& objPath) = 0;

   /**
    * Send Address Get.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendAddressGet(const ::std::string& objPath) = 0;

   /**
    * Send Class Get.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendClassGet(const ::std::string& objPath) = 0;

   /**
    * Send Icon Get.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendIconGet(const ::std::string& objPath) = 0;

   /**
    * Send UUIDs Get.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendUUIDsGet(const ::std::string& objPath) = 0;

   /**
    * Send Paired Get.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendPairedGet(const ::std::string& objPath) = 0;

   /**
    * Send Connected Get.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendConnectedGet(const ::std::string& objPath) = 0;

   /**
    * Send Trusted Get.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendTrustedGet(const ::std::string& objPath) = 0;

   /**
    * Send Trusted Set.
    *
    * @param[in] objPath: object path of device
    * @param[in] trusted: flag
    */
   virtual void sendTrustedSet(const ::std::string& objPath, bool trusted) = 0;

   /**
    * Send Blocked Get.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendBlockedGet(const ::std::string& objPath) = 0;

   /**
    * Send Blocked Set.
    *
    * @param[in] objPath: object path of device
    * @param[in] blocked: Set the connections to be blocked or not
    */
   virtual void sendBlockedSet(const ::std::string& objPath, bool blocked) = 0;

   /**
    * Send Alias Get.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendAliasGet(const ::std::string& objPath) = 0;

   /**
    * Send Alias Set.
    *
    * @param[in] objPath: object path of device
    * @param[in] alias: User friendly name of the Bluetooth adapter
    */
   virtual void sendAliasSet(const ::std::string& objPath, const ::std::string& alias) = 0;

   /**
    * Send Adapter Get.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendAdapterGet(const ::std::string& objPath) = 0;

   /**
    * Send Modalias Get.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendModaliasGet(const ::std::string& objPath) = 0;

   /**
    * Send RSSI Get.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendRSSIGet(const ::std::string& objPath) = 0;

   /**
    * Send Vendor Get.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendVendorGet(const ::std::string& objPath) = 0;

   /**
    * Send VendorSource Get.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendVendorSourceGet(const ::std::string& objPath) = 0;

   /**
    * Send Product Get.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendProductGet(const ::std::string& objPath) = 0;

   /**
    * Send MapInfo Get.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendMapInfoGet(const ::std::string& objPath) = 0;

   /**
    * Send ConnectedService Get.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendConnectedServiceGet(const ::std::string& objPath) = 0;

   /**
    * Send Version Get.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendVersionGet(const ::std::string& objPath) = 0;

   /**
    * Send PBAPInfo Get.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendPBAPInfoGet(const ::std::string& objPath) = 0;

   /**
    * Send Mode Get.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendModeGet(const ::std::string& objPath) = 0;

   /**
    * Send Role Get.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendRoleGet(const ::std::string& objPath) = 0;

   /**
    * Send EnableAvpPause Get.
    *
    * @param[in] objPath: object path of device
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendEnableAvpPauseGet(const ::std::string& objPath) = 0;

   /**
    * Send EnableAvpPause Set.
    *
    * @param[in] objPath: object path of device
    * @param[in] enableAvpPause: enable AVP pause
    */
   virtual void sendEnableAvpPauseSet(const ::std::string& objPath, bool enableAvpPause) = 0;
};

} //ccdbusif

#endif //_I_EVOLUTION_GENIVI_DBUS_DEVICE_SEND_REQUEST_IF_H_
