/**
 * @file IEvolutionGeniviDbusDMSendRequestIf.h
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Send Request Interface for DM.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Interface definition for EvolutionGenivi DBUS send request for DM.
 */

#ifndef _I_EVOLUTION_GENIVI_DBUS_DM_SEND_REQUEST_IF_H_
#define _I_EVOLUTION_GENIVI_DBUS_DM_SEND_REQUEST_IF_H_

#include <stdbool.h> // <cstdbool> is marked with C++11
#include <vector>

#include "asf/core/Types.h"

namespace ccdbusif {

// class forward declarations
class IEvolutionGeniviDbusDMCallbackIf;

/**
 * Send request interface class for DM.
 */
class IEvolutionGeniviDbusDMSendRequestIf
{
public:
   /**
    * Destructor.
    */
   virtual ~IEvolutionGeniviDbusDMSendRequestIf() {}

   /**
    * Set callback handler.
    *
    * @param[in] callbackIf: callback handler
    * @param[in] enableProxy: flag to enable/disable proxy
    */
   virtual void setCallbackIf(IEvolutionGeniviDbusDMCallbackIf* callbackIf, bool enableProxy) = 0;

   /**
    * Send BtApplDeviceConnectReq Request.
    *
    * @param[in] address: Remote device address
    * @param[in] function: The function to connect with remote device. Set bit as 1 to connect, 0 to ignore.
    *                      For each functions representation, refer to function and service bit representation table.
    * @param[in] instance: The instance index of connect
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendBtApplDeviceConnectReqRequest(const ::std::vector< uint8 >& address, uint64 function, uint8 instance) = 0;

   /**
    * Send BtApplDeviceDisconnectReq Request.
    *
    * @param[in] function: The function to connect with remote device. Set bit as 1 to connect, 0 to ignore.
    *                      For each functions representation, refer to function and service bit representation table.
    * @param[in] instance: The instance index of connect
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendBtApplDeviceDisconnectReqRequest(uint64 function, uint8 instance) = 0;

   /**
    * Send BtApplCancelConnectReq Request.
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendBtApplCancelConnectReqRequest(void) = 0;

   /**
    * Send BtApplGetHwVersionReq Request.
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendBtApplGetHwVersionReqRequest(void) = 0;

   /**
    * Send BtApplSetConfigurationReq Request.
    *
    * @param[in] instance: instance
    * @param[in] config_id: configuration id
    * @param[in] data: configuration data
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendBtApplSetConfigurationReqRequest(uint8 instance, uint16 config_id, const ::std::vector< uint8 >& data) = 0;

   /**
    * Send BtApplReadConfigurationReq Request.
    *
    * @param[in] instance: instance
    * @param[in] config_id: configuration id
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendBtApplReadConfigurationReqRequest(uint8 instance, uint16 config_id) = 0;

   /**
    * Send BtApplSspDebugModeReq Request.
    *
    * @param[in] mode: SSP debug mode
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendBtApplSspDebugModeReqRequest(uint8 mode) = 0;

   /**
    * Send BtApplTestModeReq Request.
    *
    * @param[in] enable: enable/disable flag
    * @param[in] mode: mode
    * @param[in] role: role
    * @param[in] addr: address
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendBtApplTestModeReqRequest(uint8 enable, uint8 mode, uint8 role, const ::std::vector< uint8 >& addr) = 0;

   /**
    * Send BtApplTestModeLinkQualityReq Request.
    *
    * @param[in] mode: mode
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendBtApplTestModeLinkQualityReqRequest(uint8 mode) = 0;
};

} //ccdbusif

#endif //_I_EVOLUTION_GENIVI_DBUS_DM_SEND_REQUEST_IF_H_
