/**
 * @file IEvolutionGeniviDbusAdapterTestProxyIf.h
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Test Proxy Interface for Adapter.
 *
 * @copyright (C) 2016 - 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Interface definition for EvolutionGenivi DBUS test proxy for Adapter.
 */

#ifndef _I_EVOLUTION_GENIVI_DBUS_ADAPTER_TEST_PROXY_IF_H_
#define _I_EVOLUTION_GENIVI_DBUS_ADAPTER_TEST_PROXY_IF_H_

#include "CcDbusIfTypes.h"
#include <boost/shared_ptr.hpp>

namespace ccdbusif {

// class forward declarations
class IEvolutionGeniviDbusAdapterCallbackIf;

/**
 * Test proxy interface class for Adapter.
 */
class IEvolutionGeniviDbusAdapterTestProxyIf
{
public:
   /**
    * Destructor.
    */
   virtual ~IEvolutionGeniviDbusAdapterTestProxyIf() {}

   /**
    * Create proxy.
    *
    * @param[in] callback: callback handler
    * @param[in] port: port
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    */
   virtual void createProxy(IEvolutionGeniviDbusAdapterCallbackIf& callback, const ::std::string& port, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM) = 0;

   /**
    * Destroy proxy.
    *
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    */
   virtual void destroyProxy(const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM) = 0;

   /**
    * Destroy all proxies.
    */
   virtual void destroyAllProxies(void) = 0;

   /**
    * Send StartDiscovery Request.
    *
    * @param[in] callback: callback handler
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendStartDiscoveryRequest(IEvolutionGeniviDbusAdapterCallbackIf& callback) = 0;

   /**
    * Send StopDiscovery Request.
    *
    * @param[in] callback: callback handler
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendStopDiscoveryRequest(IEvolutionGeniviDbusAdapterCallbackIf& callback) = 0;

   /**
    * Send RemoveDevice Request.
    *
    * @param[in] callback: callback handler
    * @param[in] device: device object path
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendRemoveDeviceRequest(IEvolutionGeniviDbusAdapterCallbackIf& callback, const ::std::string& device) = 0;

   /**
    * Send CreateDevice Request.
    *
    * @param[in] callback: callback handler
    * @param[in] address: device address
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendCreateDeviceRequest(IEvolutionGeniviDbusAdapterCallbackIf& callback, const ::std::string& address) = 0;

   /**
    * Send SendHCICmd Request.
    *
    * @param[in] callback: callback handler
    * @param[in] data: HCI command data
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendSendHCICmdRequest(IEvolutionGeniviDbusAdapterCallbackIf& callback, const ::std::vector< uint8 >& data) = 0;

   /**
    * Send Address Get.
    *
    * @param[in] callback: callback handler
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendAddressGet(IEvolutionGeniviDbusAdapterCallbackIf& callback) = 0;

   /**
    * Send Class Get.
    *
    * @param[in] callback: callback handler
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendClassGet(IEvolutionGeniviDbusAdapterCallbackIf& callback) = 0;

   /**
    * Send Alias Get.
    *
    * @param[in] callback: callback handler
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendAliasGet(IEvolutionGeniviDbusAdapterCallbackIf& callback) = 0;

   /**
    * Send Alias Set.
    *
    * @param[in] callback: callback handler
    * @param[in] alias: alias
    */
   virtual void sendAliasSet(IEvolutionGeniviDbusAdapterCallbackIf& callback, const ::std::string& alias) = 0;

   /**
    * Send Powered Get.
    *
    * @param[in] callback: callback handler
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendPoweredGet(IEvolutionGeniviDbusAdapterCallbackIf& callback) = 0;

   /**
    * Send Powered Set.
    *
    * @param[in] callback: callback handler
    * @param[in] powered: powered on/off flag
    */
   virtual void sendPoweredSet(IEvolutionGeniviDbusAdapterCallbackIf& callback, bool powered) = 0;

   /**
    * Send Discoverable Get.
    *
    * @param[in] callback: callback handler
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendDiscoverableGet(IEvolutionGeniviDbusAdapterCallbackIf& callback) = 0;

   /**
    * Send Discoverable Set.
    *
    * @param[in] callback: callback handler
    * @param[in] discoverable: discoverable on/off flag
    */
   virtual void sendDiscoverableSet(IEvolutionGeniviDbusAdapterCallbackIf& callback, bool discoverable) = 0;

   /**
    * Send DiscoverableTimeout Get.
    *
    * @param[in] callback: callback handler
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendDiscoverableTimeoutGet(IEvolutionGeniviDbusAdapterCallbackIf& callback) = 0;

   /**
    * Send DiscoverableTimeout Set.
    *
    * @param[in] callback: callback handler
    * @param[in] discoverableTimeout: discoverable timeout
    */
   virtual void sendDiscoverableTimeoutSet(IEvolutionGeniviDbusAdapterCallbackIf& callback, uint32 discoverableTimeout) = 0;

   /**
    * Send Pairable Get.
    *
    * @param[in] callback: callback handler
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendPairableGet(IEvolutionGeniviDbusAdapterCallbackIf& callback) = 0;

   /**
    * Send Pairable Set.
    *
    * @param[in] callback: callback handler
    * @param[in] pairable: pairable on/off flag
    */
   virtual void sendPairableSet(IEvolutionGeniviDbusAdapterCallbackIf& callback, bool pairable) = 0;

   /**
    * Send PairableTimeout Get.
    *
    * @param[in] callback: callback handler
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendPairableTimeoutGet(IEvolutionGeniviDbusAdapterCallbackIf& callback) = 0;

   /**
    * Send PairableTimeout Set.
    *
    * @param[in] callback: callback handler
    * @param[in] pairableTimeout: pairable timeout
    */
   virtual void sendPairableTimeoutSet(IEvolutionGeniviDbusAdapterCallbackIf& callback, uint32 pairableTimeout) = 0;

   /**
    * Send Discovering Get.
    *
    * @param[in] callback: callback handler
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendDiscoveringGet(IEvolutionGeniviDbusAdapterCallbackIf& callback) = 0;

   /**
    * Send UUIDs Get.
    *
    * @param[in] callback: callback handler
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendUUIDsGet(IEvolutionGeniviDbusAdapterCallbackIf& callback) = 0;

   /**
    * Send UUIDs Set.
    *
    * @param[in] callback: callback handler
    * @param[in] uUIDs: UUIDs
    */
   virtual void sendUUIDsSet(IEvolutionGeniviDbusAdapterCallbackIf& callback, const ::std::vector< ::std::string >& uUIDs) = 0;

   /**
    * Send WBSMode Get.
    *
    * @param[in] callback: callback handler
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendWBSModeGet(IEvolutionGeniviDbusAdapterCallbackIf& callback) = 0;

   /**
    * Send WBSMode Set.
    *
    * @param[in] callback: callback handler
    * @param[in] wBSMode: WBS mode
    */
   virtual void sendWBSModeSet(IEvolutionGeniviDbusAdapterCallbackIf& callback, bool wBSMode) = 0;

   /**
    * Send Connectable Get.
    *
    * @param[in] callback: callback handler
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendConnectableGet(IEvolutionGeniviDbusAdapterCallbackIf& callback) = 0;

   /**
    * Send Connectable Set.
    *
    * @param[in] callback: callback handler
    * @param[in] connectable: connectable on/off flag
    */
   virtual void sendConnectableSet(IEvolutionGeniviDbusAdapterCallbackIf& callback, bool connectable) = 0;

   /**
    * Send HCIMode Get.
    *
    * @param[in] callback: callback handler
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendHCIModeGet(IEvolutionGeniviDbusAdapterCallbackIf& callback) = 0;

   /**
    * Send HCIMode Set.
    *
    * @param[in] callback: callback handler
    * @param[in] hCIMode: HCI mode
    */
   virtual void sendHCIModeSet(IEvolutionGeniviDbusAdapterCallbackIf& callback, bool hCIMode) = 0;

   /**
    * Send Version Get.
    *
    * @param[in] callback: callback handler
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendVersionGet(IEvolutionGeniviDbusAdapterCallbackIf& callback) = 0;

   /**
    * Send TraceLevel Get.
    *
    * @param[in] callback: callback handler
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendTraceLevelGet(IEvolutionGeniviDbusAdapterCallbackIf& callback) = 0;

   /**
    * Send TraceLevel Set.
    *
    * @param[in] callback: callback handler
    * @param[in] traceLevel: trace level setting
    */
   virtual void sendTraceLevelSet(IEvolutionGeniviDbusAdapterCallbackIf& callback, const ::std::vector< ::std::string >& traceLevel) = 0;

   /**
    * Send TraceCategory Get.
    *
    * @param[in] callback: callback handler
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendTraceCategoryGet(IEvolutionGeniviDbusAdapterCallbackIf& callback) = 0;

   /**
    * Send TraceCategory Set.
    *
    * @param[in] callback: callback handler
    * @param[in] traceCategory: trace category setting
    */
   virtual void sendTraceCategorySet(IEvolutionGeniviDbusAdapterCallbackIf& callback, const ::std::vector< ::std::string >& traceCategory) = 0;

   /**
    * Send AvailableSPPServers Get.
    *
    * @param[in] callback: callback handler
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendAvailableSPPServersGet(IEvolutionGeniviDbusAdapterCallbackIf& callback) = 0;

   /**
    * Send AvailableSPPServers Set.
    *
    * @param[in] callback: callback handler
    * @param[in] availableSPPServers: available SPP server setting
    */
   virtual void sendAvailableSPPServersSet(IEvolutionGeniviDbusAdapterCallbackIf& callback, const ::std::vector< ::std::string >& availableSPPServers) = 0;
};

} //ccdbusif

#endif //_I_EVOLUTION_GENIVI_DBUS_ADAPTER_TEST_PROXY_IF_H_
