/**
 * @file EvolutionGeniviDbusParser.h
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EvolutionGenivi DBUS Property Parser.
 *
 * @copyright (C) 2016 - 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details EvolutionGenivi DBUS Property Parser.
 */

#ifndef _EVOLUTION_GENIVI_DBUS_PROPERTY_PARSER_H_
#define _EVOLUTION_GENIVI_DBUS_PROPERTY_PARSER_H_

#include "EvolutionGeniviDbusTypes.h"
#include "asf/dbus/DBusVariant.h"

namespace ccdbusif {
namespace evolution {

/**
 * DBUS property parser class for EvolutionGenivi.
 */
class EvolutionGeniviDbusParser
{
public:
   EvolutionGeniviDbusParser();
   virtual ~EvolutionGeniviDbusParser();

   /**
    * Parse function.
    *
    * @param[out] outProperties: list with parsed properties
    * @param[in] matchingInterfaces: interfaces matching the given properties
    * @param[in] inProperties: map with properties (mutable)
    * @param[in] busType: bus type
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] interfaceName: interface name
    */
   void parseProperties(::std::vector<DbusVariantProperty>& outProperties, const ::std::vector< int >& matchingInterfaces, ::std::map< ::std::string, ::asf::dbus::DBusVariant >& inProperties,
         const DbusBusType busType, const ::std::string& busName, const ::std::string& objPath, const ::std::string& interfaceName) const;

   /**
    * Parse function.
    *
    * @param[in] data: introspection data
    * @param[in] busType: bus type
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] interfaceName: interface name
    */
   void parseIntrospection(const ::std::string& data,
         const DbusBusType busType, const ::std::string& busName, const ::std::string& objPath, const ::std::string& interfaceName) const;

   /**
    * Parse function.
    *
    * @param[in] addedObjectPath: added object path
    * @param[in] inInterfaces: map with interfaces (mutable)
    * @param[in] busType: bus type
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] interfaceName: interface name
    */
   void parseInterfacesAdded(const ::std::string& addedObjectPath, ::std::map< ::std::string, ::std::map< ::std::string, ::asf::dbus::DBusVariant > >& inInterfaces,
         const DbusBusType busType, const ::std::string& busName, const ::std::string& objPath, const ::std::string& interfaceName) const;

   /**
    * Parse function.
    *
    * @param[in] removedObjectPath: removed object path
    * @param[in] inInterfaces: vector with interfaces
    * @param[in] busType: bus type
    * @param[in] busName: bus name
    * @param[in] objPath: object path
    * @param[in] interfaceName: interface name
    */
   void parseInterfacesRemoved(const ::std::string& removedObjectPath, const ::std::vector< ::std::string >& inInterfaces,
         const DbusBusType busType, const ::std::string& busName, const ::std::string& objPath, const ::std::string& interfaceName) const;

   /**
    * Parse function.
    *
    * @param[out] outPropertyList: parsed property list
    * @param[in] inProperties: map with properties (mutable)
    * @param[in] propertyName: property to be found/parsed
    * @param[in] interface: related interface
    *
    * @return = true: property found,
    * @return = false: property not found
    */
   bool findAndParseProperty(::std::vector<DbusVariantProperty>& outPropertyList, ::std::map< ::std::string, ::asf::dbus::DBusVariant >& inProperties, const ::std::string& propertyName, const int interface) const;

   /**
    * Parse function.
    *
    * @param[out] outPropertyList: parsed property list
    * @param[in] propertyValue: received property value (mutable)
    * @param[in] propertyName: received property name
    * @param[in] interface: related interface
    */
   void parseProperty(::std::vector<DbusVariantProperty>& outPropertyList, ::asf::dbus::DBusVariant& propertyValue, const ::std::string& propertyName, const int interface) const;

   /**
    * Adds an item to dictionary.
    *
    * @param[out] dictionary: dictionary
    * @param[in] key: key
    * @param[in] value: value
    */
   void addDictionaryItem(::std::map< ::std::string, ::asf::dbus::DBusVariant >& dictionary, const ::std::string& key, const DbusVariant& value) const;

   /**
    * Adds variant value to parameter.
    *
    * @param[out] item: parameter
    * @param[in] value: value
    */
   void addVariantValue(::asf::dbus::DBusVariant& item, const DbusVariant& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: string value
    *
    * @return = service state
    */
   ServiceState convertString2ServiceState(const ::std::string& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: string value
    *
    * @return = property value
    */
   AdapterProperty getAdapterProperty2Enum(const ::std::string& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: property value
    *
    * @return = equivalent string name
    */
   const ::std::string& getAdapterProperty2String(const AdapterProperty value) const;

   /**
    * Convert function.
    *
    * @param[in] value: string value
    *
    * @return = property value
    */
   AgentManagerCapabilitySuppValues getAgentManagerCapability2Enum(const ::std::string& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: property value
    *
    * @return = equivalent string name
    */
   const ::std::string& getAgentManagerCapability2String(const AgentManagerCapabilitySuppValues value) const;

   /**
    * Convert function.
    *
    * @param[in] value: string value
    *
    * @return = property value
    */
   DeviceProperty getDeviceProperty2Enum(const ::std::string& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: property value
    *
    * @return = equivalent string name
    */
   const ::std::string& getDeviceProperty2String(const DeviceProperty value) const;

   /**
    * Convert function.
    *
    * @param[in] value: string value
    *
    * @return = property value
    */
   ServiceProperty getServiceProperty2Enum(const ::std::string& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: property value
    *
    * @return = equivalent string name
    */
   const ::std::string& getServiceProperty2String(const ServiceProperty value) const;

   /**
    * Convert function.
    *
    * @param[in] value: string value
    *
    * @return = property value
    */
   HfpHandsfreeProperty getHfpHandsfreeProperty2Enum(const ::std::string& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: property value
    *
    * @return = equivalent string name
    */
   const ::std::string& getHfpHandsfreeProperty2String(const HfpHandsfreeProperty value) const;

   /**
    * Getter for mapping list.
    *
    * @return = mapping list
    */
   const ::std::map< std::string, int >& getHfpHandsfreeFeatures2EnumList(void) const;
   const ::std::map< std::string, int >* getHfpHandsfreeFeatures2EnumListPtr(void) const;

   /**
    * Getter for mapping list.
    *
    * @return = mapping list
    */
   const ::std::map< int, const std::string* >& getHfpHandsfreeFeatures2StringList(void) const;
   const ::std::map< int, const std::string* >* getHfpHandsfreeFeatures2StringListPtr(void) const;

   /**
    * Convert function.
    *
    * @param[in] value: string value
    *
    * @return = property value
    */
   HfpHandsfreeFeatures getHfpHandsfreeFeatures2Enum(const ::std::string& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: property value
    *
    * @return = equivalent string name
    */
   const ::std::string& getHfpHandsfreeFeatures2String(const HfpHandsfreeFeatures value) const;

   /**
    * Convert function.
    *
    * @param[in] value: string value
    *
    * @return = property value
    */
   HfpHandsfreeChldFeatureSuppValues getHfpHandsfreeChldFeature2Enum(const ::std::string& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: property value
    *
    * @return = equivalent string name
    */
   const ::std::string& getHfpHandsfreeChldFeature2String(const HfpHandsfreeChldFeatureSuppValues value) const;

   /**
    * Convert function.
    *
    * @param[in] value: string value
    *
    * @return = property value
    */
   HfpCallVolumeProperty getHfpCallVolumeProperty2Enum(const ::std::string& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: property value
    *
    * @return = equivalent string name
    */
   const ::std::string& getHfpCallVolumeProperty2String(const HfpCallVolumeProperty value) const;

   /**
    * Convert function.
    *
    * @param[in] value: string value
    *
    * @return = property value
    */
   HfpModemProperty getHfpModemProperty2Enum(const ::std::string& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: property value
    *
    * @return = equivalent string name
    */
   const ::std::string& getHfpModemProperty2String(const HfpModemProperty value) const;

   /**
    * Convert function.
    *
    * @param[in] value: string value
    *
    * @return = property value
    */
   HfpModemInterfaces getHfpModemInterface2Enum(const ::std::string& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: property value
    *
    * @return = equivalent string name
    */
   const ::std::string& getHfpModemInterface2String(const HfpModemInterfaces value) const;

   /**
    * Convert function.
    *
    * @param[in] value: string value
    *
    * @return = property value
    */
   HfpModemTypeSuppValues getHfpModemType2Enum(const ::std::string& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: property value
    *
    * @return = equivalent string name
    */
   const ::std::string& getHfpModemType2String(const HfpModemTypeSuppValues value) const;

   /**
    * Convert function.
    *
    * @param[in] value: string value
    *
    * @return = property value
    */
   HfpNetworkOperatorProperty getHfpNetworkOperatorProperty2Enum(const ::std::string& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: property value
    *
    * @return = equivalent string name
    */
   const ::std::string& getHfpNetworkOperatorProperty2String(const HfpNetworkOperatorProperty value) const;

   /**
    * Convert function.
    *
    * @param[in] value: string value
    *
    * @return = property value
    */
   HfpNetworkRegistrationProperty getHfpNetworkRegistrationProperty2Enum(const ::std::string& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: property value
    *
    * @return = equivalent string name
    */
   const ::std::string& getHfpNetworkRegistrationProperty2String(const HfpNetworkRegistrationProperty value) const;

   /**
    * Convert function.
    *
    * @param[in] value: string value
    *
    * @return = property value
    */
   HfpSiriProperty getHfpSiriProperty2Enum(const ::std::string& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: property value
    *
    * @return = equivalent string name
    */
   const ::std::string& getHfpSiriProperty2String(const HfpSiriProperty value) const;

   /**
    * Convert function.
    *
    * @param[in] value: string value
    *
    * @return = property value
    */
   HfpVoiceCallManagerProperty getHfpVoiceCallManagerProperty2Enum(const ::std::string& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: property value
    *
    * @return = equivalent string name
    */
   const ::std::string& getHfpVoiceCallManagerProperty2String(const HfpVoiceCallManagerProperty value) const;

   /**
    * Convert function.
    *
    * @param[in] value: string value
    *
    * @return = property value
    */
   HfpVoiceCallProperty getHfpVoiceCallProperty2Enum(const ::std::string& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: property value
    *
    * @return = equivalent string name
    */
   const ::std::string& getHfpVoiceCallProperty2String(const HfpVoiceCallProperty value) const;

   /**
    * Convert function.
    *
    * @param[in] value: string value
    *
    * @return = property value
    */
   ObexClientArgsSuppValues getObexClientType2Enum(const ::std::string& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: property value
    *
    * @return = equivalent string name
    */
   const ::std::string& getObexClientType2String(const ObexClientArgsSuppValues value) const;

   /**
    * Convert function.
    *
    * @param[in] value: string value
    *
    * @return = property value
    */
   ObexSessionProperty getObexSessionProperty2Enum(const ::std::string& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: property value
    *
    * @return = equivalent string name
    */
   const ::std::string& getObexSessionProperty2String(const ObexSessionProperty value) const;

   /**
    * Convert function.
    *
    * @param[in] value: string value
    *
    * @return = property value
    */
   ObexTransferProperty getObexTransferProperty2Enum(const ::std::string& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: property value
    *
    * @return = equivalent string name
    */
   const ::std::string& getObexTransferProperty2String(const ObexTransferProperty value) const;

   /**
    * Convert function.
    *
    * @param[in] value: string value
    *
    * @return = property value
    */
   SerialProperty getSerialProperty2Enum(const ::std::string& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: property value
    *
    * @return = equivalent string name
    */
   const ::std::string& getSerialProperty2String(const SerialProperty value) const;

   /**
    * Convert function.
    *
    * @param[in] value: string value
    *
    * @return = property value
    */
   LEAdapterProperty getLEAdapterProperty2Enum(const ::std::string& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: property value
    *
    * @return = equivalent string name
    */
   const ::std::string& getLEAdapterProperty2String(const LEAdapterProperty value) const;

   /**
    * Convert function.
    *
    * @param[in] value: string value
    *
    * @return = property value
    */
   LEDeviceProperty getLEDeviceProperty2Enum(const ::std::string& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: property value
    *
    * @return = equivalent string name
    */
   const ::std::string& getLEDeviceProperty2String(const LEDeviceProperty value) const;

   /**
    * Convert function.
    *
    * @param[in] value: string value
    *
    * @return = property value
    */
   LESeamlessPairingProperty getLESeamlessPairingProperty2Enum(const ::std::string& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: property value
    *
    * @return = equivalent string name
    */
   const ::std::string& getLESeamlessPairingProperty2String(const LESeamlessPairingProperty value) const;

   /**
    * Convert function.
    *
    * @param[in] value: string value
    *
    * @return = UUID value
    */
   UUID getUuid2Enum(const ::std::string& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: UUID value
    *
    * @return = equivalent string name
    */
   const ::std::string& getUuid2String(const UUID value) const;

   /**
    * Convert function (for tracing).
    *
    * @param[in] value: UUID value
    *
    * @return = equivalent string name
    */
   const ::std::string& getUuid2StringTrace(const UUID value) const;

   /**
    * Checks if given UUID is valid.
    *
    * @param[in] value: UUID string
    * @param[in] ignoreCaseSensitive: flag to check case sensitive or insensitive
    *
    * @return = true: valid UUID,
    * @return = false: invalid UUID
    */
   bool isValidUuid(const ::std::string& value, const bool ignoreCaseSensitive = true) const;

   /**
    * Checks if given UUID is a short UUID.
    *
    * @param[out] uuid: UUID value (if short UUID)
    * @param[in] value: UUID string
    *
    * @return = true: short UUID,
    * @return = false: no short UUID
    */
   bool isShortUuid(UUID& uuid, const ::std::string& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: string value
    *
    * @return = interface value
    */
   Interface getInterface2Enum(const ::std::string& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: interface value
    *
    * @return = equivalent string name
    */
   const ::std::string& getInterface2String(const Interface value) const;

   /**
    * Convert function (for tracing).
    *
    * @param[in] value: interface value
    *
    * @return = equivalent string name
    */
   const ::std::string& getInterface2StringTrace(const Interface value) const;

   /**
    * Convert function.
    *
    * @param[in] value: interface value
    *
    * @return = DBUS bus type
    */
   DbusBusType getInterface2BusType(const Interface value) const;
   DbusBusType getBusType(const Interface value) const;

   /**
    * Convert function.
    *
    * @param[in] value: interface value
    *
    * @return = bus name
    */
   const ::std::string& getInterface2BusName(const Interface value) const;
   const ::std::string& getBusName(const Interface value) const;

   /**
    * Convert function.
    *
    * @param[in] value: interface value
    *
    * @return = object path
    */
   const ::std::string& getInterface2ObjectPath(const Interface value) const;
   const ::std::string& getObjectPath(const Interface value) const;

   /**
    * Convert function.
    *
    * @return = root object path
    */
   const ::std::string& getRootObjectPath(void) const;

   /**
    * Convert function.
    *
    * @param[in] value: interface value
    *
    * @return = object path
    */
   const ::std::string& getInterface2Port(const Interface value) const;
   const ::std::string& getPort(const Interface value) const;

   /**
    * Convert function.
    *
    * @return = agent bus type
    */
   DbusBusType getAgentBusType(void) const;

   /**
    * Convert function.
    *
    * @return = agent bus name
    */
   const ::std::string& getAgentBusName(void) const;

   /**
    * Convert function.
    *
    * @return = agent object path
    */
   const ::std::string& getAgentObjectPath(void) const;


   /**
    * Convert function.
    *
    * @param[in] value: string value
    *
    * @return = error code value
    */
   ErrorCode getErrorCode2Enum(const ::std::string& value) const;

   /**
    * Convert function.
    *
    * @param[in] value: error code value
    *
    * @return = equivalent string name
    */
   const ::std::string& getErrorCode2String(const ErrorCode value) const;

   /**
    * Convert function (for tracing).
    *
    * @param[in] value: error code value
    *
    * @return = equivalent string name
    */
   const ::std::string& getErrorCode2StringTrace(const ErrorCode value) const;

   /**
    * Convert function.
    *
    * @param[in] errorName: error name (error code)
    * @param[in] errorMessage: error message (sub-error code)
    *
    * @return = error code value
    */
   ErrorCode getErrorCode2Enum(const ::std::string& errorName, const ::std::string& errorMessage) const;

   /**
    * Set traces enabled flag.
    *
    * @param[in] enabled: traces enabled flag
    */
   void setTraces(const bool enabled);

   /**
    * Get traces enabled flag.
    *
    * @return = traces enabled flag
    */
   inline bool getTraces(void) const { return _tracesEnabled; }

private:
   bool _tracesEnabled; /**< traces enabled flag */
};

} //evolution
} //ccdbusif

#endif //_EVOLUTION_GENIVI_DBUS_PROPERTY_PARSER_H_
