/**
 * @file EcNrDbusServiceProxyIf.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief EcNr DBUS Proxy.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details EcNr DBUS Proxy.
 */

#include "EcNrDbusServiceProxyIf.h"
#include "IEcNrDbusServiceCallbackIf.h"
#include "CcDbusIfTypes.h"
#include "ICcDbusIfControllerClient.h"
#include "CcDbusIfCreateProxyWorkItem.h"

using namespace ::org::bosch::ecnr::service;

using namespace ::asf::core;

namespace ccdbusif {

EcNrDbusServiceProxyIf::EcNrDbusServiceProxyIf()
{
   _enableServiceProxy = false;
   // _serviceProxy is set later
   _controllerClient = NULL;
   _callbackIf = NULL;
}

EcNrDbusServiceProxyIf::EcNrDbusServiceProxyIf(ICcDbusIfControllerClient* client)
{
   _enableServiceProxy = false;
   // _serviceProxy is set later
   _controllerClient = client;
   _callbackIf = NULL;
}

EcNrDbusServiceProxyIf::~EcNrDbusServiceProxyIf()
{
   _controllerClient = NULL;
   _callbackIf = NULL;
}

void EcNrDbusServiceProxyIf::createProxy(void)
{
   // HINT: this interface is called within context of thread started in CcDbusIfController.cpp ("CC_DBUS_IF_THD")

   // HINT: without/before proxy creation _serviceProxy.get() returns 0

   if(NULL != _callbackIf)
   {
      if(true == _enableServiceProxy)
      {
         _serviceProxy = ServiceProxy::createProxy("ecnrServicePort", *this);
      }
   }
}

void EcNrDbusServiceProxyIf::destroyProxy(void)
{
   // HINT: this interface is called within context of thread started in CcDbusIfController.cpp ("CC_DBUS_IF_THD")

   // HINT: without/before proxy deletion _serviceProxy.get() returns non-0

   if(NULL != _serviceProxy.get())
   {
      _serviceProxy.reset();
   }
}

void EcNrDbusServiceProxyIf::destroyAllRuntimeProxies(void)
{
   // nothing to do because this is a static proxy
}

void EcNrDbusServiceProxyIf::setCallbackIf(IEcNrDbusServiceCallbackIf* callbackIf, bool enableServiceProxy)
{
   _callbackIf = callbackIf;

   // HINT: act is provided via response->getAct() / error->getAct() in callback message
   // HINT: act shall be given to receiver and can be compared with act value returned by send function

   if(NULL != _callbackIf)
   {
      _enableServiceProxy = enableServiceProxy;
   }

   // callback interface is now available => create proxy now
   if((NULL != _callbackIf) && (NULL != _controllerClient))
   {
      _controllerClient->pushWorkItem(new CcDbusIfCreateProxyWorkItem(this));
   }
}

// ServiceAvailableIF implementation --- start
void EcNrDbusServiceProxyIf::onAvailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   if((NULL != _serviceProxy.get()) && (proxy == _serviceProxy))
   {
      if(NULL != _callbackIf)
      {
         _callbackIf->onServiceProxyAvailable(stateChange.getPreviousState(), stateChange.getCurrentState(), _serviceProxy->getDBusObjectPath());

         // nothing to register
      }
   }
}

void EcNrDbusServiceProxyIf::onUnavailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   if((NULL != _serviceProxy.get()) && (proxy == _serviceProxy))
   {
      if(NULL != _callbackIf)
      {
         _callbackIf->onServiceProxyUnavailable(stateChange.getPreviousState(), stateChange.getCurrentState(), _serviceProxy->getDBusObjectPath());
      }
      // nothing to deregister
   }
}
// ServiceAvailableIF implementation --- end

// HINT: following interfaces are called within context of thread started in CcDbusIfController.cpp ("CC_DBUS_IF_THD")

// IEcNrDbusServiceSendRequestIf implementation --- start
act_t EcNrDbusServiceProxyIf::sendEcnrInitializeRequest(uint8 ecnrAppId, uint32 ecnrConfigurationId)
{
   if((NULL != _callbackIf) && (NULL != _serviceProxy.get()))
   {
      return _serviceProxy->sendEcnrInitializeRequest(*_callbackIf, ecnrAppId, ecnrConfigurationId);
   }
   else
   {
      return DEFAULT_ACT;
   }
}

act_t EcNrDbusServiceProxyIf::sendEcnrDestroyRequest(uint8 ecnrAppId)
{
   if((NULL != _callbackIf) && (NULL != _serviceProxy.get()))
   {
      return _serviceProxy->sendEcnrDestroyRequest(*_callbackIf, ecnrAppId);
   }
   else
   {
      return DEFAULT_ACT;
   }
}

act_t EcNrDbusServiceProxyIf::sendEcnrSetConfigurationRequest(uint8 ecnrAppId, uint32 ecnrConfigurationId)
{
   if((NULL != _callbackIf) && (NULL != _serviceProxy.get()))
   {
      return _serviceProxy->sendEcnrSetConfigurationRequest(*_callbackIf, ecnrAppId, ecnrConfigurationId);
   }
   else
   {
      return DEFAULT_ACT;
   }
}

act_t EcNrDbusServiceProxyIf::sendEcnrStartAudioRequest(uint8 ecnrAppId)
{
   if((NULL != _callbackIf) && (NULL != _serviceProxy.get()))
   {
      return _serviceProxy->sendEcnrStartAudioRequest(*_callbackIf, ecnrAppId);
   }
   else
   {
      return DEFAULT_ACT;
   }
}

act_t EcNrDbusServiceProxyIf::sendEcnrStopAudioRequest(uint8 ecnrAppId)
{
   if((NULL != _callbackIf) && (NULL != _serviceProxy.get()))
   {
      return _serviceProxy->sendEcnrStopAudioRequest(*_callbackIf, ecnrAppId);
   }
   else
   {
      return DEFAULT_ACT;
   }
}

act_t EcNrDbusServiceProxyIf::sendEcnrSetSendMuteSwitchRequest(uint8 ecnrAppId, uint8 ecnrSendMuteSwitch)
{
   if((NULL != _callbackIf) && (NULL != _serviceProxy.get()))
   {
      return _serviceProxy->sendEcnrSetSendMuteSwitchRequest(*_callbackIf, ecnrAppId, ecnrSendMuteSwitch);
   }
   else
   {
      return DEFAULT_ACT;
   }
}

act_t EcNrDbusServiceProxyIf::sendEcnrGetVersionRequest(void)
{
   if((NULL != _callbackIf) && (NULL != _serviceProxy.get()))
   {
      return _serviceProxy->sendEcnrGetVersionRequest(*_callbackIf);
   }
   else
   {
      return DEFAULT_ACT;
   }
}
// IEcNrDbusServiceSendRequestIf implementation --- end

} //ccdbusif
