/**
 * @file EcNrWrapper.h
 *
 * @par SW-Component
 * Wrapper
 *
 * @brief EcNr wrapper.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Wrapper for EcNr.
 */

#ifndef _ECNR_WRAPPER_H_
#define _ECNR_WRAPPER_H_

#include "EcNrTypes.h"

namespace ccdbusif {

// class forward declarations
class ICcDbusIfController;
class EcNrDbusServiceCallbackIf;
class IEcNrDbusServiceSendRequestIf;
class ICcDbusIfControllerClient;
class TimeoutSemaphore;

/**
 * EcNr wrapper class.
 */
class EcNrWrapper
{
protected:
   /**************************************************************************
       Constructor / destructor
    **************************************************************************/
   /**
    * Default constructor.
    */
   EcNrWrapper();

   /**
    * Destructor.
    */
   virtual ~EcNrWrapper();

   /**
    * Copy constructor => must not be used.
    *
    * @param[in] ref: reference to copy
    */
   EcNrWrapper(IN const EcNrWrapper& ref);

   /**
    * Assignment operator => must not be used.
    *
    * @param[in] ref: reference to copy
    *
    * @return = reference
    */
   EcNrWrapper& operator=(IN const EcNrWrapper& ref);

public:
   /**
    * init
    *
    * @param[in] controller: Pointer to ICcDbusIfController
    *
    * @return = ECNR_OK/ECNR_ERROR
    */
   EcnrErrorCode init(IN ICcDbusIfController* controller);

   /**
    * deInit
    *
    * @return = ECNR_OK/ECNR_ERROR
    */
   EcnrErrorCode deInit(void);

   /**
    * initialize
    *
    * @param[in] appId: application id
    * @param[in] configurationId: configuration id
    *
    * @return = If workitem created and pushed successfully then ECNR_OK will be returned, else ECNR_ERROR.
    */
   EcnrErrorCode initialize(IN const EcnrAppId appId, IN const EcnrConfigurationId configurationId);

   /**
    * initialize ResultCB
    *
    * @param[in] result: ECNR_REQ_SUCCESS - response received from EcNr service
    *                    ECNR_REQ_FAILED - error received from EcNr service due to any reason
    *
    * @return = ECNR_ERROR
    */
   virtual EcnrErrorCode initializeResultCB(IN const EcnrRequestResult result) { (void)(result); return ECNR_ERROR; };

   /**
    * destroy
    *
    * @param[in] appId: application id
    *
    * @return = If workitem created and pushed successfully then ECNR_OK will be returned, else ECNR_ERROR.
    */
   EcnrErrorCode destroy(IN const EcnrAppId appId);

   /**
    * destroy ResultCB
    *
    * @param[in] result: ECNR_REQ_SUCCESS - response received from EcNr service
    *                    ECNR_REQ_FAILED - error received from EcNr service due to any reason
    *
    * @return = ECNR_ERROR
    */
   virtual EcnrErrorCode destroyResultCB(IN const EcnrRequestResult result) { (void)(result); return ECNR_ERROR; };

   /**
    * setConfiguration
    *
    * @param[in] appId: application id
    * @param[in] configurationId: configuration id
    *
    * @return = If workitem created and pushed successfully then ECNR_OK will be returned, else ECNR_ERROR.
    */
   EcnrErrorCode setConfiguration(IN const EcnrAppId appId, IN const EcnrConfigurationId configurationId);

   /**
    * setConfiguration ResultCB
    *
    * @param[in] result: ECNR_REQ_SUCCESS - response received from EcNr service
    *                    ECNR_REQ_FAILED - error received from EcNr service due to any reason
    *
    * @return = ECNR_ERROR
    */
   virtual EcnrErrorCode setConfigurationResultCB(IN const EcnrRequestResult result) { (void)(result); return ECNR_ERROR; };

   /**
    * startAudio
    *
    * @param[in] appId: application id
    *
    * @return = If workitem created and pushed successfully then ECNR_OK will be returned, else ECNR_ERROR.
    */
   EcnrErrorCode startAudio(IN const EcnrAppId appId);

   /**
    * startAudio ResultCB
    *
    * @param[in] result: ECNR_REQ_SUCCESS - response received from EcNr service
    *                    ECNR_REQ_FAILED - error received from EcNr service due to any reason
    *
    * @return = ECNR_ERROR
    */
   virtual EcnrErrorCode startAudioResultCB(IN const EcnrRequestResult result) { (void)(result); return ECNR_ERROR; };

   /**
    * stopAudio
    *
    * @param[in] appId: application id
    *
    * @return = If workitem created and pushed successfully then ECNR_OK will be returned, else ECNR_ERROR.
    */
   EcnrErrorCode stopAudio(IN const EcnrAppId appId);

   /**
    * stopAudio ResultCB
    *
    * @param[in] result: ECNR_REQ_SUCCESS - response received from EcNr service
    *                    ECNR_REQ_FAILED - error received from EcNr service due to any reason
    *
    * @return = ECNR_ERROR
    */
   virtual EcnrErrorCode stopAudioResultCB(IN const EcnrRequestResult result) { (void)(result); return ECNR_ERROR; };

   /**
    * setSendMuteSwitch
    *
    * @param[in] appId: application id
    * @param[in] sendMuteSwitch: mute switch value
    *
    * @return = If workitem created and pushed successfully then ECNR_OK will be returned, else ECNR_ERROR.
    */
   EcnrErrorCode setSendMuteSwitch(IN const EcnrAppId appId, IN const EcnrSendMuteSwitch sendMuteSwitch);

   /**
    * setSendMuteSwitch ResultCB
    *
    * @param[in] result: ECNR_REQ_SUCCESS - response received from EcNr service
    *                    ECNR_REQ_FAILED - error received from EcNr service due to any reason
    *
    * @return = ECNR_ERROR
    */
   virtual EcnrErrorCode setSendMuteSwitchResultCB(IN const EcnrRequestResult result) { (void)(result); return ECNR_ERROR; };

   /**
    * getVersion
    *
    * @return = If workitem created and pushed successfully then ECNR_OK will be returned, else ECNR_ERROR.
    */
   EcnrErrorCode getVersion(void);

   /**
    * getVersion ResultCB
    *
    * @param[in] result: ECNR_REQ_SUCCESS - response received from EcNr service
    *                    ECNR_REQ_FAILED - error received from EcNr service due to any reason
    * @param[in] versionNumber: version number
    * @param[in] versionString: version string
    * @param[in] versionComment: version comment
    *
    * @return = ECNR_ERROR
    */
   virtual EcnrErrorCode getVersionResultCB(IN const EcnrRequestResult result, IN const ::std::vector< unsigned int >& versionNumber, IN const ::std::string& versionString, IN const ::std::string& versionComment) { (void)(result); (void)(versionNumber); (void)(versionString); (void)(versionComment); return ECNR_ERROR; };

   // for internal handling
   /**
    * handleServiceAvailability
    *
    * @param[in] available: service available status true/false
    */
   void handleServiceAvailability(IN const bool available);

private:
   enum
   {
      SERVICE_WAITING_TIMEOUT_MS = 5000
   };

   // callback handler pointer
   EcNrDbusServiceCallbackIf* _ecnrServiceCallbackIf;
   IEcNrDbusServiceSendRequestIf* _ecnrServiceSendIf;
   ICcDbusIfControllerClient* _ccDbusIfControllerClient;
   bool _dbusIfAvailable;
   bool _serviceAvailable;
   TimeoutSemaphore* _serviceSemaphore;

   bool waitForServiceAvailable(void);
};

} //ccdbusif

#endif //_ECNR_WRAPPER_H_
