/**
 * @file ConnManDbusTechnologyProxyIf.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief ConnMan DBUS Proxy for Technology.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details ConnMan DBUS proxy for Technology.
 */

#include "ConnManDbusTechnologyProxyIf.h"
#include "IConnManDbusTechnologyCallbackIf.h"
#include "CcDbusIfTypes.h"
#include "ICcDbusIfControllerClient.h"
#include "CcDbusIfCreateProxyWorkItem.h"
#include "CcDbusIfCreateRuntimeProxyWorkItem.h"
#include "CcDbusIfDestroyRuntimeProxyWorkItem.h"
#include "ConnManDbusParser.h"

#include "TraceClasses.h"
#define FW_S_IMPORT_INTERFACE_TRACE
#include "framework_if_if.h"

using namespace ::net::connman::Technology;
using namespace ::asf::core;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CCDBUSIF_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/ConnManDbusTechnologyProxyIf.cpp.trc.h"
#endif
#endif

namespace ccdbusif {

ConnManDbusTechnologyProxyIf::ConnManDbusTechnologyProxyIf()
{
   // _technologyProxy is set later
   _callbackIf = NULL;
}

ConnManDbusTechnologyProxyIf::ConnManDbusTechnologyProxyIf(ICcDbusIfControllerClient* client) : BaseDbusProxyIf(client)
{
   // _technologyProxy is set later
   _callbackIf = NULL;
}

ConnManDbusTechnologyProxyIf::~ConnManDbusTechnologyProxyIf()
{
   _callbackIf = NULL;
}

void ConnManDbusTechnologyProxyIf::createProxy(void)
{
   // these proxies have to be created during runtime
}

void ConnManDbusTechnologyProxyIf::destroyProxy(void)
{
   ETG_TRACE_USR3((" destroyProxy(): Technology"));

   // these proxies have to be destroyed during runtime
   // destroy all proxies now if still available
   _technologyProxy.resetAllProxiesAndClear();
}

void ConnManDbusTechnologyProxyIf::destroyAllRuntimeProxies(void)
{
   ETG_TRACE_USR3((" destroyAllRuntimeProxies(): Technology"));

   for(unsigned int i = 0; i < _technologyProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< TechnologyProxy >& technologyProxy = _technologyProxy[i];
      if(NULL != technologyProxy.get())
      {
         technologyProxy->sendDeregisterAll();
      }
   }

   _technologyProxy.resetAllProxiesAndClear();
}

void ConnManDbusTechnologyProxyIf::createRuntimeProxy(const ::std::string& objPath)
{
   ETG_TRACE_USR3((" createRuntimeProxy(): Technology: objPath=%s", objPath.c_str()));

   if((NULL != _callbackIf) && (true == _enableProxy))
   {
      if(false == _technologyProxy.isInstanceAvailable(objPath))
      {
         ::boost::shared_ptr< TechnologyProxy > technologyProxy = TechnologyProxy::createProxy("evoConnManTechnologyPort", "net.connman", objPath, DBUS_BUS_SYSTEM, *this);
         _technologyProxy.addProxyInstance(objPath, technologyProxy);
      }
   }
}

void ConnManDbusTechnologyProxyIf::destroyRuntimeProxy(const ::std::string& objPath)
{
   ETG_TRACE_USR3((" destroyRuntimeProxy(): Technology: objPath=%s", objPath.c_str()));

   // HINT: destroying runtime proxy is not possible because ASF is storing a proxy "reference" internally; therefore reset() call will not causing destruction of proxy
   // HINT: due to this we explicitly have to call sendDeregisterAll() to force removing of signal registration (DBUS: member=RemoveMatch)
   // HINT: sendDeregisterAll() is automatically called during destruction of proxy

   ::boost::shared_ptr< TechnologyProxy > technologyProxy;
   if(true == _technologyProxy.isProxyAvailable(technologyProxy, objPath))
   {
      technologyProxy->sendDeregisterAll();
   }
   _technologyProxy.removeProxyInstance(objPath);
}

void ConnManDbusTechnologyProxyIf::setCallbackIf(IConnManDbusTechnologyCallbackIf* callbackIf, bool enableProxy)
{
   _callbackIf = callbackIf;

   if(NULL != _callbackIf)
   {
      _enableProxy = enableProxy;
   }

   // callback interface is now available => but proxy will be created later
}

void ConnManDbusTechnologyProxyIf::technologyCreated(const ::std::string& objPath)
{
   // create runtime proxy
   if((NULL != _callbackIf) && (NULL != _controllerClient))
   {
      _controllerClient->pushWorkItem(new CcDbusIfCreateRuntimeProxyWorkItem(objPath, this));
   }
}

void ConnManDbusTechnologyProxyIf::technologyRemoved(const ::std::string& objPath)
{
   // destroy runtime proxy
   if((NULL != _callbackIf) && (NULL != _controllerClient))
   {
      _controllerClient->pushWorkItem(new CcDbusIfDestroyRuntimeProxyWorkItem(objPath, this));
   }
}

// ServiceAvailableIF implementation --- start
void ConnManDbusTechnologyProxyIf::onAvailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3((" onAvailable(): Technology"));

   for(unsigned int i = 0; i < _technologyProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< TechnologyProxy >& technologyProxy = _technologyProxy[i];
      if((NULL != technologyProxy.get()) && (proxy == technologyProxy))
      {
         if(NULL != _callbackIf)
         {
            technologyProxy->sendPropertyChangedRegister(*_callbackIf);

            _callbackIf->onProxyAvailable(stateChange.getPreviousState(), stateChange.getCurrentState(), technologyProxy->getDBusObjectPath());
         }
         break;
      }
   }
}

void ConnManDbusTechnologyProxyIf::onUnavailable(const boost::shared_ptr<Proxy>& proxy, const ServiceStateChange& stateChange)
{
   ETG_TRACE_USR3((" onUnavailable(): Technology"));

   for(unsigned int i = 0; i < _technologyProxy.getNumberOfProxyInstances(); i++)
   {
      ::boost::shared_ptr< TechnologyProxy >& technologyProxy = _technologyProxy[i];
      if((NULL != technologyProxy.get()) && (proxy == technologyProxy))
      {
         technologyProxy->sendDeregisterAll();

         if(NULL != _callbackIf)
         {
            _callbackIf->onProxyUnavailable(stateChange.getPreviousState(), stateChange.getCurrentState(), technologyProxy->getDBusObjectPath());
         }
         break;
      }
   }
}
// ServiceAvailableIF implementation --- end

// IConnManDbusTechnologyProxyIf implementation --- start
act_t ConnManDbusTechnologyProxyIf::sendGetPropertiesRequest(const ::std::string& objPath)
{
   ::boost::shared_ptr< TechnologyProxy > proxy;
   if(true == _technologyProxy.isInstanceAvailable(proxy, objPath))
   {
      if((NULL != _callbackIf) && (NULL != proxy.get()))
      {
         return proxy->sendGetPropertiesRequest(*_callbackIf);
      }
   }
   return DEFAULT_ACT;
}

// IConnManDbusTechnologyProxyIf implementation --- end

} //ccdbusif
