/**
 * @file ConnManDbusParser.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief ConnMan DBUS Property Parser.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details ConnMan DBUS Property Parser.
 */

#include "ConnManDbusParser.h"
#include "ConnManDbusInterfaces.h"
#include "DbusParser.h"

#include "TraceClasses.h"
#define FW_S_IMPORT_INTERFACE_ASSERT
#define FW_S_IMPORT_INTERFACE_TRACE
#include "framework_if_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CCDBUSIF_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/ConnManDbusParser.cpp.trc.h"
#endif
#endif

namespace ccdbusif {
namespace connman {

static ConnManDbusInterfaces dbusInterfaces;
static DbusParser dbusParser;

ConnManDbusParser::ConnManDbusParser()
{
   _tracesEnabled = false;
   dbusParser.setTraces(_tracesEnabled);
}

ConnManDbusParser::~ConnManDbusParser()
{
}

void ConnManDbusParser::parseProperties(::std::vector<DbusVariantProperty>& outProperties, const ::std::vector< int >& matchingInterfaces, ::std::map< ::std::string, ::asf::dbus::DBusVariant >& inProperties,
      const DbusBusType busType, const ::std::string& busName, const ::std::string& objPath, const ::std::string& interfaceName) const
{
   dbusParser.parseProperties(outProperties, matchingInterfaces, inProperties, busType, busName, objPath, interfaceName, &dbusInterfaces);
}

void ConnManDbusParser::parseIntrospection(const ::std::string& data,
      const DbusBusType busType, const ::std::string& busName, const ::std::string& objPath, const ::std::string& interfaceName) const
{
   dbusParser.parseIntrospection(data, busType, busName, objPath, interfaceName, &dbusInterfaces);
}

void ConnManDbusParser::parseInterfacesAdded(const ::std::string& addedObjectPath, ::std::map< ::std::string, ::std::map< ::std::string, ::asf::dbus::DBusVariant > >& inInterfaces,
      const DbusBusType busType, const ::std::string& busName, const ::std::string& objPath, const ::std::string& interfaceName) const
{
   dbusParser.parseInterfacesAdded(addedObjectPath, inInterfaces, busType, busName, objPath, interfaceName, &dbusInterfaces);
}

void ConnManDbusParser::parseInterfacesRemoved(const ::std::string& removedObjectPath, const ::std::vector< ::std::string >& inInterfaces,
      const DbusBusType busType, const ::std::string& busName, const ::std::string& objPath, const ::std::string& interfaceName) const
{
   dbusParser.parseInterfacesRemoved(removedObjectPath, inInterfaces, busType, busName, objPath, interfaceName, &dbusInterfaces);
}

bool ConnManDbusParser::findAndParseProperty(::std::vector<DbusVariantProperty>& outPropertyList, ::std::map< ::std::string, ::asf::dbus::DBusVariant >& inProperties, const ::std::string& propertyName, const int interface) const
{
   return dbusParser.findAndParseProperty(outPropertyList, inProperties, propertyName, interface, &dbusInterfaces);
}

void ConnManDbusParser::parseProperty(::std::vector<DbusVariantProperty>& outPropertyList, ::asf::dbus::DBusVariant& propertyValue, const ::std::string& propertyName, const int interface) const
{
   dbusParser.parsePropertyValue(outPropertyList, propertyValue, propertyName, interface, &dbusInterfaces);
}

const ::std::string& ConnManDbusParser::getInterfaceName(const Interface interface) const
{
   return dbusInterfaces.getInterfaceName(interface);
}

ManagerProperty ConnManDbusParser::getManagerProperty2Enum(const ::std::string& value) const
{
   return dbusInterfaces.getManagerProperty2Enum(value);
}

const ::std::string& ConnManDbusParser::getManagerProperty2String(const ManagerProperty value) const
{
   return dbusInterfaces.getManagerProperty2String(value);
}

ManagerState ConnManDbusParser::getManagerState2Enum(const ::std::string& value) const
{
   return dbusInterfaces.getManagerState2Enum(value);
}

const ::std::string& ConnManDbusParser::getManagerState2String(const ManagerState value) const
{
   return dbusInterfaces.getManagerState2String(value);
}

ServiceProperty ConnManDbusParser::getServiceProperty2Enum(const ::std::string& value) const
{
   return dbusInterfaces.getServiceProperty2Enum(value);
}

const ::std::string& ConnManDbusParser::getServiceProperty2String(const ServiceProperty value) const
{
   return dbusInterfaces.getServiceProperty2String(value);
}

ServiceState ConnManDbusParser::getServiceState2Enum(const ::std::string& value) const
{
   return dbusInterfaces.getServiceState2Enum(value);
}

const ::std::string& ConnManDbusParser::getServiceState2String(const ServiceState value) const
{
   return dbusInterfaces.getServiceState2String(value);
}

ServiceType ConnManDbusParser::getServiceType2Enum(const ::std::string& value) const
{
   return dbusInterfaces.getServiceType2Enum(value);
}

const ::std::string& ConnManDbusParser::getServiceType2String(const ServiceType value) const
{
   return dbusInterfaces.getServiceType2String(value);
}

TechnologyProperty ConnManDbusParser::getTechnologyProperty2Enum(const ::std::string& value) const
{
   return dbusInterfaces.getTechnologyProperty2Enum(value);
}

const ::std::string& ConnManDbusParser::getTechnologyProperty2String(const TechnologyProperty value) const
{
   return dbusInterfaces.getTechnologyProperty2String(value);
}

void ConnManDbusParser::setTraces(const bool enabled)
{
   _tracesEnabled = enabled;
   dbusParser.setTraces(_tracesEnabled);
}

} //connman
} //ccdbusif
