/**
 * @file IConnManDbusServiceSendRequestIf.h
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief ConnMan DBUS Send Request Interface for Service.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Interface definition for ConnMan DBUS send request for Service.
 */

#ifndef _I_CONN_MAN_DBUS_SERVICE_SEND_REQUEST_IF_H_
#define _I_CONN_MAN_DBUS_SERVICE_SEND_REQUEST_IF_H_

#include <stdbool.h> // <cstdbool> is marked with C++11
#include <string>

#include "asf/core/Types.h"

namespace ccdbusif {

// class forward declarations
class IConnManDbusServiceCallbackIf;

/**
 * Send request interface class for Service.
 */
class IConnManDbusServiceSendRequestIf
{
public:
   /**
    * Destructor.
    */
   virtual ~IConnManDbusServiceSendRequestIf() {}

   /**
    * Set callback handler.
    *
    * @param[in] callbackIf: callback handler
    * @param[in] enableProxy: flag to enable/disable proxy
    */
   virtual void setCallbackIf(IConnManDbusServiceCallbackIf* callbackIf, bool enableProxy) = 0;

   /**
    * Handle signal for created service.
    *
    * @param[in] objPath: object path of created service
    */
   virtual void serviceCreated(const ::std::string& objPath) = 0;

   /**
    * Handle signal for removed service.
    *
    * @param[in] objPath: object path of removed service
    * @param[in] destroyDirectly: flag to trigger direct proxy destruction or via work item request
    */
   virtual void serviceRemoved(const ::std::string& objPath, const bool destroyDirectly = false) = 0;

   /**
    * Send GetProperties Request.
    *
    * @param[in] objPath: object path of service
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendGetPropertiesRequest(const ::std::string& objPath) = 0;

   /**
    * Send Connect Request.
    *
    * @param[in] objPath: object path of service
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendConnectRequest(const ::std::string& objPath) = 0;

   /**
    * Send Disconnect Request.
    *
    * @param[in] objPath: object path of service
    *
    * @return = asynchronous completion token
    */
   virtual act_t sendDisconnectRequest(const ::std::string& objPath) = 0;

};

} //ccdbusif

#endif //_I_CONN_MAN_DBUS_SERVICE_SEND_REQUEST_IF_H_
