/**
 * @file CcDbusIfTypes.cpp
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief Types.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Types for CcDbusIf.
 */

#include "CcDbusIfTypes.h"
#include "FwFormattedDataPrint.h"
#include <cstring>
#include <cstdio>
#include <stdint.h> // using <cstdint> compiler complains that std=c++0x or -std=gnu++0x compiler option must be enabled

#define FW_S_IMPORT_INTERFACE_ASSERT
#include "framework_if_if.h"

namespace ccdbusif {

ActInfo::ActInfo()
{
   memset(_buffer, 0, sizeof(_buffer));
}

ActInfo::ActInfo(const act_t token)
{
   memset(_buffer, 0, sizeof(_buffer));
#if __WORDSIZE == 64
   ::fw::FormattedOutputU64 output(token, true);
   (void)strncpy(_buffer, output.c_str(), (sizeof(_buffer) - 1));
#else
   ::fw::FormattedOutputU32 output(token, true);
   (void)strncpy(_buffer, output.c_str(), (sizeof(_buffer) - 1));
#endif
}

ActInfo::~ActInfo()
{
}

//------------------------------------------------------------------------------

DbusVariant::DbusVariant()
{
   _type = VARIANT_UNKNOWN;
   _intData.uint64Data = 0;
   // _stringData
   // _stringArray
}

DbusVariant::DbusVariant(const DbusVariant& ref)
{
   _type = ref._type;
   _intData.uint64Data = ref._intData.uint64Data;
   _stringData = ref._stringData;
   _stringArray = ref._stringArray;
}

DbusVariant& DbusVariant::operator=(const DbusVariant& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   _type = ref._type;
   _intData.uint64Data = ref._intData.uint64Data;
   _stringData = ref._stringData;
   _stringArray = ref._stringArray;

   return *this;
}

bool DbusVariant::operator==(const DbusVariant& ref) const
{
   bool result = true;

   result = (true == result) && (_type == ref._type);
   result = (true == result) && (_intData.uint64Data == ref._intData.uint64Data);
   result = (true == result) && (_stringData == ref._stringData);
   result = (true == result) && (_stringArray == ref._stringArray);

   return result;
}

bool DbusVariant::operator!=(const DbusVariant& ref) const
{
   return !(operator==(ref));
}

DbusVariant::~DbusVariant()
{
}

int8_t DbusVariant::getInt8(void) const
{
   FW_NORMAL_ASSERT(VARIANT_INT8 == _type);
   return _intData.int8Data;
}

uint8_t DbusVariant::getUInt8(void) const
{
   FW_NORMAL_ASSERT(VARIANT_UINT8 == _type);
   return _intData.uint8Data;
}

int16_t DbusVariant::getInt16(void) const
{
   FW_NORMAL_ASSERT(VARIANT_INT16 == _type);
   return _intData.int16Data;
}

uint16_t DbusVariant::getUInt16(void) const
{
   FW_NORMAL_ASSERT(VARIANT_UINT16 == _type);
   return _intData.uint16Data;
}

int32_t DbusVariant::getInt32(void) const
{
   FW_NORMAL_ASSERT(VARIANT_INT32 == _type);
   return _intData.int32Data;
}

uint32_t DbusVariant::getUInt32(void) const
{
   FW_NORMAL_ASSERT(VARIANT_UINT32 == _type);
   return _intData.uint32Data;
}

int64_t DbusVariant::getInt64(void) const
{
   FW_NORMAL_ASSERT(VARIANT_INT64 == _type);
   return _intData.int64Data;
}

uint64_t DbusVariant::getUInt64(void) const
{
   FW_NORMAL_ASSERT(VARIANT_UINT64 == _type);
   return _intData.uint64Data;
}

bool DbusVariant::getBool(void) const
{
   FW_NORMAL_ASSERT(VARIANT_BOOL == _type);
   return _intData.boolData;
}

int DbusVariant::getEnum(void) const
{
   FW_NORMAL_ASSERT(VARIANT_ENUM == _type);
   return _intData.enumData;
}

const ::std::string& DbusVariant::getString(void) const
{
   FW_NORMAL_ASSERT(VARIANT_STRING == _type);
   return _stringData;
}

const ::std::vector<std::string>& DbusVariant::getStringArray(void) const
{
   FW_NORMAL_ASSERT(VARIANT_STRING_ARRAY == _type);
   return _stringArray;
}

bool DbusVariant::getNone(void) const
{
   // no assert here because this check can be done every time
   return _intData.empty;
}

void DbusVariant::reset(void)
{
   _type = VARIANT_UNKNOWN;
   _intData.uint64Data = 0;
   _stringData.clear();
   _stringArray.clear();
}

//------------------------------------------------------------------------------

DbusVariantProperty::DbusVariantProperty()
{
   propEnum = 0;
   // propData
}

DbusVariantProperty::DbusVariantProperty(const DbusVariantProperty& ref)
{
   propEnum = ref.propEnum;
   propData = ref.propData;
}

DbusVariantProperty& DbusVariantProperty::operator=(const DbusVariantProperty& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   propEnum = ref.propEnum;
   propData = ref.propData;

   return *this;
}

bool DbusVariantProperty::operator==(const DbusVariantProperty& ref) const
{
   bool result = true;

   result = (true == result) && (propEnum == ref.propEnum);
   result = (true == result) && (propData == ref.propData);

   return result;
}

bool DbusVariantProperty::operator!=(const DbusVariantProperty& ref) const
{
   return !(operator==(ref));
}

DbusVariantProperty::~DbusVariantProperty()
{
}

void DbusVariantProperty::reset(void)
{
   propEnum = 0;
   propData.reset();
}

//------------------------------------------------------------------------------

DbusPropertyEntry::DbusPropertyEntry()
{
   propName = NULL;
   propEnum = 0;
}

DbusPropertyEntry::DbusPropertyEntry(const DbusPropertyEntry& ref)
{
   propName = NULL;
   setPropName(ref.propName);
   propEnum = ref.propEnum;
}

DbusPropertyEntry& DbusPropertyEntry::operator=(const DbusPropertyEntry& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   propName = NULL;
   setPropName(ref.propName);
   propEnum = ref.propEnum;

   return *this;
}

bool DbusPropertyEntry::operator==(const DbusPropertyEntry& ref) const
{
   return (0 == compare(ref));
}

bool DbusPropertyEntry::operator!=(const DbusPropertyEntry& ref) const
{
   return (0 != compare(ref));
}

bool DbusPropertyEntry::operator<(const DbusPropertyEntry& ref) const
{
   return (0 > compare(ref));
}

bool DbusPropertyEntry::operator>(const DbusPropertyEntry& ref) const
{
   return (0 < compare(ref));
}

DbusPropertyEntry::~DbusPropertyEntry()
{
   propName = NULL;
}

int DbusPropertyEntry::compare(const DbusPropertyEntry& ref) const
{
   uintptr_t left = (uintptr_t)propName;
   uintptr_t right = (uintptr_t)ref.propName;

   if(left < right)
   {
      return -1;
   }
   else if(left > right)
   {
      return 1;
   }

   // propName is equal

   if(propEnum < ref.propEnum)
   {
      return -1;
   }
   else if(propEnum > ref.propEnum)
   {
      return 1;
   }

   // propEnum is equal

   return 0;
}

//------------------------------------------------------------------------------

} //ccdbusif
