/**
 * @file DbusBaseProxyIf.h
 *
 * @par SW-Component
 * CcDbusIf
 *
 * @brief DBUS Base Proxy.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Base class for DBUS proxies.
 */

#ifndef _DBUS_BASE_PROXY_IF_H_
#define _DBUS_BASE_PROXY_IF_H_

#include "CommonBaseProxyIf.h"
#include "ICcDbusIfControllerClient.h"
#include "SetCallbackWorkItem.h"
#include "CcDbusIfUtils.h"
#include "ProxyMetaData.h"
#include <list>
#include <boost/shared_ptr.hpp>
#include "FwAssert.h"

// class forward declarations
namespace asf {
namespace core {
class ServiceAvailableIF;
} //core
} //asf

namespace ccdbusif {

/**
 * Callback entry class.
 */
template < class TCallback >
class DbusCallbackEntry
{
public:
   DbusCallbackEntry() :
   callback(0),
   methodActList(),
   metaDataList()
   {
   }

   DbusCallbackEntry(const DbusCallbackEntry& ref) :
   callback(ref.callback),
   methodActList(ref.methodActList),
   metaDataList(ref.metaDataList)
   {
   }

   DbusCallbackEntry& operator=(const DbusCallbackEntry& ref)
   {
      if(this == &ref)
      {
         return *this;
      }

      callback = ref.callback;
      methodActList = ref.methodActList;
      metaDataList = ref.metaDataList;

      return *this;
   }

   bool operator==(const DbusCallbackEntry& ref) const
   {
      bool result = true;

      result = (true == result) && (callback == ref.callback);
      result = (true == result) && (methodActList == ref.methodActList);
      result = (true == result) && (metaDataList == ref.metaDataList);

      return result;
   }

   bool operator!=(const DbusCallbackEntry& ref) const
   {
      return !(operator==(ref));
   }

   virtual ~DbusCallbackEntry()
   {
      callback = 0;
   }

   TCallback* callback; /**< callback handler */
   ::std::list< act_t > methodActList; /**< DBus method ACT list */
   ::std::vector< ProxyMetaData > metaDataList; /**< list of meta data of proxy the callback shall be used for */
};

/**
 * Proxy entry class.
 */
template < class TProxy >
class DbusProxyEntry
{
public:
   DbusProxyEntry() :
   creationState(PROXY_NOT_AVAILABLE),
   proxy()
   {
   }

   DbusProxyEntry(const DbusProxyEntry& ref) :
   creationState(ref.creationState),
   proxy(ref.proxy)
   {
   }

   DbusProxyEntry& operator=(const DbusProxyEntry& ref)
   {
      if(this == &ref)
      {
         return *this;
      }

      creationState = ref.creationState;
      proxy = ref.proxy;

      return *this;
   }

   bool operator==(const DbusProxyEntry& ref) const
   {
      bool result = true;

      result = (true == result) && (creationState == ref.creationState);
      result = (true == result) && (proxy == ref.proxy);

      return result;
   }

   bool operator!=(const DbusProxyEntry& ref) const
   {
      return !(operator==(ref));
   }

   virtual ~DbusProxyEntry()
   {
   }

   DbusProxyCreationState creationState; /**< creation state */
   ::boost::shared_ptr< TProxy > proxy; /**< proxy */
};

/**
 * DBus base proxy class.
 */
template < class TCallback, class TProxy, class TTestProxy >
class DbusBaseProxyIf : public CommonBaseProxyIf
{
protected:
   DbusBaseProxyIf(ICcDbusIfControllerClient* client) :
   CommonBaseProxyIf(client),
   _callbackList(),
   _proxyList(),
   _testProxyIf(0)
   {
   }

   DbusBaseProxyIf(ICcDbusIfControllerClient* client, TTestProxy* testProxy) :
   CommonBaseProxyIf(client),
   _callbackList(),
   _proxyList(),
   _testProxyIf(testProxy)
   {
   }

   virtual ~DbusBaseProxyIf()
   {
      _testProxyIf = 0;
   }

   ::std::map< unsigned int, DbusCallbackEntry< TCallback > > _callbackList; /**< callback list */
   ::std::map< ProxyMetaData, DbusProxyEntry< TProxy > > _proxyList; /**< proxy list */
   TTestProxy* _testProxyIf; /**< test proxy interface */

   virtual void internalCreateProxy(const unsigned int callbackId, const ::std::string& objPath, const ::std::string& busName, const DbusBusType busType) = 0;

   virtual void internalDestroyProxy(const unsigned int callbackId, const ::std::string& objPath, const ::std::string& busName, const DbusBusType busType) = 0;

   virtual void internalDestroyAllProxies(void) = 0;

   void processSetCallback(ISetCallback< TCallback >* setCallback, TCallback* callbackIf, const bool enableProxy, const unsigned int callbackId, const ::std::string& objPath, const ::std::string& busName, const DbusBusType busType)
   {
      FW_IF_NULL_PTR_RETURN(callbackIf);
      FW_IF_NULL_PTR_RETURN(_controllerClient);
      _controllerClient->pushWorkItem(new SetCallbackWorkItem< TCallback >(setCallback, callbackIf, enableProxy, callbackId, objPath, busName, busType));
   }

   void storeCallback(TCallback* callbackIf, const bool enableProxy, const unsigned int callbackId, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM)
   {
      _enableProxy = enableProxy;

      if(_callbackList.end() == _callbackList.find(callbackId))
      {
         // add new entry
         DbusCallbackEntry< TCallback >& entry = _callbackList[callbackId];

         entry.callback = callbackIf;
         // first meta data entry
         entry.metaDataList.push_back(ProxyMetaData(objPath, busName, busType));
      }
      else
      {
         // existing entry
         DbusCallbackEntry< TCallback >& entry = _callbackList[callbackId];

         FW_NORMAL_ASSERT(entry.callback == callbackIf);
         // next meta data entry
         entry.metaDataList.push_back(ProxyMetaData(objPath, busName, busType));
      }
   }

   /**
    * Check if proxy for given meta data is available.
    * <BR><B>Only the parameters used to distinguish different proxies shall be passed.</B>
    *
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    *
    * @return = true: proxy is available,
    * @return = false: proxy is not available
    */
   bool isProxyAvailable(const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM) const
   {
      const ProxyMetaData metaData(objPath, busName, busType);

      return (_proxyList.end() != _proxyList.find(metaData));
   }

   /**
    * Check if proxy for given meta data is available.
    * <BR><B>Only the parameters used to distinguish different proxies shall be passed.</B>
    *
    * @param[out] proxy: proxy
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    *
    * @return = true: proxy is available,
    * @return = false: proxy is not available
    */
   bool isProxyAvailable(::boost::shared_ptr< TProxy >& proxy, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM) const
   {
      const ProxyMetaData metaData(objPath, busName, busType);

      typename ::std::map< ProxyMetaData, DbusProxyEntry< TProxy > >::const_iterator it = _proxyList.find(metaData);

      if(_proxyList.end() != it)
      {
         proxy = it->second.proxy;

         return true;
      }

      return false;
   }

   /**
    * Check if proxy for given meta data is available.
    * <BR><B>Only the parameters used to distinguish different proxies shall be passed.</B>
    *
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    *
    * @return = true: proxy is available,
    * @return = false: proxy is not available
    */
   bool isTestProxyAvailable(const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM) const
   {
      const ProxyMetaData metaData(objPath, busName, busType);

      return (_proxyList.end() != _proxyList.find(metaData));
   }

   /**
    * Create proxy.
    * <BR><B>All parameters shall be passed.</B>
    *
    * @param[in] port: port
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    * @param[in] serviceAvailableIf: callback
    *
    * @return = created proxy
    */
   ::boost::shared_ptr< TProxy > createProxy(const ::std::string& port, const ::std::string& objPath, const ::std::string& busName, const DbusBusType busType, ::asf::core::ServiceAvailableIF& serviceAvailableIf)
   {
      ::com::bosch::cm::asf::lang::dbus::Connectors::DBusConnector connector;
      convertBusType2ConnectorOption(connector, busType);
      return TProxy::createProxy(port, busName, objPath, connector, serviceAvailableIf);
   }

   /**
    * Add proxy.
    * <BR><B>Only the parameters used to distinguish different proxies shall be passed.</B>
    *
    * @param[in] proxy: proxy
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    */
   void addProxy(::boost::shared_ptr< TProxy >& proxy, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM)
   {
      const ProxyMetaData metaData(objPath, busName, busType);

      // add new entry, check was done before outside
      DbusProxyEntry< TProxy >& entry = _proxyList[metaData];

      entry.creationState = PROXY_CREATING; // in case of instance is added then proxy creation was started
      entry.proxy = proxy; // store proxy
   }

   /**
    * Create test proxy.
    * <BR><B>All parameters shall be passed.</B>
    *
    * @param[in] port: port
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    * @param[in] serviceAvailableIf: callback
    */
   void createTestProxy(const ::std::string& port, const ::std::string& objPath, const ::std::string& busName, const DbusBusType busType, TCallback& serviceAvailableIf)
   {
      FW_IF_NULL_PTR_RETURN(_testProxyIf);
      _testProxyIf->createProxy(serviceAvailableIf, port, objPath, busName, busType);
   }

   /**
    * Add test proxy.
    * <BR><B>Only the parameters used to distinguish different proxies shall be passed.</B>
    *
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    */
   void addTestProxy(const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM)
   {
      const ProxyMetaData metaData(objPath, busName, busType);

      // add new entry, check was done before outside
      DbusProxyEntry< TProxy >& entry = _proxyList[metaData];

      entry.creationState = PROXY_CREATING; // in case of instance is added then proxy creation was started
      // no proxy to be added
   }

   /**
    * Remove proxy.
    * <BR><B>Only the parameters used to distinguish different proxies shall be passed.</B>
    *
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    */
   void removeProxy(const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM)
   {
      const ProxyMetaData metaData(objPath, busName, busType);

      typename ::std::map< ProxyMetaData, DbusProxyEntry< TProxy > >::iterator it = _proxyList.find(metaData);

      if(_proxyList.end() != it)
      {
         it->second.proxy.reset();
         _proxyList.erase(it);
      }
   }

   /**
    * Destroy test proxy.
    * <BR><B>All parameters shall be passed.</B>
    *
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    */
   void destroyTestProxy(const ::std::string& objPath, const ::std::string& busName, const DbusBusType busType)
   {
      FW_IF_NULL_PTR_RETURN(_testProxyIf);
      _testProxyIf->destroyProxy(objPath, busName, busType);
   }

   /**
    * Remove test proxy.
    * <BR><B>Only the parameters used to distinguish different proxies shall be passed.</B>
    *
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    */
   void removeTestProxy(const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM)
   {
      const ProxyMetaData metaData(objPath, busName, busType);

      typename ::std::map< ProxyMetaData, DbusProxyEntry< TProxy > >::iterator it = _proxyList.find(metaData);

      if(_proxyList.end() != it)
      {
         // no proxy available
         _proxyList.erase(it);
      }
   }

   void removeAllProxies(void)
   {
      for(typename ::std::map< ProxyMetaData, DbusProxyEntry< TProxy > >::iterator it = _proxyList.begin(); it != _proxyList.end(); ++it)
      {
         it->second.proxy.reset();
      }
      _proxyList.clear();
   }

   void destroyAllTestProxies(void)
   {
      FW_IF_NULL_PTR_RETURN(_testProxyIf);
      _testProxyIf->destroyAllProxies();
   }

   void removeAllTestProxies(void)
   {
      // no proxies available
      _proxyList.clear();
   }

   void removeCallbacks(void)
   {
      _callbackList.clear();
   }

   size_t getProxyListSize(void) const
   {
      return _proxyList.size();
   }

   ::boost::shared_ptr< TProxy >* getProxy(const size_t index)
   {
      if(_proxyList.size() <= index)
      {
         // forgot to check size before?
         FW_NORMAL_ASSERT_ALWAYS();
         return 0;
      }

      typename ::std::map< ProxyMetaData, DbusProxyEntry< TProxy > >::iterator it = _proxyList.begin();

      for(size_t i = 0; i < index; i++)
      {
         ++it;
      }

      return &(it->second.proxy);
   }

   const ::boost::shared_ptr< TProxy >* getProxy(const size_t index) const
   {
      if(_proxyList.size() <= index)
      {
         // forgot to check size before?
         FW_NORMAL_ASSERT_ALWAYS();
         return 0;
      }

      typename ::std::map< ProxyMetaData, DbusProxyEntry< TProxy > >::const_iterator it = _proxyList.begin() + index;

      return &(it->second.proxy);
   }

   size_t getCallbackListSize(void) const
   {
      return _callbackList.size();
   }

   TCallback* getCallbackEntry(const size_t index)
   {
      if(_callbackList.size() <= index)
      {
         // forgot to check size before?
         FW_NORMAL_ASSERT_ALWAYS();
         return 0;
      }

      typename ::std::map< unsigned int, DbusCallbackEntry< TCallback > >::iterator it = _callbackList.begin();

      for(size_t i = 0; i < index; i++)
      {
         ++it;
      }

      return it->second.callback;
   }

   const TCallback* getCallbackEntry(const size_t index) const
   {
      if(_callbackList.size() <= index)
      {
         // forgot to check size before?
         FW_NORMAL_ASSERT_ALWAYS();
         return 0;
      }

      typename ::std::map< unsigned int, DbusCallbackEntry< TCallback > >::const_iterator it = _callbackList.begin() + index;

      return it->second.callback;
   }

   void getProxyMetaData(const size_t index, ::std::string& objPath, ::std::string& busName, DbusBusType& busType) const
   {
      if(_proxyList.size() <= index)
      {
         // forgot to check size before?
         FW_NORMAL_ASSERT_ALWAYS();
         return;
      }

      typename ::std::map< ProxyMetaData, DbusProxyEntry< TProxy > >::const_iterator it = _proxyList.begin() + index;

      objPath = it->first.objPath;
      busName = it->first.busName;
      busType = it->first.busType;
   }

   /**
    * Get proxy.
    * <BR><B>Only the parameters used to distinguish different proxies shall be passed.</B>
    *
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    *
    * @return = proxy
    */
   ::boost::shared_ptr< TProxy >* getProxy(const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM)
   {
      const ProxyMetaData metaData(objPath, busName, busType);

      typename ::std::map< ProxyMetaData, DbusProxyEntry< TProxy > >::iterator it = _proxyList.find(metaData);

      if(_proxyList.end() != it)
      {
         return &(it->second.proxy);
      }

      FW_NORMAL_ASSERT_ALWAYS();

      return 0;
   }

   /**
    * Get proxy.
    * <BR><B>Only the parameters used to distinguish different proxies shall be passed.</B>
    *
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    *
    * @return = proxy
    */
   const ::boost::shared_ptr< TProxy >* getProxy(const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM) const
   {
      const ProxyMetaData metaData(objPath, busName, busType);

      typename ::std::map< ProxyMetaData, DbusProxyEntry< TProxy > >::const_iterator it = _proxyList.find(metaData);

      if(_proxyList.end() != it)
      {
         return &(it->second.proxy);
      }

      FW_NORMAL_ASSERT_ALWAYS();

      return 0;
   }

   /**
    * Get creation state.
    * <BR><B>Only the parameters used to distinguish different proxies shall be passed.</B>
    *
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    */
   DbusProxyCreationState getCreationState(const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM) const
   {
      const ProxyMetaData metaData(objPath, busName, busType);

      typename ::std::map< ProxyMetaData, DbusProxyEntry< TProxy > >::const_iterator it = _proxyList.find(metaData);

      if(_proxyList.end() != it)
      {
         return it->second.creationState;
      }

      FW_NORMAL_ASSERT_ALWAYS();

      return PROXY_NOT_AVAILABLE;
   }

   /**
    * Set creation state.
    * <BR><B>Only the parameters used to distinguish different proxies shall be passed.</B>
    *
    * @param[in] available: availability state
    * @param[in] objPath: object path
    * @param[in] busName: bus name
    * @param[in] busType: bus type
    */
   void setCreationState(const bool available, const ::std::string& objPath = ::std::string(), const ::std::string& busName = ::std::string(), const DbusBusType busType = BUS_TYPE_SYSTEM)
   {
      const ProxyMetaData metaData(objPath, busName, busType);

      typename ::std::map< ProxyMetaData, DbusProxyEntry< TProxy > >::iterator it = _proxyList.find(metaData);

      if(_proxyList.end() != it)
      {
         if(true == available)
         {
            it->second.creationState = PROXY_AVAILABLE;
         }
         else
         {
            it->second.creationState = PROXY_NOT_AVAILABLE;
         }

         return;
      }

      FW_NORMAL_ASSERT_ALWAYS();
   }

   TCallback* getCallback(const unsigned int callbackId)
   {
      typename ::std::map< unsigned int, DbusCallbackEntry< TCallback > >::iterator it = _callbackList.find(callbackId);

      if(_callbackList.end() != it)
      {
         return it->second.callback;
      }

      FW_NORMAL_ASSERT_ALWAYS();

      return 0;
   }

   void updateCurrentAvailableStatus(const unsigned int callbackId, const DbusProxyCreationState creationState, const ::std::string& objPath, const ::std::string& busName, const DbusBusType busType)
   {
      // inform via callback handler
      if(PROXY_AVAILABLE == creationState)
      {
         TCallback* cbHandler(getCallback(callbackId));
         FW_IF_NULL_PTR_RETURN(cbHandler);
         cbHandler->onAvailableCb(true, objPath, busName, busType);
      }
      else if(PROXY_NOT_AVAILABLE == creationState)
      {
         TCallback* cbHandler(getCallback(callbackId));
         FW_IF_NULL_PTR_RETURN(cbHandler);
         cbHandler->onUnavailableCb(false, objPath, busName, busType);
      }
   }

   /**
    * Update availability state.
    * <BR><B>Only the parameters used to distinguish different proxies shall be passed.</B>
    *
    * @param[in] available: availability state
    * @param[in] objPath: object path (to be reported)
    * @param[in] busName: bus name (to be reported)
    * @param[in] busType: bus type (to be reported)
    * @param[in] proxyObjPath: object path (to check meta data)
    * @param[in] proxyBusName: bus name (to check meta data)
    * @param[in] proxyBusType: bus type (to check meta data)
    */
   void updateAvailableStatusToAll(const bool available, const ::std::string& objPath, const ::std::string& busName, const DbusBusType busType, const ::std::string& proxyObjPath = ::std::string(), const ::std::string& proxyBusName = ::std::string(), const DbusBusType proxyBusType = BUS_TYPE_SYSTEM)
   {
      const ProxyMetaData metaData(proxyObjPath, proxyBusName, proxyBusType);

      // inform via callback handler
      for(typename ::std::map< unsigned int, DbusCallbackEntry< TCallback > >::iterator it = _callbackList.begin(); it != _callbackList.end(); ++it)
      {
         if(0 != it->second.callback)
         {
            for(size_t i = 0; i < it->second.metaDataList.size(); i++)
            {
               if(metaData == it->second.metaDataList[i])
               {
                  it->second.callback->onAvailableCb(available, objPath, busName, busType);
               }
            }
         }
      }
   }

   /**
    * Update availability state.
    * <BR><B>Only the parameters used to distinguish different proxies shall be passed.</B>
    *
    * @param[in] available: availability state
    * @param[in] objPath: object path (to be reported)
    * @param[in] busName: bus name (to be reported)
    * @param[in] busType: bus type (to be reported)
    * @param[in] proxyObjPath: object path (to check meta data)
    * @param[in] proxyBusName: bus name (to check meta data)
    * @param[in] proxyBusType: bus type (to check meta data)
    */
   void updateUnavailableStatusToAll(const bool available, const ::std::string& objPath, const ::std::string& busName, const DbusBusType busType, const ::std::string& proxyObjPath = ::std::string(), const ::std::string& proxyBusName = ::std::string(), const DbusBusType proxyBusType = BUS_TYPE_SYSTEM)
   {
      const ProxyMetaData metaData(proxyObjPath, proxyBusName, proxyBusType);

      // inform via callback handler
      for(typename ::std::map< unsigned int, DbusCallbackEntry< TCallback > >::iterator it = _callbackList.begin(); it != _callbackList.end(); ++it)
      {
         if(0 != it->second.callback)
         {
            for(size_t i = 0; i < it->second.metaDataList.size(); i++)
            {
               if(metaData == it->second.metaDataList[i])
               {
                  it->second.callback->onUnavailableCb(available, objPath, busName, busType);
               }
            }
         }
      }
   }

   void addAct(const unsigned int callbackId, const act_t act)
   {
      if(DEFAULT_ACT == act)
      {
         return;
      }

      typename ::std::map< unsigned int, DbusCallbackEntry< TCallback > >::iterator it = _callbackList.find(callbackId);

      if(_callbackList.end() != it)
      {
         it->second.methodActList.push_back(act);
         return;
      }

      FW_NORMAL_ASSERT_ALWAYS();
   }

   TCallback* removeActAndFindCallback(const act_t act)
   {
      for(typename ::std::map< unsigned int, DbusCallbackEntry< TCallback > >::iterator it = _callbackList.begin(); it != _callbackList.end(); ++it)
      {
         ::std::list< act_t >& localMethodActList = it->second.methodActList;
         for(::std::list< act_t >::iterator itAct = localMethodActList.begin(); itAct != localMethodActList.end(); ++itAct)
         {
            if(act == *itAct)
            {
               localMethodActList.erase(itAct);
               return it->second.callback;
            }
         }
      }

      // no assert because in case of property update we will not have any matching stored ACT

      return 0;
   }
};

} //ccdbusif

#endif //_DBUS_BASE_PROXY_IF_H_
