/**
 * @file WblWiFiSetupStatusWbl.cpp
 *
 * @par SW-Component
 * State machine for WBL WiFi setup status
 *
 * @brief Implementation of WBL WiFi setup status state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of WBL WiFi setup status state machine.
 */

#include "WblWiFiSetupStatusWbl.h"
#include "IWblWiFiSetupStatusCallback.h"
#include "Bts2Ipc_MessageWrapper_WBL.h"
#include "WblObjectPathManagerWbl.h"
#include "cc_dbus_if/WblDbusParser.h"

#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/WblWiFiSetupStatusWbl.cpp.trc.h"
#endif
#endif

namespace btstackif {
namespace wbl {

WblWiFiSetupStatusWbl::WblWiFiSetupStatusWbl() :
_callback(0),
_controlIf(0),
_objectPathManager(0),
_idList()
{
}

WblWiFiSetupStatusWbl::~WblWiFiSetupStatusWbl()
{
   _callback = 0;
   _controlIf = 0;
   _objectPathManager = 0;
}

void WblWiFiSetupStatusWbl::setCallback(IN IWblWiFiSetupStatusCallback* callback)
{
   _callback = callback;

   FW_NORMAL_ASSERT(0 != _callback);
}

void WblWiFiSetupStatusWbl::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_NORMAL_ASSERT(0 != _controlIf);
}

void WblWiFiSetupStatusWbl::getWifiSetupList(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   (void)(bts2AppMsgList);

   createGetManagedObjectsMsg(bts2IpcMsgList);
}

void WblWiFiSetupStatusWbl::getWifiMode(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSWblWiFiSetupId id)
{
   (void)(bts2AppMsgList);

   FW_IF_NULL_PTR_RETURN(_objectPathManager);

   // all necessary checks are done before

   BTSObjectPath objPath;
   if(false == _objectPathManager->getObjectPath4Id(objPath, id))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   createGetWifiModeMsg(bts2IpcMsgList, id, objPath);
}

void WblWiFiSetupStatusWbl::getWifiPowerState(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSWblWiFiSetupId id)
{
   (void)(bts2AppMsgList);

   FW_IF_NULL_PTR_RETURN(_objectPathManager);

   // all necessary checks are done before

   BTSObjectPath objPath;
   if(false == _objectPathManager->getObjectPath4Id(objPath, id))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   createGetWifiPowerStateMsg(bts2IpcMsgList, id, objPath);
}

void WblWiFiSetupStatusWbl::getWifiFrequency(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSWblWiFiSetupId id)
{
   (void)(bts2AppMsgList);

   FW_IF_NULL_PTR_RETURN(_objectPathManager);

   // all necessary checks are done before

   BTSObjectPath objPath;
   if(false == _objectPathManager->getObjectPath4Id(objPath, id))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   createGetWifiFrequencyMsg(bts2IpcMsgList, id, objPath);
}

void WblWiFiSetupStatusWbl::getAccessPointConfigType(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSWblWiFiSetupId id)
{
   (void)(bts2AppMsgList);

   FW_IF_NULL_PTR_RETURN(_objectPathManager);

   // all necessary checks are done before

   BTSObjectPath objPath;
   if(false == _objectPathManager->getObjectPath4Id(objPath, id))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   createGetAccessPointConfigTypeMsg(bts2IpcMsgList, id, objPath);
}

void WblWiFiSetupStatusWbl::getSupportedRestrictions(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSWblWiFiSetupId id)
{
   (void)(bts2AppMsgList);

   FW_IF_NULL_PTR_RETURN(_objectPathManager);

   // all necessary checks are done before

   BTSObjectPath objPath;
   if(false == _objectPathManager->getObjectPath4Id(objPath, id))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   createGetSupportedFrequenciesMsg(bts2IpcMsgList, id, objPath);
}

void WblWiFiSetupStatusWbl::getWifiHealthinessIndicator(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSWblWiFiSetupId id)
{
   (void)(bts2AppMsgList);

   FW_IF_NULL_PTR_RETURN(_objectPathManager);

   // all necessary checks are done before

   BTSObjectPath objPath;
   if(false == _objectPathManager->getObjectPath4Id(objPath, id))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   createGetWifiConnectionStatusMsg(bts2IpcMsgList, id, objPath);
}

IWblWiFiSetupStatusRequest* WblWiFiSetupStatusWbl::getRequestIf(void)
{
   return this;
}

void WblWiFiSetupStatusWbl::setObjectPathManagerIf(IN IWblObjectPathManagerWbl* manager)
{
   _objectPathManager = manager;

   FW_NORMAL_ASSERT(0 != _objectPathManager);
}

void WblWiFiSetupStatusWbl::handleGetManagedObjectsResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result)
{
   FW_IF_NULL_PTR_RETURN(_callback);

   _callback->getWifiSetupListResult(bts2IpcMsgList, bts2AppMsgList, messageItem, result);
}

void WblWiFiSetupStatusWbl::updateServiceAvailability(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSWblDbusServiceInterface interface, IN const BTSDbusServiceAvailability availabilityEvent, IN const BTSObjectPath& objPath)
{
   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManager);

   if(BTS_WBL_DBUS_SERVICE_WIFI_SETUP != interface)
   {
      return;
   }

   // object path was added before, id was added to internal list before

   BTSWblWiFiSetupId storedId(0);
   if(true == _objectPathManager->getId4ObjectPath(storedId, objPath))
   {
      FW_IF_NULL_PTR_RETURN(storedId);
   }
   else
   {
      // something went wrong!?
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // check if id is part of internal list
   if(false == isIdAvailable(storedId))
   {
      // something went wrong!?
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(BTS_DBUS_SERVICE_AVAILABLE == availabilityEvent)
   {
      _callback->wifiSetupEntryAvailable(bts2IpcMsgList, bts2AppMsgList, messageItem, storedId);
   }
   else if(BTS_DBUS_SERVICE_NOT_AVAILABLE == availabilityEvent)
   {
      _callback->wifiSetupEntryUnavailable(bts2IpcMsgList, bts2AppMsgList, messageItem, storedId);
   }
   else if(BTS_DBUS_SERVICE_WAITING == availabilityEvent)
   {
      // ignore
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }
}

void WblWiFiSetupStatusWbl::wifiSetupEntryAdded(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& objPath)
{
   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManager);

   BTSWblWiFiSetupId storedId(0);
   if(true == _objectPathManager->getId4ObjectPath(storedId, objPath))
   {
      FW_IF_NULL_PTR_RETURN(storedId);
   }
   else
   {
      // get free id and store new mapping
      storedId = _objectPathManager->getFreeId();
      FW_IF_NULL_PTR_RETURN(storedId);
      _objectPathManager->setId2ObjectPath(storedId, objPath);
   }

   // check if id is part of internal list
   if(false == isIdAvailable(storedId))
   {
      // add id to internal list
      addId(storedId);

      _callback->wifiSetupEntryAdded(bts2IpcMsgList, bts2AppMsgList, messageItem, storedId);
   }
}

void WblWiFiSetupStatusWbl::wifiSetupEntryRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& objPath)
{
   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManager);

   BTSWblWiFiSetupId storedId(0);
   if(true == _objectPathManager->getId4ObjectPath(storedId, objPath))
   {
      FW_IF_NULL_PTR_RETURN(storedId);
   }
   else
   {
      // no matching entry
      return;
   }

   // check if id is part of internal list
   if(true == isIdAvailable(storedId))
   {
      // remove id from internal list
      removeId(storedId);

      _callback->wifiSetupEntryRemoved(bts2IpcMsgList, bts2AppMsgList, messageItem, storedId);
   }
}

void WblWiFiSetupStatusWbl::updateWifiMode(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const BTSObjectPath& objPath, IN const BTSWblWifiMode mode)
{
   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManager);

   BTSWblWiFiSetupId storedId(0);
   if(false == _objectPathManager->getId4ObjectPath(storedId, objPath))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   FW_IF_NULL_PTR_RETURN(storedId);

   _callback->updateWifiMode(bts2IpcMsgList, bts2AppMsgList, messageItem, result, storedId, mode);
}

void WblWiFiSetupStatusWbl::updateWifiPowerState(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const BTSObjectPath& objPath, IN const BTSWblWifiPowerState powerState)
{
   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManager);

   BTSWblWiFiSetupId storedId(0);
   if(false == _objectPathManager->getId4ObjectPath(storedId, objPath))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   FW_IF_NULL_PTR_RETURN(storedId);

   _callback->updateWifiPowerState(bts2IpcMsgList, bts2AppMsgList, messageItem, result, storedId, powerState);
}

void WblWiFiSetupStatusWbl::updateWifiFrequency(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const BTSObjectPath& objPath, IN const BTSWblWifiFrequency frequency)
{
   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManager);

   BTSWblWiFiSetupId storedId(0);
   if(false == _objectPathManager->getId4ObjectPath(storedId, objPath))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   FW_IF_NULL_PTR_RETURN(storedId);

   _callback->updateWifiFrequency(bts2IpcMsgList, bts2AppMsgList, messageItem, result, storedId, frequency);
}

void WblWiFiSetupStatusWbl::updateAccessPointConfigType(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const BTSObjectPath& objPath, IN const BTSWblAccessPointConfigType type)
{
   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManager);

   BTSWblWiFiSetupId storedId(0);
   if(false == _objectPathManager->getId4ObjectPath(storedId, objPath))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   FW_IF_NULL_PTR_RETURN(storedId);

   _callback->updateAccessPointConfigType(bts2IpcMsgList, bts2AppMsgList, messageItem, result, storedId, type);
}

void WblWiFiSetupStatusWbl::updateSupportedRestrictions(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const BTSObjectPath& objPath, IN const BTSWblSupportedRestrictionList& supportedRestrictions)
{
   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManager);

   BTSWblWiFiSetupId storedId(0);
   if(false == _objectPathManager->getId4ObjectPath(storedId, objPath))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   FW_IF_NULL_PTR_RETURN(storedId);

   _callback->updateSupportedRestrictions(bts2IpcMsgList, bts2AppMsgList, messageItem, result, storedId, supportedRestrictions);
}

void WblWiFiSetupStatusWbl::updateWifiConnectionStatus(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const BTSObjectPath& objPath, IN const bool connected, IN const BTSWblHealthinessIndicator healthIndicator)
{
   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManager);

   BTSWblWiFiSetupId storedId(0);
   if(false == _objectPathManager->getId4ObjectPath(storedId, objPath))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   FW_IF_NULL_PTR_RETURN(storedId);

   _callback->updateWifiHealthinessIndicator(bts2IpcMsgList, bts2AppMsgList, messageItem, result, storedId, connected, healthIndicator);
}

void WblWiFiSetupStatusWbl::removeId(const BTSWblWiFiSetupId id)
{
   ::std::map< BTSWblWiFiSetupId, bool >::iterator it = _idList.find(id);
   if(_idList.end() != it)
   {
      _idList.erase(it);
   }
}

bool WblWiFiSetupStatusWbl::isIdAvailable(const BTSWblWiFiSetupId id) const
{
   return (_idList.end() != _idList.find(id));
}

void WblWiFiSetupStatusWbl::addId(const BTSWblWiFiSetupId id)
{
   _idList[id] = true;
}

void WblWiFiSetupStatusWbl::createGetManagedObjectsMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList)
{
   Bts2Ipc_GetManagedObjectsWbl* msg = ptrNew_Bts2Ipc_GetManagedObjectsWbl();
   if(0 != msg)
   {
      msg->setResponseMessageFlag(true);

      ::ccdbusif::wbl::WblDbusParser parser;
      msg->setBusType((BTSCommonEnumClass)parser.getBusType(::ccdbusif::wbl::IF_LAST_INTENDED_MODE));
      msg->setBusName(parser.getBusName(::ccdbusif::wbl::IF_LAST_INTENDED_MODE));
      msg->setObjPath(parser.getRootObjectPath());

      bts2IpcMsgList.push_back(msg);
   }
}

void WblWiFiSetupStatusWbl::createGetWifiModeMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSWblWiFiSetupId id, IN const BTSObjectPath& objPath)
{
   Bts2Ipc_GetWifiMode* msg = ptrNew_Bts2Ipc_GetWifiMode();
   if(0 != msg)
   {
      msg->setResponseMessageFlag(true);
      msg->setId(id);
      msg->setObjPath(objPath);

      bts2IpcMsgList.push_back(msg);
   }
}

void WblWiFiSetupStatusWbl::createGetWifiPowerStateMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSWblWiFiSetupId id, IN const BTSObjectPath& objPath)
{
   Bts2Ipc_GetWifiPowerState* msg = ptrNew_Bts2Ipc_GetWifiPowerState();
   if(0 != msg)
   {
      msg->setResponseMessageFlag(true);
      msg->setId(id);
      msg->setObjPath(objPath);

      bts2IpcMsgList.push_back(msg);
   }
}

void WblWiFiSetupStatusWbl::createGetWifiFrequencyMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSWblWiFiSetupId id, IN const BTSObjectPath& objPath)
{
   Bts2Ipc_GetWifiFrequency* msg = ptrNew_Bts2Ipc_GetWifiFrequency();
   if(0 != msg)
   {
      msg->setResponseMessageFlag(true);
      msg->setId(id);
      msg->setObjPath(objPath);

      bts2IpcMsgList.push_back(msg);
   }
}

void WblWiFiSetupStatusWbl::createGetAccessPointConfigTypeMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSWblWiFiSetupId id, IN const BTSObjectPath& objPath)
{
   Bts2Ipc_GetAccessPointConfigType* msg = ptrNew_Bts2Ipc_GetAccessPointConfigType();
   if(0 != msg)
   {
      msg->setResponseMessageFlag(true);
      msg->setId(id);
      msg->setObjPath(objPath);

      bts2IpcMsgList.push_back(msg);
   }
}

void WblWiFiSetupStatusWbl::createGetSupportedFrequenciesMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSWblWiFiSetupId id, IN const BTSObjectPath& objPath)
{
   Bts2Ipc_GetSupportedFrequencies* msg = ptrNew_Bts2Ipc_GetSupportedFrequencies();
   if(0 != msg)
   {
      msg->setResponseMessageFlag(true);
      msg->setId(id);
      msg->setObjPath(objPath);

      bts2IpcMsgList.push_back(msg);
   }
}

void WblWiFiSetupStatusWbl::createGetWifiConnectionStatusMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSWblWiFiSetupId id, IN const BTSObjectPath& objPath)
{
   Bts2Ipc_GetWifiConnectionStatus* msg = ptrNew_Bts2Ipc_GetWifiConnectionStatus();
   if(0 != msg)
   {
      msg->setResponseMessageFlag(true);
      msg->setId(id);
      msg->setObjPath(objPath);

      bts2IpcMsgList.push_back(msg);
   }
}

} //wbl
} //btstackif
