/**
 * @file WblStartupWbl.cpp
 *
 * @par SW-Component
 * State machine for WBL startup
 *
 * @brief Implementation of WBL startup state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of WBL startup state machine.
 */

#include "WblStartupWbl.h"
#include "IWblStartupCallback.h"

#include "cc_dbus_if/WblDbusParser.h"

#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/WblStartupWbl.cpp.trc.h"
#endif
#endif

namespace btstackif {
namespace wbl {

WblStartupWbl::WblStartupWbl() :
_callback(0),
_controlIf(0),
_serviceGroupGeneral(),
_serviceLastIntendedMode(),
_serviceConflictManagement(),
_serviceObjectManager()
{
   _serviceGroupGeneral.setEnabled(true);
   _serviceGroupGeneral.disableTimer();
   _serviceGroupGeneral.setName("GroupGeneral");
   _serviceGroupGeneral.addServiceInfo("");

   _serviceLastIntendedMode.setEnabled(true);
   _serviceLastIntendedMode.disableTimer(); // we will not track any timeout
   _serviceLastIntendedMode.setName("WblLastIntendedMode");
   _serviceLastIntendedMode.setInterface((BTSCommonEnumClass)BTS_WBL_DBUS_SERVICE_LAST_INTENDED_MODE);

   _serviceConflictManagement.setEnabled(true);
   _serviceConflictManagement.disableTimer(); // we will not track any timeout
   _serviceConflictManagement.setName("WblConflictManagement");
   _serviceConflictManagement.setInterface((BTSCommonEnumClass)BTS_WBL_DBUS_SERVICE_CONFLICT_MANAGEMENT);

   _serviceObjectManager.setEnabled(true);
   _serviceObjectManager.disableTimer(); // we will not track any timeout
   _serviceObjectManager.setName("WblObjectManager");
   _serviceObjectManager.setInterface((BTSCommonEnumClass)BTS_WBL_DBUS_SERVICE_OBJECT_MANAGER_SYSTEM);

   _serviceGroupGeneral.addServiceStatus(&_serviceLastIntendedMode);
   _serviceGroupGeneral.addServiceStatus(&_serviceConflictManagement);
   _serviceGroupGeneral.addServiceStatus(&_serviceObjectManager);
}

WblStartupWbl::~WblStartupWbl()
{
   _callback = 0;
   _controlIf = 0;
}

void WblStartupWbl::setCallback(IN IWblStartupCallback* callback)
{
   _callback = callback;

   FW_NORMAL_ASSERT(0 != _callback);
}

void WblStartupWbl::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_NORMAL_ASSERT(0 != _controlIf);
}

IWblStartupRequest* WblStartupWbl::getRequestIf(void)
{
   return this;
}

void WblStartupWbl::addInterfaces(OUT ::std::vector< BTSDbusInterfaceItem >& dbusInterfaces)
{
   FW_IF_NULL_PTR_RETURN(_controlIf);

   ::ccdbusif::wbl::WblDbusParser parser;

   _serviceLastIntendedMode.setIpc2BtsSendIf(_controlIf);
   _serviceLastIntendedMode.addServiceInfo(parser.getObjectPath(::ccdbusif::wbl::IF_LAST_INTENDED_MODE));
   _serviceLastIntendedMode.setAvailability(BTS_DBUS_SERVICE_WAITING);

   _serviceConflictManagement.setIpc2BtsSendIf(_controlIf);
   _serviceConflictManagement.addServiceInfo(parser.getObjectPath(::ccdbusif::wbl::IF_CONFLICT_MANAGEMENT));
   _serviceConflictManagement.setAvailability(BTS_DBUS_SERVICE_WAITING);

   _serviceObjectManager.setIpc2BtsSendIf(_controlIf);
   _serviceObjectManager.addServiceInfo(parser.getRootObjectPath());
   _serviceObjectManager.setAvailability(BTS_DBUS_SERVICE_WAITING);

   BTSDbusInterfaceItem dbusItem;
   dbusItem.dbusInterface = _serviceLastIntendedMode.getInterface();
   dbusInterfaces.push_back(dbusItem);
   dbusItem.dbusInterface = _serviceConflictManagement.getInterface();
   dbusInterfaces.push_back(dbusItem);
   dbusItem.dbusInterface = (BTSCommonEnumClass)BTS_WBL_DBUS_SERVICE_WIFI_SETUP; // just to register the callback instance
   dbusInterfaces.push_back(dbusItem);
   dbusItem.dbusInterface = _serviceObjectManager.getInterface();
   dbusItem.busType = (BTSCommonEnumClass)parser.getBusType(::ccdbusif::wbl::IF_LAST_INTENDED_MODE);
   dbusItem.busName = parser.getBusName(::ccdbusif::wbl::IF_LAST_INTENDED_MODE);
   dbusItem.objPath = parser.getRootObjectPath();
   dbusInterfaces.push_back(dbusItem);

   _serviceGroupGeneral.checkAvailability();

   // message with availability status WAITING will follow
}

void WblStartupWbl::updateServiceAvailability(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSWblDbusServiceInterface interface, IN const BTSDbusServiceAvailability availabilityEvent)
{
   FW_IF_NULL_PTR_RETURN(_callback);

   bool checkGroupAvailability(false);

   switch(interface)
   {
      case BTS_WBL_DBUS_SERVICE_LAST_INTENDED_MODE:
         _serviceLastIntendedMode.setAvailability(availabilityEvent);
         checkGroupAvailability = true;
         break;
      case BTS_WBL_DBUS_SERVICE_CONFLICT_MANAGEMENT:
         _serviceConflictManagement.setAvailability(availabilityEvent);
         checkGroupAvailability = true;
         break;
      case BTS_WBL_DBUS_SERVICE_WIFI_SETUP:
         // ignore because created during runtime
         break;
      case BTS_WBL_DBUS_SERVICE_OBJECT_MANAGER_SYSTEM:
         _serviceObjectManager.setAvailability(availabilityEvent);
         checkGroupAvailability = true;
         break;
      case BTS_WBL_DBUS_SERVICE_LAST:
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }

   if(true == checkGroupAvailability)
   {
      _serviceGroupGeneral.checkAvailability();
      _callback->updateServiceAvailability(bts2IpcMsgList, bts2AppMsgList, messageItem, _serviceGroupGeneral.getAvailability());
   }
}

} //wbl
} //btstackif
