/**
 * @file IWblWiFiSetupStatusWbl.h
 *
 * @par SW-Component
 * State machine for WBL WiFi setup status
 *
 * @brief Interface definition for WBL WiFi setup status state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the WBL interface for WiFi setup status state machine.
 */

#ifndef _I_WBL_WIFI_SETUP_STATUS_WBL_H_
#define _I_WBL_WIFI_SETUP_STATUS_WBL_H_

#include "WblStackTypes.h"

namespace btstackif {

// class forward declarations
class IWblWiFiSetupStatusRequest;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

namespace wbl {

// class forward declarations
class IWblObjectPathManagerWbl;

/**
 * Interface definition.
 */
class IWblWiFiSetupStatusWbl
{
public:
   /**
    * Destructor.
    */
   virtual ~IWblWiFiSetupStatusWbl() {}

   /**
    * Get request interface.
    *
    * @return = request interface
    */
   virtual IWblWiFiSetupStatusRequest* getRequestIf(void) = 0;

   /**
    * Set object path manager interface.
    *
    * @param[in] manager: object path manager interface
    */
   virtual void setObjectPathManagerIf(IN IWblObjectPathManagerWbl* manager) = 0;

   /**
    * Update service availability.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] interface: service interface
    * @param[in] availabilityEvent: availability event
    * @param[in] objPath: object path
    */
   virtual void updateServiceAvailability(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSWblDbusServiceInterface interface, IN const BTSDbusServiceAvailability availabilityEvent, IN const BTSObjectPath& objPath) = 0;

   /**
    * Handle result for GetManagedObjects.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] result: result
    */
   virtual void handleGetManagedObjectsResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result) = 0;

   /**
    * WiFi setup entry added.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] objPath: object path
    */
   virtual void wifiSetupEntryAdded(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& objPath) = 0;

   /**
    * WiFi setup entry removed.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] objPath: object path
    */
   virtual void wifiSetupEntryRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& objPath) = 0;

   /**
    * Update WiFi mode.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] result: result
    * @param[in] objPath: object path
    * @param[in] mode: mode
    */
   virtual void updateWifiMode(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const BTSObjectPath& objPath, IN const BTSWblWifiMode mode) = 0;

   /**
    * Update WiFi power state.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] result: result
    * @param[in] objPath: object path
    * @param[in] powerState: power state
    */
   virtual void updateWifiPowerState(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const BTSObjectPath& objPath, IN const BTSWblWifiPowerState powerState) = 0;

   /**
    * Update WiFi frequency.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] result: result
    * @param[in] objPath: object path
    * @param[in] frequency: frequency
    */
   virtual void updateWifiFrequency(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const BTSObjectPath& objPath, IN const BTSWblWifiFrequency frequency) = 0;

   /**
    * Update access point config type.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] result: result
    * @param[in] objPath: object path
    * @param[in] type: type
    */
   virtual void updateAccessPointConfigType(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const BTSObjectPath& objPath, IN const BTSWblAccessPointConfigType type) = 0;

   /**
    * Update supported restrictions.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] result: result
    * @param[in] objPath: object path
    * @param[in] supportedRestrictions: supported restrictions
    */
   virtual void updateSupportedRestrictions(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const BTSObjectPath& objPath, IN const BTSWblSupportedRestrictionList& supportedRestrictions) = 0;

   /**
    * Update WiFi connection status (healthiness indicator).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] result: result
    * @param[in] objPath: object path
    * @param[in] connected: connected flag
    * @param[in] healthIndicator: healthiness indicator
    */
   virtual void updateWifiConnectionStatus(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const BTSObjectPath& objPath, IN const bool connected, IN const BTSWblHealthinessIndicator healthIndicator) = 0;
};

} //wbl
} //btstackif

#endif //_I_WBL_WIFI_SETUP_STATUS_WBL_H_
