/**
 * @file WblOrgFreedesktopObjectManagerCallbackIf.cpp
 *
 * @par SW-Component
 * BtStackIf
 *
 * @brief OrgFreedesktop DBUS Callback for ObjectManager.
 *
 * @copyright (C) 2016 - 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details OrgFreedesktop DBUS Callback implementation for ObjectManager.
 */

#include "WblOrgFreedesktopObjectManagerCallbackIf.h"
#include "Ipc2Bts_MessageWrapper_WBL.h"
#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"
#include "cc_dbus_if/WblDbusParser.h"
#include "cc_dbus_if/IWblDbusWifiSetupSendRequestIf.h"
#include "IWblWifiSetupCallbackIf.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_DBUS
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/WblOrgFreedesktopObjectManagerCallbackIf.cpp.trc.h"
#endif
#endif

using namespace ::objmgr::org::freedesktop::DBus::ObjectManager;

namespace btstackif {
namespace wbl {

WblOrgFreedesktopObjectManagerCallbackIf::WblOrgFreedesktopObjectManagerCallbackIf(IDbusRecHandler* recHandler) :
DbusCallbackIf(recHandler),
_callbackId(0),
_wifiSetupIf(0),
_wifiSetupCb(0),
_objPathList()
{
}

WblOrgFreedesktopObjectManagerCallbackIf::~WblOrgFreedesktopObjectManagerCallbackIf()
{
   _wifiSetupIf = 0;
   _wifiSetupCb = 0;
}

// "ServiceAvailableIF" implementation --- start
void WblOrgFreedesktopObjectManagerCallbackIf::onAvailableCb(const bool available, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/)
{
   BTSDbusServiceAvailability availabilityEvent(BTS_DBUS_SERVICE_NOT_AVAILABLE);

   if(true == available)
   {
      availabilityEvent = BTS_DBUS_SERVICE_AVAILABLE;
   }

   Ipc2Bts_ServiceAvailabilityWbl* msg = ptrNew_Ipc2Bts_ServiceAvailabilityWbl();

   if(0 != msg)
   {
      msg->setInterface(BTS_WBL_DBUS_SERVICE_OBJECT_MANAGER_SYSTEM);
      msg->setAvailabilityEvent(availabilityEvent);
      msg->setObjPath(objPath);
      msg->setBusName(busName);
      msg->setBusType((BTSCommonEnumClass)busType);
   }

   onSignal(msg, ::ccdbusif::DEFAULT_ACT, true);
}

void WblOrgFreedesktopObjectManagerCallbackIf::onUnavailableCb(const bool available, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/)
{
   BTSDbusServiceAvailability availabilityEvent(BTS_DBUS_SERVICE_NOT_AVAILABLE);

   if(true == available)
   {
      availabilityEvent = BTS_DBUS_SERVICE_AVAILABLE;
   }

   Ipc2Bts_ServiceAvailabilityWbl* msg = ptrNew_Ipc2Bts_ServiceAvailabilityWbl();

   if(0 != msg)
   {
      msg->setInterface(BTS_WBL_DBUS_SERVICE_OBJECT_MANAGER_SYSTEM);
      msg->setAvailabilityEvent(availabilityEvent);
      msg->setObjPath(objPath);
      msg->setBusName(busName);
      msg->setBusType((BTSCommonEnumClass)busType);
   }

   onSignal(msg, ::ccdbusif::DEFAULT_ACT, true);
}
// "ServiceAvailableIF" implementation --- end

// /objmgr/org/freedesktop/DBus/ObjectManagerProxy implementation --- start
void WblOrgFreedesktopObjectManagerCallbackIf::onGetManagedObjectsErrorCb(const ::boost::shared_ptr< GetManagedObjectsError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(interfaceName);

   Ipc2Bts_GetManagedObjectsWbl* msg = ptrNew_Ipc2Bts_GetManagedObjectsWbl();
   if(0 != msg)
   {
      msg->setObjPath(objPath);
      msg->setBusName(busName);
      msg->setBusType((BTSCommonEnumClass)busType);
   }
   onError(msg, error);
}

void WblOrgFreedesktopObjectManagerCallbackIf::onGetManagedObjectsResponseCb(const ::boost::shared_ptr< GetManagedObjectsResponse >& response, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   // HINT: proxy object delivers following data:
   //    objPath
   //    busName
   //    busType
   //    interfaceName

   ::std::map< ::std::string, ::std::map< ::std::string, ::std::map< ::std::string, ::asf::dbus::DBusVariant > > >& allObjects = response->getObjectsMutable();
   for(::std::map< ::std::string, ::std::map< ::std::string, ::std::map< ::std::string, ::asf::dbus::DBusVariant > > >::iterator it = allObjects.begin(); it != allObjects.end(); ++it)
   {
      // it->first: object path
      // it->second: interfaces and properties (same as onInterfacesAddedSignalCb)
      parseAddedInterface(it->first, it->second, objPath, busName, busType, interfaceName);
   }

   Ipc2Bts_GetManagedObjectsWbl* msg = ptrNew_Ipc2Bts_GetManagedObjectsWbl();
   if(0 != msg)
   {
      msg->setObjPath(objPath);
      msg->setBusName(busName);
      msg->setBusType((BTSCommonEnumClass)busType);
   }
   onResponse(msg, response->getAct());
}

void WblOrgFreedesktopObjectManagerCallbackIf::onInterfacesAddedErrorCb(const ::boost::shared_ptr< InterfacesAddedError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   // not used
   (void)(error);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
}

void WblOrgFreedesktopObjectManagerCallbackIf::onInterfacesAddedSignalCb(const ::boost::shared_ptr< InterfacesAddedSignal >& signal, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   // HINT: proxy object delivers following data:
   //    objPath
   //    busName
   //    busType
   //    interfaceName

   ::std::map< ::std::string, ::std::map< ::std::string, ::asf::dbus::DBusVariant > >& inInterfaces = signal->getInterfacesMutable();
   parseAddedInterface(signal->getObject(), inInterfaces, objPath, busName, busType, interfaceName);
}

void WblOrgFreedesktopObjectManagerCallbackIf::onInterfacesRemovedErrorCb(const ::boost::shared_ptr< InterfacesRemovedError >& error, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   // not used
   (void)(error);
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);
}

void WblOrgFreedesktopObjectManagerCallbackIf::onInterfacesRemovedSignalCb(const ::boost::shared_ptr< InterfacesRemovedSignal >& signal, const ::std::string& objPath /*= ::std::string()*/, const ::std::string& busName /*= ::std::string()*/, const ::ccdbusif::DbusBusType busType /*= ::ccdbusif::BUS_TYPE_SYSTEM*/, const ::std::string& interfaceName /*= ::std::string()*/)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);

   // HINT: proxy object delivers following data:
   //    objPath
   //    busName
   //    busType
   //    interfaceName

   ::ccdbusif::wbl::WblDbusParser parser;
   parser.setTraces(true);
   const ::std::vector< ::std::string >& removedInterfaces = signal->getInterfaces();
   // parser.parseInterfacesRemoved(signal->getObject(), removedInterfaces, busType, busName, objPath, interfaceName);

   for(size_t i = 0; i < removedInterfaces.size(); ++i)
   {
      if(parser.getInterfaceName(::ccdbusif::wbl::IF_WIFI_SETUP) == removedInterfaces[i])
      {
         // destroy proxy
         if((0 != _wifiSetupIf) && (0 != _wifiSetupCb))
         {
            _wifiSetupIf->destroyProxyIf(_wifiSetupCb->getCallbackId(), true, signal->getObject());
         }

         // send removed mapping
         Ipc2Bts_DelWifiSetupObjectPath* msg = ptrNew_Ipc2Bts_DelWifiSetupObjectPath();
         if(0 != msg)
         {
            msg->setObjPath(signal->getObject());
            msg->setObjectId(signal->getObject());
         }
         onSignal(msg, signal->getAct());

         // remove object path from internal list
         removeObjPath(signal->getObject());
      }
   }
}
// /objmgr/org/freedesktop/DBus/ObjectManagerProxy implementation --- end

void WblOrgFreedesktopObjectManagerCallbackIf::parseAddedInterface(const ::std::string& addedObjPath, ::std::map< ::std::string, ::std::map< ::std::string, ::asf::dbus::DBusVariant > >& addedInterfaces, const ::std::string& objPath, const ::std::string& busName, const ::ccdbusif::DbusBusType busType, const ::std::string& interfaceName)
{
   (void)(objPath);
   (void)(busName);
   (void)(busType);
   (void)(interfaceName);

   ::ccdbusif::wbl::WblDbusParser parser;
   parser.setTraces(true);
   // parser.parseInterfacesAdded(addedObjPath, addedInterfaces, busType, busName, objPath, interfaceName);

   for(::std::map< ::std::string, ::std::map< ::std::string, ::asf::dbus::DBusVariant > >::iterator it = addedInterfaces.begin(); it != addedInterfaces.end(); ++it)
   {
      if(parser.getInterfaceName(::ccdbusif::wbl::IF_WIFI_SETUP) == it->first)
      {
         // check if object path is in internal list
         if(false == isObjPathAvailable(addedObjPath))
         {
            // send added mapping
            Ipc2Bts_AddWifiSetupObjectPath* msg = ptrNew_Ipc2Bts_AddWifiSetupObjectPath();
            if(0 != msg)
            {
               msg->setObjPath(addedObjPath);
            }
            onSignal(msg, ::ccdbusif::DEFAULT_ACT, true);

            // create proxy
            if((0 != _wifiSetupIf) && (0 != _wifiSetupCb))
            {
               _wifiSetupIf->createProxyIf(_wifiSetupCb->getCallbackId(), true, addedObjPath);
            }

            // add object path to internal list
            addObjPath(addedObjPath);
         }
      }
   }
}

void WblOrgFreedesktopObjectManagerCallbackIf::removeObjPath(const ::std::string& objPath)
{
   ::std::map< ::std::string, bool >::iterator it = _objPathList.find(objPath);
   if(_objPathList.end() != it)
   {
      _objPathList.erase(it);
   }
}

bool WblOrgFreedesktopObjectManagerCallbackIf::isObjPathAvailable(const ::std::string& objPath) const
{
   return (_objPathList.end() != _objPathList.find(objPath));
}

void WblOrgFreedesktopObjectManagerCallbackIf::addObjPath(const ::std::string& objPath)
{
   _objPathList[objPath] = true;
}

} //wbl
} //btstackif
