/**
 * @file WblDbusIfHandler.cpp
 *
 * @par SW-Component
 * IPC
 *
 * @brief DBUS WBL handling.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of DBUS WBL handling.
 */

#include "cc_dbus_if/ICcDbusIfController.h"
#include "cc_dbus_if/ICcDbusIfControllerWbl.h"
#include "cc_dbus_if/ICcDbusIfControllerObjectManager.h"

// send request interfaces
#include "cc_dbus_if/IWblDbusLastIntendedModeSendRequestIf.h"
#include "cc_dbus_if/IWblDbusConflictManagementSendRequestIf.h"
#include "cc_dbus_if/IWblDbusWifiSetupSendRequestIf.h"
#include "cc_dbus_if/IOrgFreedesktopDbusObjectManagerSendRequestIf.h"

// callback interfaces
#include "IWblLastIntendedModeCallbackIf.h"
#include "IWblConflictManagementCallbackIf.h"
#include "IWblWifiSetupCallbackIf.h"
#include "IWblOrgFreedesktopObjectManagerCallbackIf.h"

// factory classes
#include "WblLastIntendedModeFactory.h"
#include "WblConflictManagementFactory.h"
#include "WblWifiSetupFactory.h"
#include "WblOrgFreedesktopObjectManagerFactory.h"

#include "WblDbusIfHandler.h"
#include "IDbusBase.h"
#include "Bts2Ipc_MessageWrapper_WBL.h"
#include "Ipc2Bts_MessageWrapper_WBL.h"
#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_DBUS
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/WblDbusIfHandler.cpp.trc.h"
#endif
#endif

namespace btstackif {
namespace wbl {

WblDbusIfHandler::WblDbusIfHandler(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent) :
_component(component),
_stackInterface(stackInterface),
_subComponent(subComponent),
_getUpdateTimeout(500U),
_dbusBaseIf(0),
_dbusRecHandler(0),
_lastIntendedModeIf(0),
_conflictManagementIf(0),
_wifiSetupIf(0),
_objMgrIf(0),
_lastIntendedModeCb(0),
_conflictManagementCb(0),
_wifiSetupCb(0),
_objMgrCb(0)
{
}

WblDbusIfHandler::~WblDbusIfHandler()
{
   _dbusBaseIf = 0;
   _dbusRecHandler = 0;

   _lastIntendedModeIf = 0;
   _conflictManagementIf = 0;
   _wifiSetupIf = 0;
   _objMgrIf = 0;

    WblLastIntendedModeFactory::getInstance().destroyCb();
    _lastIntendedModeCb = 0;
    WblConflictManagementFactory::getInstance().destroyCb();
    _conflictManagementCb = 0;
    WblWifiSetupFactory::getInstance().destroyCb();
    _wifiSetupCb = 0;
    WblOrgFreedesktopObjectManagerFactory::getInstance().destroyCb();
    _objMgrCb = 0;
}

void WblDbusIfHandler::setDbusBaseIf(IN IDbusBase* dbusBase)
{
   _dbusBaseIf = dbusBase;

   FW_NORMAL_ASSERT(0 != _dbusBaseIf);
}

void WblDbusIfHandler::setDbusRecHandlerIf(IN IDbusRecHandler* dbusRecHandler)
{
   _dbusRecHandler = dbusRecHandler;

   FW_NORMAL_ASSERT(0 != _dbusRecHandler);
}

void WblDbusIfHandler::enable(IN const bool enable)
{
   (void)(enable);
   // ignore
}

void WblDbusIfHandler::initialize(void)
{
   // ignore
}

void WblDbusIfHandler::destroy(void)
{
   // ignore
}

void WblDbusIfHandler::setCcDbusIfControllerIf(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode,
         IN ::ccdbusif::ICcDbusIfController* controller, IN const ::std::vector< BTSDbusInterfaceItem >& dbusInterfaces, IN const BTSLocalConfigurationContainer& configuration)
{
   (void)(userMode);

   ETG_TRACE_USR1((" setCcDbusIfControllerIf(): [enter]"));

   FW_IF_NULL_PTR_RETURN(_dbusBaseIf);
   FW_IF_NULL_PTR_RETURN(_dbusRecHandler);
   FW_IF_NULL_PTR_RETURN(controller);

   FW_NORMAL_ASSERT(component == _component); // already set
   FW_NORMAL_ASSERT(stackInterface == _stackInterface); // already set
   FW_NORMAL_ASSERT(subComponent == _subComponent); // already set
   FW_NORMAL_ASSERT(true == configuration.wblActive); // configuration for WBL part is needed

   ::ccdbusif::ICcDbusIfControllerWbl* wblIf = controller->getCcDbusIfControllerWbl();
   FW_IF_NULL_PTR_RETURN(wblIf);
   ::ccdbusif::ICcDbusIfControllerObjectManager* objectManagerMainIf = controller->getCcDbusIfControllerObjectManager();
   FW_IF_NULL_PTR_RETURN(objectManagerMainIf);

   // get send interface
   if(0 == _lastIntendedModeIf)
   {
      _lastIntendedModeIf = wblIf->getWblLastIntendedModeProxy();
   }
   FW_IF_NULL_PTR_RETURN(_lastIntendedModeIf);

   if(0 == _conflictManagementIf)
   {
      _conflictManagementIf = wblIf->getWblConflictManagementProxy();
   }
   FW_IF_NULL_PTR_RETURN(_conflictManagementIf);

   if(0 == _wifiSetupIf)
   {
      _wifiSetupIf = wblIf->getWblWifiSetupProxy();
   }
   FW_IF_NULL_PTR_RETURN(_wifiSetupIf);

   if(0 == _objMgrIf)
   {
      _objMgrIf = objectManagerMainIf->getObjectManagerProxy();
   }
   FW_IF_NULL_PTR_RETURN(_objMgrIf);

   // get callback interface
   if(0 == _lastIntendedModeCb)
   {
      _lastIntendedModeCb = WblLastIntendedModeFactory::getInstance().getCb(_dbusRecHandler);
   }
   FW_IF_NULL_PTR_RETURN(_lastIntendedModeCb);

   if(0 == _conflictManagementCb)
   {
      _conflictManagementCb = WblConflictManagementFactory::getInstance().getCb(_dbusRecHandler);
   }
   FW_IF_NULL_PTR_RETURN(_conflictManagementCb);

   if(0 == _wifiSetupCb)
   {
      _wifiSetupCb = WblWifiSetupFactory::getInstance().getCb(_dbusRecHandler);
   }
   FW_IF_NULL_PTR_RETURN(_wifiSetupCb);

   if(0 == _objMgrCb)
   {
      _objMgrCb = WblOrgFreedesktopObjectManagerFactory::getInstance().getCb(_dbusRecHandler);
   }
   FW_IF_NULL_PTR_RETURN(_objMgrCb);

   // get callback id
   if(0 == _lastIntendedModeCb->getCallbackId())
   {
      _lastIntendedModeCb->setCallbackId(controller->getCallbackId());
   }
   FW_IF_NULL_PTR_RETURN(_lastIntendedModeCb->getCallbackId());

   if(0 == _conflictManagementCb->getCallbackId())
   {
      _conflictManagementCb->setCallbackId(controller->getCallbackId());
   }
   FW_IF_NULL_PTR_RETURN(_conflictManagementCb->getCallbackId());

   if(0 == _wifiSetupCb->getCallbackId())
   {
      _wifiSetupCb->setCallbackId(controller->getCallbackId());
   }
   FW_IF_NULL_PTR_RETURN(_wifiSetupCb->getCallbackId());

   if(0 == _objMgrCb->getCallbackId())
   {
      _objMgrCb->setCallbackId(controller->getCallbackId());
   }
   FW_IF_NULL_PTR_RETURN(_objMgrCb->getCallbackId());

   // set callback handler (if no send interface and no callback handler were created nothing will happen during next step)
   for(size_t i = 0; i < dbusInterfaces.size(); i++)
   {
      const BTSWblDbusServiceInterface interface((BTSWblDbusServiceInterface)dbusInterfaces[i].dbusInterface);

      switch(interface)
      {
         case BTS_WBL_DBUS_SERVICE_LAST_INTENDED_MODE:
            // if((0 != _lastIntendedModeIf) && (0 != _lastIntendedModeCb))
            {
               _dbusBaseIf->createDbusServiceAvailabilityMessage(dbusInterfaces[i].dbusInterface, BTS_DBUS_SERVICE_WAITING);
               _lastIntendedModeIf->setCallbackIf(_lastIntendedModeCb, true, _lastIntendedModeCb->getCallbackId());
               _lastIntendedModeIf->createProxyIf(_lastIntendedModeCb->getCallbackId());
            }
            break;
         case BTS_WBL_DBUS_SERVICE_CONFLICT_MANAGEMENT:
            // if((0 != _conflictManagementIf) && (0 != _conflictManagementCb))
            {
               _dbusBaseIf->createDbusServiceAvailabilityMessage(dbusInterfaces[i].dbusInterface, BTS_DBUS_SERVICE_WAITING);
               _conflictManagementIf->setCallbackIf(_conflictManagementCb, true, _conflictManagementCb->getCallbackId());
               _conflictManagementIf->createProxyIf(_conflictManagementCb->getCallbackId());
            }
            break;
         case BTS_WBL_DBUS_SERVICE_WIFI_SETUP:
            // if((0 != _wifiSetupIf) && (0 != _wifiSetupCb))
            {
               // created during runtime - do not create service availability message
               _wifiSetupIf->setCallbackIf(_wifiSetupCb, true, _wifiSetupCb->getCallbackId());
               // created during runtime - do not create proxy
            }
            break;
         case BTS_WBL_DBUS_SERVICE_OBJECT_MANAGER_SYSTEM:
            // if((0 != _objMgrIf) && (0 != _objMgrCb))
            {
               _dbusBaseIf->createDbusServiceAvailabilityMessage(dbusInterfaces[i].dbusInterface, BTS_DBUS_SERVICE_WAITING);
               _objMgrIf->setCallbackIf(_objMgrCb, true, _objMgrCb->getCallbackId(), dbusInterfaces[i].objPath, dbusInterfaces[i].busName, (::ccdbusif::DbusBusType)dbusInterfaces[i].busType);
               _objMgrIf->createProxyIf(_objMgrCb->getCallbackId(), false, dbusInterfaces[i].objPath, dbusInterfaces[i].busName, (::ccdbusif::DbusBusType)dbusInterfaces[i].busType);
            }
            break;
         case BTS_WBL_DBUS_SERVICE_LAST:
         default:
            FW_NORMAL_ASSERT_ALWAYS();
            break;
      }
   }

   // set references
   _objMgrCb->setWifiSetupIf(_wifiSetupIf);
   _objMgrCb->setWifiSetupCb(_wifiSetupCb);

   ETG_TRACE_USR1((" setCcDbusIfControllerIf(): [exit]"));
}

void WblDbusIfHandler::resetCcDbusIfControllerIf(void)
{
   ETG_TRACE_USR1((" resetCcDbusIfControllerIf(): [enter]"));

   _lastIntendedModeIf = 0;
   _conflictManagementIf = 0;
   _wifiSetupIf = 0;
   _objMgrIf = 0;

   // do not delete callback handler

   ETG_TRACE_USR1((" resetCcDbusIfControllerIf(): [exit]"));
}

bool WblDbusIfHandler::setSimulationTestCommand(IN const char* testCommand, IN const unsigned int testData)
{
   (void)(testCommand);
   (void)(testData);
   bool handled(false);
   return handled;
}

bool WblDbusIfHandler::setSimulationTestCommand(IN const char* testCommand, IN const unsigned char* testData)
{
   (void)(testCommand);
   (void)(testData);
   bool handled(false);
   return handled;
}

bool WblDbusIfHandler::setSimulationTestCommand(IN const char* testCommand, IN const Ipc2Bts_BaseMessage& testData)
{
   (void)(testCommand);
   (void)(testData);
   bool handled(false);
   return handled;
}

bool WblDbusIfHandler::sendMessage(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_BaseMessage* message)
{
   FW_IF_NULL_PTR_RETURN_FALSE(message);
   Bts2Ipc_BaseMessage& messageRef(*message);

   bool handled(true);
   const BTSWblBts2IpcOpcode opCode((BTSWblBts2IpcOpcode)messageRef.getOpCode());

   switch(opCode)
   {
      case Bts2IpcOC_GetManagedObjectsWbl:
         handleBts2IpcGetManagedObjectsWbl(deleteMsg, sendErrorMsg, static_cast< Bts2Ipc_GetManagedObjectsWbl& >(messageRef));
         break;
      case Bts2IpcOC_GetIntrospectionWbl:
         handleBts2IpcGetIntrospectionWbl(deleteMsg, sendErrorMsg, static_cast< Bts2Ipc_GetIntrospectionWbl& >(messageRef));
         break;
      case Bts2IpcOC_GetLastIntendedMode:
         handleBts2IpcGetLastIntendedMode(deleteMsg, sendErrorMsg, static_cast< Bts2Ipc_GetLastIntendedMode& >(messageRef));
         break;
      case Bts2IpcOC_GetSupportedFrequencies:
         handleBts2IpcGetSupportedFrequencies(deleteMsg, sendErrorMsg, static_cast< Bts2Ipc_GetSupportedFrequencies& >(messageRef));
         break;
      case Bts2IpcOC_GetWifiMode:
         handleBts2IpcGetWifiMode(deleteMsg, sendErrorMsg, static_cast< Bts2Ipc_GetWifiMode& >(messageRef));
         break;
      case Bts2IpcOC_GetWifiPowerState:
         handleBts2IpcGetWifiPowerState(deleteMsg, sendErrorMsg, static_cast< Bts2Ipc_GetWifiPowerState& >(messageRef));
         break;
      case Bts2IpcOC_GetWifiFrequency:
         handleBts2IpcGetWifiFrequency(deleteMsg, sendErrorMsg, static_cast< Bts2Ipc_GetWifiFrequency& >(messageRef));
         break;
      case Bts2IpcOC_GetAccessPointConfigType:
         handleBts2IpcGetAccessPointConfigType(deleteMsg, sendErrorMsg, static_cast< Bts2Ipc_GetAccessPointConfigType& >(messageRef));
         break;
      case Bts2IpcOC_GetWifiConnectionStatus:
         handleBts2IpcGetWifiConnectionStatus(deleteMsg, sendErrorMsg, static_cast< Bts2Ipc_GetWifiConnectionStatus& >(messageRef));
         break;
      default:
         handled = false;
         break;
   }

   return handled;
}

bool WblDbusIfHandler::createErrorMessage(OUT Ipc2Bts_BaseMessage** ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage, IN const BTSIpcCommonErrorCode errorCode) const
{
   FW_IF_NULL_PTR_RETURN_FALSE(ipc2BtsMessage);
   FW_IF_NULL_PTR_RETURN_FALSE(bts2IpcMessage);
   const Bts2Ipc_BaseMessage& messageRef(*bts2IpcMessage);

   bool handled(true);
   const BTSWblBts2IpcOpcode opCode((BTSWblBts2IpcOpcode)messageRef.getOpCode());

   switch(opCode)
   {
      case Bts2IpcOC_GetManagedObjectsWbl:
         *ipc2BtsMessage = ptrNew_Ipc2Bts_GetManagedObjectsWbl();
         break;
      case Bts2IpcOC_GetIntrospectionWbl:
         // no Ipc2Bts message available
         break;
      case Bts2IpcOC_GetLastIntendedMode:
         *ipc2BtsMessage = ptrNew_Ipc2Bts_LastIntendedModeUpdate();
         break;
      case Bts2IpcOC_GetSupportedFrequencies:
         *ipc2BtsMessage = ptrNew_Ipc2Bts_SupportedFrequenciesUpdate();
         break;
      case Bts2IpcOC_GetWifiMode:
         *ipc2BtsMessage = ptrNew_Ipc2Bts_WifiModeUpdate();
         break;
      case Bts2IpcOC_GetWifiPowerState:
         *ipc2BtsMessage = ptrNew_Ipc2Bts_WifiPowerStateUpdate();
         break;
      case Bts2IpcOC_GetWifiFrequency:
         *ipc2BtsMessage = ptrNew_Ipc2Bts_WifiFrequencyUpdate();
         break;
      case Bts2IpcOC_GetAccessPointConfigType:
         *ipc2BtsMessage = ptrNew_Ipc2Bts_AccessPointConfigTypeUpdate();
         break;
      case Bts2IpcOC_GetWifiConnectionStatus:
         *ipc2BtsMessage = ptrNew_Ipc2Bts_WifiConnectionStatusUpdate();
         break;
      default:
         handled = false;
         break;
   }

   if(0 != (*ipc2BtsMessage))
   {
      (*ipc2BtsMessage)->setIpcCommonErrorCode(errorCode);
   }

   return handled;
}

bool WblDbusIfHandler::transferMessageData(OUT Ipc2Bts_BaseMessage* ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage) const
{
   FW_IF_NULL_PTR_RETURN_FALSE(ipc2BtsMessage);
   FW_IF_NULL_PTR_RETURN_FALSE(bts2IpcMessage);
   Ipc2Bts_BaseMessage& ipc2BtsMessageRef(*ipc2BtsMessage);
   const Bts2Ipc_BaseMessage& bts2IpcMessageRef(*bts2IpcMessage);

   bool handled(true);
   const BTSWblBts2IpcOpcode opCode((BTSWblBts2IpcOpcode)bts2IpcMessageRef.getOpCode());

   switch(opCode)
   {
      case Bts2IpcOC_GetManagedObjectsWbl:
      {
         // nothing to transfer
         break;
      }
      case Bts2IpcOC_GetIntrospectionWbl:
      {
         // no Ipc2Bts message available
         break;
      }
      case Bts2IpcOC_GetLastIntendedMode:
      {
         // nothing to transfer
         break;
      }
      case Bts2IpcOC_GetSupportedFrequencies:
      {
         // copy object path
         Ipc2Bts_SupportedFrequenciesUpdate& outMsg = static_cast< Ipc2Bts_SupportedFrequenciesUpdate& >(ipc2BtsMessageRef);
         const Bts2Ipc_GetSupportedFrequencies& inMsg = static_cast< const Bts2Ipc_GetSupportedFrequencies& >(bts2IpcMessageRef);
         outMsg.setObjPath(inMsg.getObjPath());
         break;
      }
      case Bts2IpcOC_GetWifiMode:
      {
         // copy object path
         Ipc2Bts_WifiModeUpdate& outMsg = static_cast< Ipc2Bts_WifiModeUpdate& >(ipc2BtsMessageRef);
         const Bts2Ipc_GetWifiMode& inMsg = static_cast< const Bts2Ipc_GetWifiMode& >(bts2IpcMessageRef);
         outMsg.setObjPath(inMsg.getObjPath());
         break;
      }
      case Bts2IpcOC_GetWifiPowerState:
      {
         // copy object path
         Ipc2Bts_WifiPowerStateUpdate& outMsg = static_cast< Ipc2Bts_WifiPowerStateUpdate& >(ipc2BtsMessageRef);
         const Bts2Ipc_GetWifiPowerState& inMsg = static_cast< const Bts2Ipc_GetWifiPowerState& >(bts2IpcMessageRef);
         outMsg.setObjPath(inMsg.getObjPath());
         break;
      }
      case Bts2IpcOC_GetWifiFrequency:
      {
         // copy object path
         Ipc2Bts_WifiFrequencyUpdate& outMsg = static_cast< Ipc2Bts_WifiFrequencyUpdate& >(ipc2BtsMessageRef);
         const Bts2Ipc_GetWifiFrequency& inMsg = static_cast< const Bts2Ipc_GetWifiFrequency& >(bts2IpcMessageRef);
         outMsg.setObjPath(inMsg.getObjPath());
         break;
      }
      case Bts2IpcOC_GetAccessPointConfigType:
      {
         // copy object path
         Ipc2Bts_AccessPointConfigTypeUpdate& outMsg = static_cast< Ipc2Bts_AccessPointConfigTypeUpdate& >(ipc2BtsMessageRef);
         const Bts2Ipc_GetAccessPointConfigType& inMsg = static_cast< const Bts2Ipc_GetAccessPointConfigType& >(bts2IpcMessageRef);
         outMsg.setObjPath(inMsg.getObjPath());
         break;
      }
      case Bts2IpcOC_GetWifiConnectionStatus:
      {
         // copy object path
         Ipc2Bts_WifiConnectionStatusUpdate& outMsg = static_cast< Ipc2Bts_WifiConnectionStatusUpdate& >(ipc2BtsMessageRef);
         const Bts2Ipc_GetWifiConnectionStatus& inMsg = static_cast< const Bts2Ipc_GetWifiConnectionStatus& >(bts2IpcMessageRef);
         outMsg.setObjPath(inMsg.getObjPath());
         break;
      }
      default:
      {
         handled = false;
         break;
      }
   }

   return handled;
}

void WblDbusIfHandler::handleBts2IpcGetManagedObjectsWbl(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_GetManagedObjectsWbl& message)
{
   if((0 == _objMgrIf) || (0 == _objMgrCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteMsg = true;
      sendErrorMsg = true;
   }
   else
   {
      const act_t token = _objMgrIf->sendGetManagedObjectsRequest(_objMgrCb->getCallbackId(), message.getObjPath(), message.getBusName(), (::ccdbusif::DbusBusType)message.getBusType());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteMsg = true;
         sendErrorMsg = true;
      }
      else
      {
         // sending success
         message.setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetManagedObjectsWbl(): Bts2Ipc 0x%08X success: token=%s", message.getMessageTraceId(), info.getInfo()));
      }
   }
}

void WblDbusIfHandler::handleBts2IpcGetIntrospectionWbl(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_GetIntrospectionWbl& message)
{
   (void)(message);
   FW_NORMAL_ASSERT_ALWAYS();
   // to be implemented
   deleteMsg = true;
   sendErrorMsg = false;
}

void WblDbusIfHandler::handleBts2IpcGetLastIntendedMode(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_GetLastIntendedMode& message)
{
   if((0 == _lastIntendedModeIf) || (0 == _lastIntendedModeCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteMsg = true;
      sendErrorMsg = true;
   }
   else
   {
      const act_t token = _lastIntendedModeIf->sendLastIntendedModeGet(_lastIntendedModeCb->getCallbackId());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteMsg = true;
         sendErrorMsg = true;
      }
      else
      {
         // sending success
         message.setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetLastIntendedMode(): Bts2Ipc 0x%08X success: token=%s", message.getMessageTraceId(), info.getInfo()));
         message.setTimeout(_getUpdateTimeout);
      }
   }
}

void WblDbusIfHandler::handleBts2IpcGetSupportedFrequencies(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_GetSupportedFrequencies& message)
{
   if((0 == _wifiSetupIf) || (0 == _wifiSetupCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteMsg = true;
      sendErrorMsg = true;
   }
   else
   {
      const act_t token = _wifiSetupIf->sendSupportedFrequenciesGet(_wifiSetupCb->getCallbackId(), message.getObjPath());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteMsg = true;
         sendErrorMsg = true;
      }
      else
      {
         // sending success
         message.setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetSupportedFrequencies(): Bts2Ipc 0x%08X success: token=%s", message.getMessageTraceId(), info.getInfo()));
         message.setTimeout(_getUpdateTimeout);
         message.setObjectId(message.getObjPath());
      }
   }
}

void WblDbusIfHandler::handleBts2IpcGetWifiMode(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_GetWifiMode& message)
{
   if((0 == _wifiSetupIf) || (0 == _wifiSetupCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteMsg = true;
      sendErrorMsg = true;
   }
   else
   {
      const act_t token = _wifiSetupIf->sendModeGet(_wifiSetupCb->getCallbackId(), message.getObjPath());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteMsg = true;
         sendErrorMsg = true;
      }
      else
      {
         // sending success
         message.setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetWifiMode(): Bts2Ipc 0x%08X success: token=%s", message.getMessageTraceId(), info.getInfo()));
         message.setTimeout(_getUpdateTimeout);
         message.setObjectId(message.getObjPath());
      }
   }
}

void WblDbusIfHandler::handleBts2IpcGetWifiPowerState(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_GetWifiPowerState& message)
{
   if((0 == _wifiSetupIf) || (0 == _wifiSetupCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteMsg = true;
      sendErrorMsg = true;
   }
   else
   {
      const act_t token = _wifiSetupIf->sendPowerStateGet(_wifiSetupCb->getCallbackId(), message.getObjPath());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteMsg = true;
         sendErrorMsg = true;
      }
      else
      {
         // sending success
         message.setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetWifiPowerState(): Bts2Ipc 0x%08X success: token=%s", message.getMessageTraceId(), info.getInfo()));
         message.setTimeout(_getUpdateTimeout);
         message.setObjectId(message.getObjPath());
      }
   }
}

void WblDbusIfHandler::handleBts2IpcGetWifiFrequency(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_GetWifiFrequency& message)
{
   if((0 == _wifiSetupIf) || (0 == _wifiSetupCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteMsg = true;
      sendErrorMsg = true;
   }
   else
   {
      const act_t token = _wifiSetupIf->sendFrequencyGet(_wifiSetupCb->getCallbackId(), message.getObjPath());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteMsg = true;
         sendErrorMsg = true;
      }
      else
      {
         // sending success
         message.setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetWifiFrequency(): Bts2Ipc 0x%08X success: token=%s", message.getMessageTraceId(), info.getInfo()));
         message.setTimeout(_getUpdateTimeout);
         message.setObjectId(message.getObjPath());
      }
   }
}

void WblDbusIfHandler::handleBts2IpcGetAccessPointConfigType(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_GetAccessPointConfigType& message)
{
   if((0 == _wifiSetupIf) || (0 == _wifiSetupCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteMsg = true;
      sendErrorMsg = true;
   }
   else
   {
      const act_t token = _wifiSetupIf->sendAPConfigGet(_wifiSetupCb->getCallbackId(), message.getObjPath());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
         FW_NORMAL_ASSERT_ALWAYS();
         deleteMsg = true;
         sendErrorMsg = true;
      }
      else
      {
         // sending success
         message.setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetAccessPointConfigType(): Bts2Ipc 0x%08X success: token=%s", message.getMessageTraceId(), info.getInfo()));
         message.setTimeout(_getUpdateTimeout);
         message.setObjectId(message.getObjPath());
      }
   }
}

void WblDbusIfHandler::handleBts2IpcGetWifiConnectionStatus(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_GetWifiConnectionStatus& message)
{
   if((0 == _wifiSetupIf) || (0 == _wifiSetupCb))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      deleteMsg = true;
      sendErrorMsg = true;
   }
   else
   {
      const act_t token = _wifiSetupIf->sendConnectionStatusGet(_wifiSetupCb->getCallbackId(), message.getObjPath());

      if(::ccdbusif::DEFAULT_ACT == token)
      {
         // sending failed (valid use case)
#ifdef VARIANT_S_FTR_ENABLE_CONNECTION_STATUS
         FW_NORMAL_ASSERT_ALWAYS();
#endif
         deleteMsg = true;
         sendErrorMsg = true;
      }
      else
      {
         // sending success
         message.setDbusToken(token);
         ::ccdbusif::ActInfo info(token);
         ETG_TRACE_USR3((" handleBts2IpcGetWifiConnectionStatus(): Bts2Ipc 0x%08X success: token=%s", message.getMessageTraceId(), info.getInfo()));
         message.setTimeout(_getUpdateTimeout);
         message.setObjectId(message.getObjPath());
      }
   }
}

} //wbl
} //btstackif
