/**
 * @file RssiGenivi.cpp
 *
 * @par SW-Component
 * State machine for RSSI
 *
 * @brief Implementation of Genivi RSSI state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of Genivi RSSI state machine.
 */

#include "RssiGenivi.h"
#include "IRssiCallback.h"
#include "IObjectPathManagerGenivi.h"
#include "FwAssert.h"
#include "Bts2Ipc_MessageWrapper_GEN.h"

namespace btstackif {
namespace genivi {

RssiGenivi::RssiGenivi() :
_callback(0),
_objectPathManagerIf(0)
{
}

RssiGenivi::~RssiGenivi()
{
   _callback = 0;
   _objectPathManagerIf = 0;
}

void RssiGenivi::reset(void)
{
}

void RssiGenivi::setCallback(IN IRssiCallback* callback)
{
   _callback = callback;

   FW_NORMAL_ASSERT(0 != _callback);
}

void RssiGenivi::getLinkQuality(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address)
{
   (void)(bts2AppMsgList);

   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   // all necessary checks are done before

   BTSObjectPath objPath;
   if(false == _objectPathManagerIf->getObjectPath4Address(objPath, address))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   createGetDeviceRssiMsg(bts2IpcMsgList, address, objPath);
}

void RssiGenivi::setObjectPathManagerIf(IN IObjectPathManagerGenivi* objectPathManager)
{
   _objectPathManagerIf = objectPathManager;

   FW_NORMAL_ASSERT(0 != _objectPathManagerIf);
}

void RssiGenivi::handleDeviceRssiUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSRssi rssi, IN const BTSIpcCommonErrorCode errorCode, IN const bool response)
{
   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   BTSBDAddress address;
   if(false == _objectPathManagerIf->getAddress4ObjectPath(address, device))
   {
      // should never happen
      return;
   }

   if(true == response)
   {
      bool success(false);
      BTSRssi sendRssi(defaultRssiValue);

      if(BTS_IPC_SUCCESS == errorCode)
      {
         success = true;
         sendRssi = rssi;
      }

      _callback->getLinkQualityResult(bts2IpcMsgList, bts2AppMsgList, messageItem, address, success, true, sendRssi, false, defaultLinkQualityValue);
   }
   else
   {
      // ignore update
   }
}

void RssiGenivi::createGetDeviceRssiMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSObjectPath& device) const
{
   Bts2Ipc_GetDeviceRssi* msg = ptrNew_Bts2Ipc_GetDeviceRssi();
   if(0 != msg)
   {
      msg->setDevice(device);
      msg->setBDAddress(address);

      bts2IpcMsgList.push_back(msg);
   }
}

} //genivi
} //btstackif
