/**
 * @file PairingGenivi.cpp
 *
 * @par SW-Component
 * State machine for pairing
 *
 * @brief Implementation of Genivi pairing state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of Genivi pairing state machine.
 */

#include "PairingGenivi.h"
#include "IPairingCallback.h"
#include "IBasicControl.h"
#include "IObjectPathManagerGenivi.h"
#include "FwAssert.h"
#include "Bts2Ipc_MessageWrapper_GEN.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"
#include "BtsUtils.h"
#include "EvolutionGeniviUtils.h"

#include "cc_dbus_if/EvolutionGeniviDbusParser.h"

namespace btstackif {
namespace genivi {

PairingGenivi::PairingGenivi() :
_callback(0),
_controlIf(0),
_objectPathManagerIf(0),
_tokenList(),
_pendingList(),
_testTriggerSimulateStackReset(false)
{
}

PairingGenivi::~PairingGenivi()
{
   _callback = 0;
   _controlIf = 0;
   _objectPathManagerIf = 0;
}

void PairingGenivi::reset(void)
{
   _tokenList.clear();
   _pendingList.clear();
   _testTriggerSimulateStackReset = false;
}

void PairingGenivi::setCallback(IN IPairingCallback* callback)
{
   _callback = callback;

   FW_NORMAL_ASSERT(0 != _callback);
}

void PairingGenivi::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_NORMAL_ASSERT(0 != _controlIf);
}

void PairingGenivi::startPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address)
{
   (void)(bts2AppMsgList);

   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   // all necessary checks are done before

   BTSObjectPath device;
   if(false == _objectPathManagerIf->getObjectPath4Address(device, address))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   createPairMsg(bts2IpcMsgList, address, device);
}

void PairingGenivi::cancelPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address)
{
   (void)(bts2AppMsgList);

   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   // all necessary checks are done before

   BTSObjectPath device;
   if(false == _objectPathManagerIf->getObjectPath4Address(device, address))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   createCancelPairingMsg(bts2IpcMsgList, address, device);
}

void PairingGenivi::setPinCode(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSPinCode& pinCode)
{
   (void)(bts2AppMsgList);

   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   // all necessary checks are done before

   BTSObjectPath device;
   if(false == _objectPathManagerIf->getObjectPath4Address(device, address))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   ::std::map< BTSObjectPath, act_t >::iterator it = _tokenList.find(device);
   if(_tokenList.end() != it)
   {
      // entry is available
      createRequestPinCodeRes(bts2IpcMsgList, address, pinCode, it->second, true);

      _tokenList.erase(it);
   }
   else
   {
      // entry is not available => should never happen
      FW_NORMAL_ASSERT_ALWAYS();
   }

   // check if given device is in pending list (could happen if first pairing request is not answered by application); if found remove
   removeDevice(device);
}

void PairingGenivi::confirmSecureSimplePairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSSspMode mode, IN const BTSNumericValue& numValue, IN const BTSConfirmationMode confirm)
{
   (void)(bts2AppMsgList);

   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   // all necessary checks are done before

   BTSObjectPath device;
   if(false == _objectPathManagerIf->getObjectPath4Address(device, address))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   ::std::map< BTSObjectPath, act_t >::iterator it = _tokenList.find(device);
   if(_tokenList.end() != it)
   {
      // entry is available
      switch(mode)
      {
         case BTS_SSP_WITH_NUMERIC_COMPARISON:
         {
            createRequestConfirmationRes(bts2IpcMsgList, address, confirm, it->second, true);
            break;
         }
         case BTS_SSP_WITH_JUST_WORK_MODE: // TODO: to be tested
         {
            createRequestAuthorizationRes(bts2IpcMsgList, address, confirm, it->second, true);
            break;
         }
         case BTS_SSP_WITH_PASSKEY_ENTRY_MODE_INPUT_BY_REMOTE_DEVICE: // TODO: to be tested
         {
            // response was already sent (make DBUS happy)
            // wait for timeout because we are completely in a passive mode (input must be done on remote device)
            break;
         }
         case BTS_SSP_WITH_PASSKEY_ENTRY_MODE_INPUT_BY_LOCAL_DEVICE: // TODO: to be tested
         {
            createRequestPasskeyRes(bts2IpcMsgList, address, numValue, it->second, true);
            break;
         }
         case BTS_SSP_LAST:
         default:
         {
            // should never happen
            FW_NORMAL_ASSERT_ALWAYS();
            break;
         }
      }

      _tokenList.erase(it);
   }
   else
   {
      // entry is not available => should never happen
      FW_NORMAL_ASSERT_ALWAYS();
   }

   // check if given device is in pending list (could happen if first pairing request is not answered by application); if found remove
   removeDevice(device);
}

void PairingGenivi::setObjectPathManagerIf(IN IObjectPathManagerGenivi* objectPathManager)
{
   _objectPathManagerIf = objectPathManager;

   FW_NORMAL_ASSERT(0 != _objectPathManagerIf);
}

void PairingGenivi::handlePairingRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSSspMode sspMode, IN const BTSNumericValue& numValue, IN const act_t token)
{
   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   // store token
   ::std::map< BTSObjectPath, act_t >::iterator it = _tokenList.find(device);
   if(_tokenList.end() != it)
   {
      // entry is already stored => should never happen
      _tokenList[device] = token; // update and continue with latest request
   }
   else
   {
      // add entry
      _tokenList[device] = token;
   }

   // check if mapping is available
   BTSBDAddress address;
   if(true == _objectPathManagerIf->getAddress4ObjectPath(address, device))
   {
      // we can directly forward
      checkForDisplayPinCodeRequest(bts2IpcMsgList, device, address, sspMode);
      forwardRequest(bts2IpcMsgList, bts2AppMsgList, messageItem, address, sspMode, numValue);
   }
   else
   {
      // mapping is not available => add to pending list
      if(true == isDevicePending(device))
      {
         // entry is already stored => should never happen
         FW_NORMAL_ASSERT_ALWAYS();
      }
      else
      {
         // add entry
         addDevice(device, sspMode, numValue);
      }
   }
}

bool PairingGenivi::handleCancelRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool sent, IN const act_t token)
{
   (void)(bts2AppMsgList);
   (void)(messageItem);

   /*
    * Cancel request will be triggered by Evolution in following scenarios:
    * - timeout RequestPinCode: 25s => will be handled by IncomingPairingComp (failed pairing)
    * - timeout RequestPasskey: 25s => will be handled by IncomingPairingComp (failed pairing)
    * - timeout RequestConfirmation: 25s => will be handled by IncomingPairingComp (failed pairing)
    */

   // but ensure that answer is sent
   if(false == sent)
   {
      createCancelRes(bts2IpcMsgList, token, true);
   }

   return true;
}

bool PairingGenivi::handleCancelRequestRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const bool sent, IN const act_t token)
{
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(device);

   /*
    * Cancel request will be triggered by Evolution in following scenarios:
    * - timeout RequestPinCode: 25s => will be handled by IncomingPairingComp (failed pairing)
    * - timeout RequestPasskey: 25s => will be handled by IncomingPairingComp (failed pairing)
    * - timeout RequestConfirmation: 25s => will be handled by IncomingPairingComp (failed pairing)
    */

   // but ensure that answer is sent
   if(false == sent)
   {
      createCancelRequestRes(bts2IpcMsgList, token, true);
   }

   return true;
}

void PairingGenivi::handlePairResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSIpcCommonErrorCode errorCode)
{
   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   // check if mapping is available
   BTSBDAddress address;
   if(false == _objectPathManagerIf->getAddress4ObjectPath(address, device))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   _callback->startPairingResult(bts2IpcMsgList, bts2AppMsgList, messageItem, address, convertPairingCommonErrorCode2Result(errorCode));
}

void PairingGenivi::handleCancelPairingResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSIpcCommonErrorCode errorCode)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(device);
   (void)(errorCode);

   // ignore
}

void PairingGenivi::handleReportLinkKey(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSLinkKey& linkKey)
{
   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   // check if mapping is available
   BTSBDAddress address;
   if(false == _objectPathManagerIf->getAddress4ObjectPath(address, device))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   BTSLinkKey appLinkKey;
   BTSRequestResult result(BTS_REQ_PAIRING_UNKNOWN);
   if((false == linkKey.empty()) && (true == isValidLinkKey(linkKey)))
   {
      result = BTS_REQ_SUCCESS;
      appLinkKey = linkKey;
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   const BTSDLinkKey dLinkKey;
   _callback->indicateLinkKey(bts2IpcMsgList, bts2AppMsgList, messageItem, address, result, BTS_LINK_KEY_TYPE_UNKNOWN, appLinkKey, dLinkKey);
}

void PairingGenivi::handleIncomingPairingComp(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSRequestResult result)
{
   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   // check if mapping is available
   BTSBDAddress address;
   if(false == _objectPathManagerIf->getAddress4ObjectPath(address, device))
   {
      // should never happen
      return;
   }

   if(BTS_REQ_SUCCESS == result)
   {
      _callback->handleEndOfSuccessfulIncomingPairing(bts2IpcMsgList, bts2AppMsgList, messageItem, address);
   }
   else
   {
      const BTSLinkKey linkKey;
      const BTSDLinkKey dLinkKey;
      _callback->indicateLinkKey(bts2IpcMsgList, bts2AppMsgList, messageItem, address, result, BTS_LINK_KEY_TYPE_UNKNOWN, linkKey, dLinkKey);
   }
}

void PairingGenivi::setTriggerSimulateStackReset(IN const bool enable)
{
   _testTriggerSimulateStackReset = enable;
}

void PairingGenivi::deviceAdded(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   BTSObjectPath device;
   if(false == _objectPathManagerIf->getObjectPath4Address(device, address))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // check if given device is in pending list
   BTSSspMode sspMode(BTS_SSP_LAST);
   BTSNumericValue numValue;

   if(true == isDevicePending(sspMode, numValue, device))
   {
      // entry is available => forward request and remove
      checkForDisplayPinCodeRequest(bts2IpcMsgList, device, address, sspMode);
      forwardRequest(bts2IpcMsgList, bts2AppMsgList, messageItem, address, sspMode, numValue);
      removeDevice(device);
   }
}

void PairingGenivi::deviceRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   BTSObjectPath device;
   if(false == _objectPathManagerIf->getObjectPath4Address(device, address))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // check if given device is in pending list; if found remove
   removeDevice(device);
}

void PairingGenivi::deviceAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   // information not needed
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(address);
}

void PairingGenivi::deviceUnavailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   // information not needed
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(address);
}

void PairingGenivi::deviceConnectionStatus(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSStatusTransition aclTransition, IN const bool aclConnected, IN const BTSStatusTransition anyProfileTransition, IN const bool anyProfileConnected)
{
   // information not needed
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(address);
   (void)(aclTransition);
   (void)(aclConnected);
   (void)(anyProfileTransition);
   (void)(anyProfileConnected);
}

void PairingGenivi::deviceCreationFinished(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result)
{
   // information not needed
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(address);
   (void)(result);
}

void PairingGenivi::deviceRemovalFinished(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result)
{
   // information not needed
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(address);
   (void)(result);
}

void PairingGenivi::createPairMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSObjectPath& device) const
{
   Bts2Ipc_Pair* msg = ptrNew_Bts2Ipc_Pair();
   if(0 != msg)
   {
      msg->setDevice(device);
      msg->setBDAddress(address);
      msg->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg);
   }
}

void PairingGenivi::createCancelPairingMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSObjectPath& device) const
{
   Bts2Ipc_CancelPairing* msg = ptrNew_Bts2Ipc_CancelPairing();
   if(0 != msg)
   {
      msg->setDevice(device);
      msg->setBDAddress(address);
      msg->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg);
   }
}

void PairingGenivi::createRequestPinCodeRes(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSPinCode& pinCode, IN const act_t token, IN const bool responseFlag /*= false*/) const
{
   Bts2Ipc_RequestPinCodeRes* msg = ptrNew_Bts2Ipc_RequestPinCodeRes();
   if(0 != msg)
   {
      msg->setBDAddress(address);
      msg->setPinCode(pinCode);
      msg->setDbusToken(token);
      msg->setResponseMessageFlag(responseFlag);

      bts2IpcMsgList.push_back(msg);
   }
}

void PairingGenivi::createRequestConfirmationRes(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSConfirmationMode confirm, IN const act_t token, IN const bool responseFlag /*= false*/) const
{
   Bts2Ipc_RequestConfirmationRes* msg = ptrNew_Bts2Ipc_RequestConfirmationRes();
   if(0 != msg)
   {
      msg->setBDAddress(address);
      msg->setAccept(confirm);
      msg->setDbusToken(token);
      msg->setResponseMessageFlag(responseFlag);

      bts2IpcMsgList.push_back(msg);
   }
}

void PairingGenivi::createRequestAuthorizationRes(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSConfirmationMode confirm, IN const act_t token, IN const bool responseFlag /*= false*/) const
{
   Bts2Ipc_RequestAuthorizationRes* msg = ptrNew_Bts2Ipc_RequestAuthorizationRes();
   if(0 != msg)
   {
      msg->setBDAddress(address);
      msg->setAccept(confirm);
      msg->setDbusToken(token);
      msg->setResponseMessageFlag(responseFlag);

      bts2IpcMsgList.push_back(msg);
   }
}

void PairingGenivi::createRequestPasskeyRes(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSNumericValue& numValue, IN const act_t token, IN const bool responseFlag /*= false*/) const
{
   Bts2Ipc_RequestPasskeyRes* msg = ptrNew_Bts2Ipc_RequestPasskeyRes();
   if(0 != msg)
   {
      msg->setBDAddress(address);
      msg->setPasskey(numValue);
      msg->setDbusToken(token);
      msg->setResponseMessageFlag(responseFlag);

      bts2IpcMsgList.push_back(msg);
   }
}

void PairingGenivi::createDisplayPinCodeRes(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& address, IN const act_t token, IN const bool responseFlag /*= false*/) const
{
   Bts2Ipc_DisplayPinCodeRes* msg = ptrNew_Bts2Ipc_DisplayPinCodeRes();
   if(0 != msg)
   {
      msg->setBDAddress(address);
      msg->setDbusToken(token);
      msg->setResponseMessageFlag(responseFlag);

      bts2IpcMsgList.push_back(msg);
   }
}

void PairingGenivi::createCancelRes(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const act_t token, IN const bool responseFlag /*= false*/) const
{
   Bts2Ipc_CancelRes* msg = ptrNew_Bts2Ipc_CancelRes();
   if(0 != msg)
   {
      msg->setDbusToken(token);
      msg->setResponseMessageFlag(responseFlag);

      bts2IpcMsgList.push_back(msg);
   }
}

void PairingGenivi::createCancelRequestRes(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const act_t token, IN const bool responseFlag /*= false*/) const
{
   Bts2Ipc_CancelRequestRes* msg = ptrNew_Bts2Ipc_CancelRequestRes();
   if(0 != msg)
   {
      msg->setDbusToken(token);
      msg->setResponseMessageFlag(responseFlag);

      bts2IpcMsgList.push_back(msg);
   }
}

void PairingGenivi::forwardRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSSspMode sspMode, IN const BTSNumericValue& numValue)
{
   FW_IF_NULL_PTR_RETURN(_callback);

   if(BTS_SSP_LAST == sspMode)
   {
      _callback->indicatePinCodeRequest(bts2IpcMsgList, bts2AppMsgList, messageItem, address);
   }
   else
   {
      _callback->indicateSecureSimplePairingRequest(bts2IpcMsgList, bts2AppMsgList, messageItem, address, sspMode, numValue);
   }

   //===================================================================================================================
   // debug section start
   if(true == _testTriggerSimulateStackReset)
   {
      _testTriggerSimulateStackReset = false;

      if(0 != _controlIf)
      {
         Ipc2Bts_ServiceAvailabilityConnection* msg = ptrNew_Ipc2Bts_ServiceAvailabilityConnection();
         if(0 != msg)
         {
            ::ccdbusif::evolution::EvolutionGeniviDbusParser evoParser;

            msg->setIpcCommonErrorCode(BTS_IPC_SUCCESS);
            msg->setInterface(BTS_GEN_DBUS_SERVICE_OBJECT_MANAGER_SYSTEM);
            msg->setAvailabilityEvent(BTS_DBUS_SERVICE_NOT_AVAILABLE);
            msg->setBusType((BTSCommonEnumClass)evoParser.getInterface2BusType(::ccdbusif::evolution::IF_ADAPTER));
            msg->setBusName(evoParser.getInterface2BusName(::ccdbusif::evolution::IF_ADAPTER));
            msg->setObjPath(evoParser.getRootObjectPath());

            _controlIf->sendInternalIpc2BtsMessage(msg);
         }
      }
   }
   // debug section end
   //===================================================================================================================
}

void PairingGenivi::checkForDisplayPinCodeRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSObjectPath& device, IN const BTSBDAddress& address, IN const BTSSspMode sspMode) const
{
   if(BTS_SSP_WITH_PASSKEY_ENTRY_MODE_INPUT_BY_REMOTE_DEVICE == sspMode)
   {
      // there is no accept/reject from user necessary on local side (head unit) => send response now
      ::std::map< BTSObjectPath, act_t >::const_iterator it = _tokenList.find(device);
      if(_tokenList.end() != it)
      {
         // entry is available
         createDisplayPinCodeRes(bts2IpcMsgList, address, it->second, true);

         // keep token in list
      }
      else
      {
         // entry is not available => should never happen
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }
}

void PairingGenivi::addDevice(IN const BTSObjectPath& device, IN const BTSSspMode sspMode, IN const BTSNumericValue& numValue)
{
   PairingGeniviData entry;

   entry.device = device;
   entry.sspMode = sspMode;
   entry.numValue = numValue;

   _pendingList.push_back(entry);
}

bool PairingGenivi::isDevicePending(IN const BTSObjectPath& device) const
{
   for(::std::vector< PairingGeniviData >::const_iterator it = _pendingList.begin(); it != _pendingList.end(); ++it)
   {
      if(it->device == device)
      {
         return true;
      }
   }

   return false;
}

bool PairingGenivi::isDevicePending(OUT BTSSspMode& sspMode, OUT BTSNumericValue& numValue, IN const BTSObjectPath& device) const
{
   for(::std::vector< PairingGeniviData >::const_iterator it = _pendingList.begin(); it != _pendingList.end(); ++it)
   {
      if(it->device == device)
      {
         sspMode = it->sspMode;
         numValue = it->numValue;
         return true;
      }
   }

   return false;
}

void PairingGenivi::removeDevice(IN const BTSObjectPath& device)
{
   for(::std::vector< PairingGeniviData >::iterator it = _pendingList.begin(); it != _pendingList.end(); ++it)
   {
      if(it->device == device)
      {
         _pendingList.erase(it);
         break;
      }
   }
}

} //genivi
} //btstackif
