/**
 * @file ObjectPathManagerGenivi.cpp
 *
 * @par SW-Component
 * Control
 *
 * @brief StateMachine for object path handling.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details StateMachine for object path handling.
 */

#include "ObjectPathManagerGenivi.h"
#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/ObjectPathManagerGenivi.cpp.trc.h"
#endif
#endif

namespace btstackif {
namespace genivi {

ObjectPathManager::ObjectPathManager() :
_device2ObjectPath(),
_objectPath2Device(),
_protocol2ObjectPath(),
_objectPath2Protocol(),
_modem2ObjectPath(),
_objectPath2Modem()
{
}

ObjectPathManager::~ObjectPathManager()
{
}

IObjectPathManagerGenivi& ObjectPathManager::getInstance(void)
{
   static ObjectPathManager manager;
   return manager;
}

void ObjectPathManager::reset(void)
{
   _device2ObjectPath.clear();
   _objectPath2Device.clear();
   _protocol2ObjectPath.clear();
   _objectPath2Protocol.clear();
   _modem2ObjectPath.clear();
   _objectPath2Modem.clear();
#if 0
   _call2ObjectPath.clear();
   _objectPath2Call.clear();
   _operator2ObjectPath.clear();
   _objectPath2Operator.clear();
   _transfer2ObjectPath.clear();
   _objectPath2Transfer.clear();
   _message2ObjectPath.clear();
   _objectPath2Message.clear();
#endif
}

//------------------------------------------------------------------------------

void ObjectPathManager::setAddress2ObjectPath(IN const BTSBDAddress& address, IN const BTSObjectPath& objPath)
{
   if(_device2ObjectPath.end() == _device2ObjectPath.find(address))
   {
      _device2ObjectPath[address] = objPath;
      _objectPath2Device[objPath] = address;
   }
}

void ObjectPathManager::removeAddress2ObjectPath(IN const BTSObjectPath& objPath)
{
   ::std::map<BTSObjectPath, BTSBDAddress>::iterator it1 = _objectPath2Device.find(objPath);
   if(_objectPath2Device.end() != it1)
   {
      ::std::map<BTSBDAddress, BTSObjectPath>::iterator it2 = _device2ObjectPath.find(it1->second);
      if(_device2ObjectPath.end() != it2)
      {
         _device2ObjectPath.erase(it2);
      }
      _objectPath2Device.erase(it1);
   }
}

bool ObjectPathManager::getObjectPath4Address(OUT BTSObjectPath& objPath, IN const BTSBDAddress& address) const
{
   bool found = false;

   ::std::map<BTSBDAddress, BTSObjectPath>::const_iterator it = _device2ObjectPath.find(address);
   if(_device2ObjectPath.end() != it)
   {
      objPath = it->second;
      found = true;
   }

   return found;
}

bool ObjectPathManager::getAddress4ObjectPath(OUT BTSBDAddress& address, IN const BTSObjectPath& objPath) const
{
   bool found = false;

   ::std::map<BTSObjectPath, BTSBDAddress>::const_iterator it = _objectPath2Device.find(objPath);
   if(_objectPath2Device.end() != it)
   {
      address = it->second;
      found = true;
   }

   return found;
}

void ObjectPathManager::getAllObjectPaths4Address(OUT ::std::vector< BTSObjectPath >& objPathList) const
{
   for(::std::map<BTSObjectPath, BTSBDAddress>::const_iterator it = _objectPath2Device.begin(); it != _objectPath2Device.end(); ++it)
   {
      objPathList.push_back(it->first);
   }
}

void ObjectPathManager::removeAllAddress2ObjectPathMappings(void)
{
   _device2ObjectPath.clear();
   _objectPath2Device.clear();
}

//------------------------------------------------------------------------------

void ObjectPathManager::setProtocol2ObjectPath(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSObjectPath& objPath)
{
   switch(protocol)
   {
      case BTS_PROTO_HFP:
         intSetProtocol2ObjectPath(address, protocol, BTSUuid(), 0, objPath);
         break;
      case BTS_PROTO_AVP:
         intSetProtocol2ObjectPath(address, protocol, BTSUuid(), 0, objPath);
         break;
      case BTS_PROTO_PIM:
         intSetProtocol2ObjectPath(address, protocol, BTSUuid(), 0, objPath);
         break;
      case BTS_PROTO_MSG:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_DUN:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_PAN:
         intSetProtocol2ObjectPath(address, protocol, BTSUuid(), 0, objPath);
         break;
      case BTS_PROTO_SPP:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_LAST:
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }
}

void ObjectPathManager::setProtocol2ObjectPath(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSMasInstanceId masInstance, IN const BTSObjectPath& objPath)
{
   switch(protocol)
   {
      case BTS_PROTO_HFP:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_AVP:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_PIM:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_MSG:
         intSetProtocol2ObjectPath(address, protocol, BTSUuid(), masInstance, objPath);
         break;
      case BTS_PROTO_DUN:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_PAN:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_SPP:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_LAST:
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }
}

void ObjectPathManager::setProtocol2ObjectPath(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSObjectPath& objPath)
{
   switch(protocol)
   {
      case BTS_PROTO_HFP:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_AVP:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_PIM:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_MSG:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_DUN:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_PAN:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_SPP:
         intSetProtocol2ObjectPath(address, protocol, uuid, 0, objPath);
         break;
      case BTS_PROTO_LAST:
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }
}

void ObjectPathManager::removeProtocol2ObjectPath(IN const BTSObjectPath& objPath)
{
   ::std::map<BTSObjectPath, ProtocolEntry>::iterator it1 = _objectPath2Protocol.find(objPath);
   if(_objectPath2Protocol.end() != it1)
   {
      ::std::map<ProtocolEntry, BTSObjectPath>::iterator it2 = _protocol2ObjectPath.find(it1->second);
      if(_protocol2ObjectPath.end() != it2)
      {
         _protocol2ObjectPath.erase(it2);
      }
      _objectPath2Protocol.erase(it1);
   }
}

void ObjectPathManager::removeProtocol2ObjectPath(IN const BTSBDAddress& address, IN const BTSProtocolId protocol)
{
   switch(protocol)
   {
      case BTS_PROTO_HFP:
         intRemoveProtocol2ObjectPath(address, protocol, BTSUuid(), 0);
         break;
      case BTS_PROTO_AVP:
         intRemoveProtocol2ObjectPath(address, protocol, BTSUuid(), 0);
         break;
      case BTS_PROTO_PIM:
         intRemoveProtocol2ObjectPath(address, protocol, BTSUuid(), 0);
         break;
      case BTS_PROTO_MSG:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_DUN:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_PAN:
         intRemoveProtocol2ObjectPath(address, protocol, BTSUuid(), 0);
         break;
      case BTS_PROTO_SPP:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_LAST:
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }
}

void ObjectPathManager::removeProtocol2ObjectPath(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSMasInstanceId masInstance)
{
   switch(protocol)
   {
      case BTS_PROTO_HFP:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_AVP:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_PIM:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_MSG:
         intRemoveProtocol2ObjectPath(address, protocol, BTSUuid(), masInstance);
         break;
      case BTS_PROTO_DUN:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_PAN:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_SPP:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_LAST:
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }
}

void ObjectPathManager::removeProtocol2ObjectPath(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid)
{
   switch(protocol)
   {
      case BTS_PROTO_HFP:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_AVP:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_PIM:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_MSG:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_DUN:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_PAN:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_SPP:
         intRemoveProtocol2ObjectPath(address, protocol, uuid, 0);
         break;
      case BTS_PROTO_LAST:
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }
}

bool ObjectPathManager::getObjectPath4Protocol(OUT BTSObjectPath& objPath, IN const BTSBDAddress& address, IN const BTSProtocolId protocol) const
{
   bool result = false;

   switch(protocol)
   {
      case BTS_PROTO_HFP:
         result = intGetObjectPath4Protocol(objPath, address, protocol, BTSUuid(), 0);
         break;
      case BTS_PROTO_AVP:
         result = intGetObjectPath4Protocol(objPath, address, protocol, BTSUuid(), 0);
         break;
      case BTS_PROTO_PIM:
         result = intGetObjectPath4Protocol(objPath, address, protocol, BTSUuid(), 0);
         break;
      case BTS_PROTO_MSG:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_DUN:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_PAN:
         result = intGetObjectPath4Protocol(objPath, address, protocol, BTSUuid(), 0);
         break;
      case BTS_PROTO_SPP:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_LAST:
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }

   return result;
}

bool ObjectPathManager::getObjectPath4Protocol(OUT BTSObjectPath& objPath, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSMasInstanceId masInstance) const
{
   bool result = false;

   switch(protocol)
   {
      case BTS_PROTO_HFP:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_AVP:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_PIM:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_MSG:
         result = intGetObjectPath4Protocol(objPath, address, protocol, BTSUuid(), masInstance);
         break;
      case BTS_PROTO_DUN:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_PAN:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_SPP:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_LAST:
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }

   return result;
}

bool ObjectPathManager::getObjectPath4Protocol(OUT BTSObjectPath& objPath, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid) const
{
   bool result = false;

   switch(protocol)
   {
      case BTS_PROTO_HFP:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_AVP:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_PIM:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_MSG:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_DUN:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_PAN:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case BTS_PROTO_SPP:
         result = intGetObjectPath4Protocol(objPath, address, protocol, uuid, 0);
         break;
      case BTS_PROTO_LAST:
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }

   return result;
}

bool ObjectPathManager::getProtocol4ObjectPath(OUT BTSBDAddress& address, OUT BTSProtocolId& protocol, IN const BTSObjectPath& objPath) const
{
   BTSUuid uuid;
   BTSMasInstanceId masInstance;
   return intGetProtocol4ObjectPath(address, protocol, uuid, masInstance, objPath);
}

bool ObjectPathManager::getProtocol4ObjectPath(OUT BTSBDAddress& address, OUT BTSProtocolId& protocol, OUT BTSMasInstanceId& masInstance, IN const BTSObjectPath& objPath) const
{
   BTSUuid uuid;
   return intGetProtocol4ObjectPath(address, protocol, uuid, masInstance, objPath);
}

bool ObjectPathManager::getProtocol4ObjectPath(OUT BTSBDAddress& address, OUT BTSProtocolId& protocol, OUT BTSUuid& uuid, IN const BTSObjectPath& objPath) const
{
   BTSMasInstanceId masInstance;
   return intGetProtocol4ObjectPath(address, protocol, uuid, masInstance, objPath);
}

void ObjectPathManager::removeAllProtocol2ObjectPathMappings(void)
{
   _protocol2ObjectPath.clear();
   _objectPath2Protocol.clear();
}

void ObjectPathManager::printObjectPath2ProtocolMapping(IN const BTSObjectPath& objPath) const
{
   ETG_TRACE_USR1((" printObjectPath2ProtocolMapping(): _objectPath2Protocol.size()=%u objPath=%s", _objectPath2Protocol.size(), objPath.c_str()));

   size_t i = 0;
   for(::std::map<BTSObjectPath, ProtocolEntry>::const_iterator it = _objectPath2Protocol.begin(); it != _objectPath2Protocol.end(); ++it)
   {
      const ProtocolEntry& entry = it->second;
      ETG_TRACE_USR1((" printObjectPath2ProtocolMapping(): _objectPath2Protocol[%u]: objPath=%s", i, it->first.c_str()));
      ETG_TRACE_USR1((" printObjectPath2ProtocolMapping(): _objectPath2Protocol[%u]: protocolId=%d masInstanceId=%u deviceAddress=%20s sppUuid=%s", i, entry.protocolId, entry.masInstanceId, entry.deviceAddress.c_str(), entry.sppUuid.c_str()));
      i++;
   }
}

//------------------------------------------------------------------------------

void ObjectPathManager::setModem2ObjectPath(IN const BTSBDAddress& address, IN const BTSObjectPath& objPath)
{
   if(_modem2ObjectPath.end() == _modem2ObjectPath.find(address))
   {
      _modem2ObjectPath[address] = objPath;
      _objectPath2Modem[objPath] = address;
   }
}

void ObjectPathManager::removeModem2ObjectPath(IN const BTSObjectPath& objPath)
{
   ::std::map<BTSObjectPath, BTSBDAddress>::iterator it1 = _objectPath2Modem.find(objPath);
   if(_objectPath2Modem.end() != it1)
   {
      ::std::map<BTSBDAddress, BTSObjectPath>::iterator it2 = _modem2ObjectPath.find(it1->second);
      if(_modem2ObjectPath.end() != it2)
      {
         _modem2ObjectPath.erase(it2);
      }
      _objectPath2Modem.erase(it1);
   }
}

bool ObjectPathManager::getObjectPath4Modem(OUT BTSObjectPath& objPath, IN const BTSBDAddress& address) const
{
   bool found = false;

   ::std::map<BTSBDAddress, BTSObjectPath>::const_iterator it = _modem2ObjectPath.find(address);
   if(_modem2ObjectPath.end() != it)
   {
      objPath = it->second;
      found = true;
   }

   return found;
}

bool ObjectPathManager::getModem4ObjectPath(OUT BTSBDAddress& address, IN const BTSObjectPath& objPath) const
{
   bool found = false;

   ::std::map<BTSObjectPath, BTSBDAddress>::const_iterator it = _objectPath2Modem.find(objPath);
   if(_objectPath2Modem.end() != it)
   {
      address = it->second;
      found = true;
   }

   return found;
}

void ObjectPathManager::removeAllModem2ObjectPathMappings(void)
{
   _modem2ObjectPath.clear();
   _objectPath2Modem.clear();
}

//------------------------------------------------------------------------------

void ObjectPathManager::intSetProtocol2ObjectPath(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSObjectPath& objPath)
{
   ProtocolEntry entry;
   entry.deviceAddress = address;
   entry.protocolId = protocol;
   entry.sppUuid = uuid;
   entry.masInstanceId = masInstance;

   if(_protocol2ObjectPath.end() == _protocol2ObjectPath.find(entry))
   {
      _protocol2ObjectPath[entry] = objPath;
      _objectPath2Protocol[objPath] = entry;
   }
}

void ObjectPathManager::intRemoveProtocol2ObjectPath(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance)
{
   ProtocolEntry entry;
   entry.deviceAddress = address;
   entry.protocolId = protocol;
   entry.sppUuid = uuid;
   entry.masInstanceId = masInstance;

   ::std::map<ProtocolEntry, BTSObjectPath>::iterator it1 = _protocol2ObjectPath.find(entry);
   if(_protocol2ObjectPath.end() != it1)
   {
      ::std::map<BTSObjectPath, ProtocolEntry>::iterator it2 = _objectPath2Protocol.find(it1->second);
      if(_objectPath2Protocol.end() != it2)
      {
         _objectPath2Protocol.erase(it2);
      }
      _protocol2ObjectPath.erase(it1);
   }
}

bool ObjectPathManager::intGetObjectPath4Protocol(OUT BTSObjectPath& objPath, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance) const
{
   bool found = false;

   ProtocolEntry entry;
   entry.deviceAddress = address;
   entry.protocolId = protocol;
   entry.sppUuid = uuid;
   entry.masInstanceId = masInstance;

   ::std::map<ProtocolEntry, BTSObjectPath>::const_iterator it = _protocol2ObjectPath.find(entry);
   if(_protocol2ObjectPath.end() != it)
   {
      objPath = it->second;
      found = true;
   }

   return found;
}

bool ObjectPathManager::intGetProtocol4ObjectPath(OUT BTSBDAddress& address, OUT BTSProtocolId& protocol, OUT BTSUuid& uuid, OUT BTSMasInstanceId& masInstance, IN const BTSObjectPath& objPath) const
{
   bool found = false;

   ::std::map<BTSObjectPath, ProtocolEntry>::const_iterator it = _objectPath2Protocol.find(objPath);
   if(_objectPath2Protocol.end() != it)
   {
      address = it->second.deviceAddress;
      protocol = it->second.protocolId;
      uuid = it->second.sppUuid;
      masInstance = it->second.masInstanceId;
      found = true;
   }

   return found;
}

//------------------------------------------------------------------------------

ObjectPathManager::ProtocolEntry::ProtocolEntry()
{
   // deviceAddress
   protocolId = BTS_PROTO_LAST;
   // sppUuid
   masInstanceId = 0;
}

ObjectPathManager::ProtocolEntry::ProtocolEntry(const ProtocolEntry& ref)
{
   deviceAddress = ref.deviceAddress;
   protocolId = ref.protocolId;
   sppUuid = ref.sppUuid;
   masInstanceId = ref.masInstanceId;
}

ObjectPathManager::ProtocolEntry& ObjectPathManager::ProtocolEntry::operator=(const ProtocolEntry& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   deviceAddress = ref.deviceAddress;
   protocolId = ref.protocolId;
   sppUuid = ref.sppUuid;
   masInstanceId = ref.masInstanceId;

   return *this;
}

bool ObjectPathManager::ProtocolEntry::operator==(const ProtocolEntry& ref) const
{
   return (0 == compare(ref));
}

bool ObjectPathManager::ProtocolEntry::operator!=(const ProtocolEntry& ref) const
{
   return (0 != compare(ref));
}

bool ObjectPathManager::ProtocolEntry::operator<(const ProtocolEntry& ref) const
{
   return (0 > compare(ref));
}

bool ObjectPathManager::ProtocolEntry::operator>(const ProtocolEntry& ref) const
{
   return (0 < compare(ref));
}

ObjectPathManager::ProtocolEntry::~ProtocolEntry()
{
}

int ObjectPathManager::ProtocolEntry::compare(const ProtocolEntry& ref) const
{
   int result;

   result = deviceAddress.compare(ref.deviceAddress);
   if(result < 0)
   {
      return -1;
   }
   else if(result > 0)
   {
      return 1;
   }

   // device address is equal

   if(protocolId < ref.protocolId)
   {
      return -1;
   }
   else if(protocolId > ref.protocolId)
   {
      return 1;
   }

   // protocol id is equal

   result = sppUuid.compare(ref.sppUuid);
   if(result < 0)
   {
      return -1;
   }
   else if(result > 0)
   {
      return 1;
   }

   // SPP UUID is equal

   if(masInstanceId < ref.masInstanceId)
   {
      return -1;
   }
   else if(masInstanceId > ref.masInstanceId)
   {
      return 1;
   }

   // MAS instance id is equal

   return 0;
}

//------------------------------------------------------------------------------

#if 0
ObjectPathManager::SimpleCountEntry::SimpleCountEntry()
{
   // deviceAddress
   count = 0;
}

ObjectPathManager::SimpleCountEntry::SimpleCountEntry(const SimpleCountEntry& ref)
{
   deviceAddress = ref.deviceAddress;
   count = ref.count;
}

ObjectPathManager::SimpleCountEntry& ObjectPathManager::SimpleCountEntry::operator=(const SimpleCountEntry& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   deviceAddress = ref.deviceAddress;
   count = ref.count;

   return *this;
}

bool ObjectPathManager::SimpleCountEntry::operator==(const SimpleCountEntry& ref) const
{
   return (0 == compare(ref));
}

bool ObjectPathManager::SimpleCountEntry::operator!=(const SimpleCountEntry& ref) const
{
   return (0 != compare(ref));
}

bool ObjectPathManager::SimpleCountEntry::operator<(const SimpleCountEntry& ref) const
{
   return (0 > compare(ref));
}

bool ObjectPathManager::SimpleCountEntry::operator>(const SimpleCountEntry& ref) const
{
   return (0 < compare(ref));
}

ObjectPathManager::SimpleCountEntry::~SimpleCountEntry()
{
}

int ObjectPathManager::SimpleCountEntry::compare(const SimpleCountEntry& ref) const
{
   int result;

   result = deviceAddress.compare(ref.deviceAddress);
   if(result < 0)
   {
      return -1;
   }
   else if(result > 0)
   {
      return 1;
   }

   // device address is equal

   if(count < ref.count)
   {
      return -1;
   }
   else if(count > ref.count)
   {
      return 1;
   }

   // count id is equal

   return 0;
}

//------------------------------------------------------------------------------

ObjectPathManager::ExtendedCountEntry::ExtendedCountEntry()
: ObjectPathManager::ProtocolEntry()
{
   count = 0;
}

ObjectPathManager::ExtendedCountEntry::ExtendedCountEntry(const ExtendedCountEntry& ref)
: ObjectPathManager::ProtocolEntry(ref)
{
   count = ref.count;
}

ObjectPathManager::ExtendedCountEntry& ObjectPathManager::ExtendedCountEntry::operator=(const ExtendedCountEntry& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   ProtocolEntry::operator=(ref);

   count = ref.count;

   return *this;
}

bool ObjectPathManager::ExtendedCountEntry::operator==(const ExtendedCountEntry& ref) const
{
   return (0 == compare(ref));
}

bool ObjectPathManager::ExtendedCountEntry::operator!=(const ExtendedCountEntry& ref) const
{
   return (0 != compare(ref));
}

bool ObjectPathManager::ExtendedCountEntry::operator<(const ExtendedCountEntry& ref) const
{
   return (0 > compare(ref));
}

bool ObjectPathManager::ExtendedCountEntry::operator>(const ExtendedCountEntry& ref) const
{
   return (0 < compare(ref));
}

ObjectPathManager::ExtendedCountEntry::~ExtendedCountEntry()
{
}

int ObjectPathManager::ExtendedCountEntry::compare(const ExtendedCountEntry& ref) const
{
   int result;

   result = ProtocolEntry::compare(ref);
   if(result < 0)
   {
      return -1;
   }
   else if(result > 0)
   {
      return 1;
   }

   // ProtocolEntry is equal

   if(count < ref.count)
   {
      return -1;
   }
   else if(count > ref.count)
   {
      return 1;
   }

   // count is equal

   return 0;
}
#endif

} //genivi
} //btstackif
