/**
 * @file ObexConnectDisconnectGenivi.cpp
 *
 * @par SW-Component
 * State machine for OBEX connect/disconnect
 *
 * @brief Implementation of Genivi OBEX connect/disconnect state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of Genivi OBEX connect/disconnect state machine.
 */

#include "ObexConnectDisconnectGenivi.h"
#include "IProtocolManagerCallback.h"
#include "ProtocolManagerData.h"
#include "IBasicControl.h"
#include "ISwitchBluetooth.h"
#include "IDeviceManager.h"
#include "IServiceSearch.h"
#include "IObjectPathManagerGenivi.h"
#include "Bts2Ipc_MessageWrapper_GEN.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"
#include "BtsUtils.h"
#include "FwErrmemPrint.h"
#include "TraceClasses.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/ObexConnectDisconnectGenivi.cpp.trc.h"
#endif
#endif

// if enabled connect and disconnect SMs shall be updated and tested
// #define ENABLE_BUSY
#undef ENABLE_BUSY

namespace btstackif {
namespace genivi {

ObexConnectDisconnectGenivi::ObexConnectDisconnectGenivi() :
_callback(0),
_controlIf(0),
_timerPoolIf(0),
_switchBluetoothIf(0),
_deviceManagerIf(0),
_serviceSearchIf(0),
_objectPathManagerIf(0),
_connectDisconnectDataList(),
_statusUpdateTimeout(2000),
_testTriggerIgnoreProtocolAdded(false)
{
}

ObexConnectDisconnectGenivi::~ObexConnectDisconnectGenivi()
{
   _callback = 0;
   _controlIf = 0;
   _timerPoolIf = 0;
   _switchBluetoothIf = 0;
   _deviceManagerIf = 0;
   _serviceSearchIf = 0;
   _objectPathManagerIf = 0;
}

void ObexConnectDisconnectGenivi::reset(void)
{
   // stop and release all timer
   for(::std::map< BTSProtocolBaseEntry, ObexConnectDisconnectDataGenivi >::iterator it = _connectDisconnectDataList.begin(); it != _connectDisconnectDataList.end(); ++it)
   {
      releaseTimer(it->second.statusUpdateTimer);
   }
   _connectDisconnectDataList.clear();
   _testTriggerIgnoreProtocolAdded = false;

   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);
   _objectPathManagerIf->removeAllProtocol2ObjectPathMappings();
}

void ObexConnectDisconnectGenivi::setCallback(IN IProtocolManagerCallback* callback)
{
   _callback = callback;

   FW_ERRMEM_ASSERT(0 != _callback);
}

void ObexConnectDisconnectGenivi::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_ERRMEM_ASSERT(0 != _controlIf);
}

void ObexConnectDisconnectGenivi::setTimerPoolIf(IN ITimerPool* timerPool)
{
   _timerPoolIf = timerPool;

   FW_ERRMEM_ASSERT(0 != _timerPoolIf);
}

void ObexConnectDisconnectGenivi::prepareProtocolConnect(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   // simulate some states: protocol added + status updated
   // DBUS service availability is not needed
   BTSHandleIpc2BtsMessageItem messageItem;
   _callback->protocolAdded(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName);
   FW_ERRMEM_ASSERT(false == messageItem.deleteMessage);
   FW_ERRMEM_ASSERT(0 == messageItem.message);

   // initial status is always disconnected => no action required

   // check if all data is complete
   checkForDataComplete(bts2IpcMsgList, bts2AppMsgList, messageItem, *_callback, address, protocol, sppInstance, uuid, masInstance, masName);
   FW_ERRMEM_ASSERT(false == messageItem.deleteMessage);
   FW_ERRMEM_ASSERT(0 == messageItem.message);
}

void ObexConnectDisconnectGenivi::setDeviceAvailability(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const bool available)
{
   (void)(available);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   checkForDataComplete(bts2IpcMsgList, bts2AppMsgList, messageItem, *_callback, address, protocol, sppInstance, uuid, masInstance, masName);
}

bool ObexConnectDisconnectGenivi::isServiceAvailable(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const
{
   (void)(address);
   (void)(protocol);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masInstance);
   (void)(masName);

   // all needed DBUS services are available (DBUS server application is up and running)
   return true;
}

void ObexConnectDisconnectGenivi::waitForServiceAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(address);
   (void)(protocol);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masInstance);
   (void)(masName);

   // isServiceAvailable() was answered with true, therefore this function shall never be called
   FW_ERRMEM_ASSERT_ALWAYS();
}

bool ObexConnectDisconnectGenivi::isServiceSearchNeeded(OUT BTSSearchType& searchType, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const
{
   (void)(address);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masInstance);
   (void)(masName);

   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_callback);

   // in case of PBAP/MAP we will always do a service search
   if(BTS_PROTO_PIM == protocol)
   {
      searchType = BTS_SEARCH_PBAP;
      return true;
   }
   else if(BTS_PROTO_MSG == protocol)
   {
      searchType = BTS_SEARCH_MAP;

      // if there is already 1 MAS instance connected then no service search is needed
      return (false == isAnyMasInstanceConnected(*_callback, address));
   }

   FW_ERRMEM_ASSERT_ALWAYS();
   return false;
}

bool ObexConnectDisconnectGenivi::isSettingUuidNeeded(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const
{
   (void)(address);
   (void)(protocol);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masInstance);
   (void)(masName);

   // not needed
   return false;
}

void ObexConnectDisconnectGenivi::waitForProtocolAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(address);
   (void)(protocol);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masInstance);
   (void)(masName);

   // not necessary
   FW_ERRMEM_ASSERT_ALWAYS();
}

void ObexConnectDisconnectGenivi::connect(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName)
{
   (void)(bts2AppMsgList);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masName);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_switchBluetoothIf);

   createCreateSessionRequestMsg(bts2IpcMsgList, address, _switchBluetoothIf->getLocalAddress(), protocol, masInstance);
}

void ObexConnectDisconnectGenivi::disconnect(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const bool pauseBtStreaming)
{
   (void)(bts2AppMsgList);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masName);
   (void)(pauseBtStreaming);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   BTSObjectPath session;

   if(BTS_PROTO_PIM == protocol)
   {
      if(false == _objectPathManagerIf->getObjectPath4Protocol(session, address, protocol))
      {
         FW_ERRMEM_ASSERT_ALWAYS();
         return;
      }
   }
   else if(BTS_PROTO_MSG == protocol)
   {
      if(false == _objectPathManagerIf->getObjectPath4Protocol(session, address, protocol, masInstance))
      {
         FW_ERRMEM_ASSERT_ALWAYS();
         return;
      }
   }
   else
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   createRemoveSessionRequestMsg(bts2IpcMsgList, address, protocol, masInstance, session);
}

void ObexConnectDisconnectGenivi::cancel(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName)
{
   (void)(bts2AppMsgList);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masName);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_switchBluetoothIf);

   // sending of disconnect during ongoing connect disabled because extension of SM is necessary
   createCancelSessionRequestMsg(bts2IpcMsgList, address, _switchBluetoothIf->getLocalAddress(), protocol, masInstance);
}

void ObexConnectDisconnectGenivi::accept(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const bool accept)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(address);
   (void)(protocol);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masInstance);
   (void)(masName);
   (void)(accept);

   // no remote connect for PBAP/MAP
   FW_ERRMEM_ASSERT_ALWAYS();
}

void ObexConnectDisconnectGenivi::sendVirtualFailedConnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSIpcCommonErrorCode errorCode) const
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masName);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   Ipc2Bts_CreateSessionResult* msg = ptrNew_Ipc2Bts_CreateSessionResult();
   if(0 != msg)
   {
      msg->setBDAddress(address);
      msg->setProtocolId(protocol);
      msg->setInstanceId(masInstance);
      msg->setIpcCommonErrorCode(errorCode);
   }

   _controlIf->sendInternalIpc2BtsMessage(msg, true);
}

void ObexConnectDisconnectGenivi::sendVirtualFailedDisconnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSIpcCommonErrorCode errorCode) const
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masName);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   BTSObjectPath session;

   if(BTS_PROTO_PIM == protocol)
   {
      if(false == _objectPathManagerIf->getObjectPath4Protocol(session, address, protocol))
      {
         FW_ERRMEM_ASSERT_ALWAYS();
         return;
      }
   }
   else if(BTS_PROTO_MSG == protocol)
   {
      if(false == _objectPathManagerIf->getObjectPath4Protocol(session, address, protocol, masInstance))
      {
         FW_ERRMEM_ASSERT_ALWAYS();
         return;
      }
   }
   else
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   Ipc2Bts_RemoveSessionResult* msg = ptrNew_Ipc2Bts_RemoveSessionResult();
   if(0 != msg)
   {
      msg->setBDAddress(address);
      msg->setProtocolId(protocol);
      msg->setInstanceId(masInstance);
      msg->setIpcCommonErrorCode(errorCode);
   }

   _controlIf->sendInternalIpc2BtsMessage(msg, true);
}

void ObexConnectDisconnectGenivi::sendVirtualConnectedUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const bool connected, IN const BTSIpcCommonErrorCode errorCode)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masName);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   BTSObjectPath session;

   if(BTS_PROTO_PIM == protocol)
   {
      if(false == _objectPathManagerIf->getObjectPath4Protocol(session, address, protocol))
      {
         FW_ERRMEM_ASSERT_ALWAYS();
         return;
      }
   }
   else if(BTS_PROTO_MSG == protocol)
   {
      if(false == _objectPathManagerIf->getObjectPath4Protocol(session, address, protocol, masInstance))
      {
         FW_ERRMEM_ASSERT_ALWAYS();
         return;
      }
   }
   else
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   if(true == connected)
   {
      // note: virtual message for Ipc2Bts_AddProtocolObjectPathMapping is not possible because session object path is needed
      FW_ERRMEM_ASSERT_ALWAYS();
   }
   else
   {
      Ipc2Bts_DelProtocolObjectPathMapping* msg1 = ptrNew_Ipc2Bts_DelProtocolObjectPathMapping();
      if(0 != msg1)
      {
         msg1->setObjPath(session);
         msg1->setIpcCommonErrorCode(errorCode);
      }

      _controlIf->sendInternalIpc2BtsMessage(msg1, true);

      Ipc2Bts_RemoveSessionResult* msg2 = ptrNew_Ipc2Bts_RemoveSessionResult();
      if(0 != msg2)
      {
         msg2->setBDAddress(address);
         msg2->setProtocolId(protocol);
         msg2->setInstanceId(masInstance);
         msg2->setIpcCommonErrorCode(errorCode);
      }

      _controlIf->sendInternalIpc2BtsMessage(msg2, true);
   }
}

IProtocolManagerRequest* ObexConnectDisconnectGenivi::getRequestIf(void)
{
   return this;
}

void ObexConnectDisconnectGenivi::setSwitchBluetoothIf(IN ISwitchBluetooth* switchBluetoothIf)
{
   _switchBluetoothIf = switchBluetoothIf;

   FW_ERRMEM_ASSERT(0 != _switchBluetoothIf);
}

void ObexConnectDisconnectGenivi::setDeviceManagerIf(IN IDeviceManager* deviceManagerIf)
{
   _deviceManagerIf = deviceManagerIf;

   FW_ERRMEM_ASSERT(0 != _deviceManagerIf);
}

void ObexConnectDisconnectGenivi::setServiceSearchIf(IN IServiceSearch* serviceSearchIf)
{
   _serviceSearchIf = serviceSearchIf;

   FW_ERRMEM_ASSERT(0 != _serviceSearchIf);
}

void ObexConnectDisconnectGenivi::setObjectPathManagerIf(IN IObjectPathManagerGenivi* objectPathManager)
{
   _objectPathManagerIf = objectPathManager;

   FW_ERRMEM_ASSERT(0 != _objectPathManagerIf);
}

void ObexConnectDisconnectGenivi::protocolAdded(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSObjectPath& protocolObjectPath, IN const BTSProtocolId protocol, IN const BTSMasInstanceId masInstance)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   ETG_TRACE_USR3((" protocolAdded: protocol=%d address=%20s protocolObjectPath=%40s masInstance=%u", protocol, address.c_str(), protocolObjectPath.c_str(), masInstance));

   //===================================================================================================================
   // debug section start
   if(true == _testTriggerIgnoreProtocolAdded)
   {
      _testTriggerIgnoreProtocolAdded = false;
      ETG_TRACE_USR1((" protocolAdded: ignore"));
      return;
   }
   // debug section end
   //===================================================================================================================

   if(BTS_PROTO_PIM == protocol)
   {
      // store mapping
      _objectPathManagerIf->setProtocol2ObjectPath(address, protocol, protocolObjectPath);
   }
   else if(BTS_PROTO_MSG == protocol)
   {
      // store mapping
      _objectPathManagerIf->setProtocol2ObjectPath(address, protocol, masInstance, protocolObjectPath);
   }
   else
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   // if this function is triggered then the session was created successfully (protocol connected)

   const BTSSppInstanceId sppInstance(0);
   const BTSUuid uuid;
   BTSMasInstanceName masName;
   getMasInstanceName(masName, address, protocol, masInstance);

   // stop timer
   ObexConnectDisconnectDataGenivi& entry = checkDataList(address, protocol, masInstance);
   stopTimer(entry.statusUpdateTimer);

   // forward new connection status
   const BTSConnectionStatus connStatus(BTS_CONN_CONNECTED);
   const BTSDisconnectReason reason(BTS_DISCONNECT_REASON_LAST);
   (void)_callback->updateStatus(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName, connStatus, reason);
}

void ObexConnectDisconnectGenivi::protocolRemoved(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& protocolObjectPath)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   // service object path was removed
   BTSBDAddress address;
   BTSProtocolId protocol;
   BTSMasInstanceId masInstance(0);
   if(false == _objectPathManagerIf->getProtocol4ObjectPath(address, protocol, masInstance, protocolObjectPath))
   {
      // ignore
      return;
   }

   if((BTS_PROTO_PIM != protocol) && (BTS_PROTO_MSG != protocol))
   {
      // ignore
      return;
   }

   ETG_TRACE_USR3((" protocolRemoved: protocolObjectPath=%s", protocolObjectPath.c_str()));

   // if this function is triggered then the session was destroyed successfully (protocol disconnected)

   const BTSSppInstanceId sppInstance(0);
   const BTSUuid uuid;
   BTSMasInstanceName masName;
   getMasInstanceName(masName, address, protocol, masInstance);

   // stop timer
   ObexConnectDisconnectDataGenivi& entry = checkDataList(address, protocol, masInstance);
   stopTimer(entry.statusUpdateTimer);

   // forward new connection status
   const BTSConnectionStatus connStatus(BTS_CONN_DISCONNECTED);
   const BTSDisconnectReason reason(BTS_DISCONNECT_REASON_LAST);
   (void)_callback->updateStatus(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName, connStatus, reason);

   _objectPathManagerIf->removeProtocol2ObjectPath(protocolObjectPath);
}

void ObexConnectDisconnectGenivi::handleCreateSessionResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, OUT bool& connectFailed, OUT BTSBDAddress& failedAddress, IN const BTSBDAddress& remoteAddress, IN const BTSProtocolId protocol, IN const BTSMasInstanceId masInstance, IN const BTSObjectPath& session, IN const BTSIpcCommonErrorCode errorCode)
{
   (void)(session);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   ETG_TRACE_USR3((" handleCreateSessionResult: remoteAddress=%s", remoteAddress.c_str()));

   const BTSSppInstanceId sppInstance(0);
   const BTSUuid uuid;
   BTSMasInstanceName masName;
   getMasInstanceName(masName, remoteAddress, protocol, masInstance);
   BTSRequestResult result(BTS_REQ_FAILED);

   // check error code
   if(BTS_IPC_SUCCESS == errorCode)
   {
      result = BTS_REQ_SUCCESS;
   }
   else if(BTS_IPC_BUSY == errorCode)
   {
#ifdef ENABLE_BUSY
      result = BTS_REQ_BUSY;
#endif
   }
   else if(BTS_IPC_RETRY_ABORTED == errorCode)
   {
      result = BTS_REQ_CONNECT_ABORTED;
   }
   else
   {
      // other error codes
   }

   // update connecting before result in case of successful result
   if(BTS_IPC_SUCCESS == errorCode)
   {
      _callback->connecting(bts2IpcMsgList, bts2AppMsgList, messageItem, remoteAddress, protocol, sppInstance, uuid, masInstance, masName);

      // check for starting timer to wait for status update (protocol added)
      checkForStartingStatusTimer(remoteAddress, protocol, masInstance);
   }

   // forward connect result
   if(true == _callback->connectResult(bts2IpcMsgList, bts2AppMsgList, messageItem, remoteAddress, protocol, sppInstance, uuid, masInstance, masName, result))
   {
      connectFailed = true;
      failedAddress = remoteAddress;
   }
}

void ObexConnectDisconnectGenivi::updateConnectStatus(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSStatusCode status)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   if((BTS_PROTO_PIM != protocol) && (BTS_PROTO_MSG != protocol))
   {
      return;
   }

   ETG_TRACE_USR3((" updateConnectStatus: address=%s", address.c_str()));

   // optional information, contains e.g. page timeout information

   // forward only in case of failed connect
   if(BTS_STATUS_CODE_SUCCESS == status)
   {
      return;
   }

   // forward failed connect reason
   const BTSSppInstanceId sppInstance(0);
   const BTSUuid uuid;
   BTSMasInstanceId masInstance(0);
   BTSMasInstanceName masName;
   BTSDisconnectReason reason(BTS_DISCONNECT_REASON_PROTO_CONNECT_FAILED);

   // check for page timeout, aborted, missing key
   if(BTS_STATUS_CODE_PAGE_TIMEOUT == status)
   {
      reason = BTS_DISCONNECT_REASON_OUT_OF_RANGE; // e.g. device is out of range
   }
   else if(BTS_STATUS_CODE_ABORTED == status)
   {
      reason = BTS_DISCONNECT_REASON_PROTO_CONNECT_ABORTED; // e.g. connect was aborted/cancelled
   }
   else if(BTS_STATUS_CODE_PIN_OR_KEY_MISSING == status)
   {
      reason = BTS_DISCONNECT_REASON_MISSING_LINK_KEY; // e.g. pairing information on BT device was removed
   }

   // we need matching masInstance + masName
   if(true == findMatchingConnectMasInstance(masInstance, masName, *_callback, address, protocol))
   {
      _callback->updateFailedConnectReason(address, protocol, sppInstance, uuid, masInstance, masName, reason);
   }
   else
   {
      // not found due to whatever reason, do not forward
   }
}

void ObexConnectDisconnectGenivi::handleConnectInd(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSStatusCode status)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   if((BTS_PROTO_PIM != protocol) && (BTS_PROTO_MSG != protocol))
   {
      return;
   }

   ETG_TRACE_USR3((" handleConnectInd: address=%s", address.c_str()));

   // optional information

   // forward connect indication, use success or failed as result, reason for failed connect was forwarded via updateFailedConnectReason() before
   const BTSSppInstanceId sppInstance(0);
   const BTSUuid uuid;
   BTSMasInstanceId masInstance(0);
   BTSMasInstanceName masName;
   const BTSRequestResult result((BTS_STATUS_CODE_SUCCESS == status) ? BTS_REQ_SUCCESS : BTS_REQ_FAILED);

   // we need matching masInstance + masName
   if(true == findMatchingConnectMasInstance(masInstance, masName, *_callback, address, protocol))
   {
      _callback->connectIndication(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName, result);
   }
   else
   {
      // not found due to whatever reason, do not forward
   }
}

void ObexConnectDisconnectGenivi::handleRemoveSessionResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& remoteAddress, IN const BTSProtocolId protocol, IN const BTSMasInstanceId masInstance, IN const BTSIpcCommonErrorCode errorCode)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   ETG_TRACE_USR3((" handleRemoveSessionResult: remoteAddress=%s", remoteAddress.c_str()));

   const BTSSppInstanceId sppInstance(0);
   const BTSUuid uuid;
   BTSMasInstanceName masName;
   getMasInstanceName(masName, remoteAddress, protocol, masInstance);
   BTSRequestResult result(BTS_REQ_FAILED);

   // check error code
   if((BTS_IPC_SUCCESS == errorCode) || (BTS_IPC_NOT_CONNECTED == errorCode))
   {
      result = BTS_REQ_SUCCESS;
   }
   else if(BTS_IPC_BUSY == errorCode)
   {
#ifdef ENABLE_BUSY
      result = BTS_REQ_BUSY;
#endif
   }
   else if(BTS_IPC_RETRY_ABORTED == errorCode)
   {
      result = BTS_REQ_DISCONNECT_ABORTED;
   }
   else
   {
      // other error codes
      // use case: remove session request is received on BT stack side while session is already destroyed => error is returned => check if object path is available
      bool connected(false);
      BTSObjectPath session;

      if(BTS_PROTO_PIM == protocol)
      {
         connected = _objectPathManagerIf->getObjectPath4Protocol(session, remoteAddress, protocol);
      }
      else if(BTS_PROTO_MSG == protocol)
      {
         connected = _objectPathManagerIf->getObjectPath4Protocol(session, remoteAddress, protocol, masInstance);
      }
      else
      {
         FW_ERRMEM_ASSERT_ALWAYS();
      }

      if(false == connected)
      {
         result = BTS_REQ_SUCCESS;
      }
   }

   // update disconnecting before result in case of successful result
   if(BTS_IPC_SUCCESS == errorCode)
   {
      _callback->disconnecting(bts2IpcMsgList, bts2AppMsgList, messageItem, remoteAddress, protocol, sppInstance, uuid, masInstance, masName);
   }

   // forward disconnect result
   _callback->disconnectResult(bts2IpcMsgList, bts2AppMsgList, messageItem, remoteAddress, protocol, sppInstance, uuid, masInstance, masName, result);
}

void ObexConnectDisconnectGenivi::handleCancelSessionResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& remoteAddress, IN const BTSProtocolId protocol, IN const BTSMasInstanceId masInstance, IN const BTSIpcCommonErrorCode errorCode)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   ETG_TRACE_USR3((" handleCancelSessionResult: remoteAddress=%s", remoteAddress.c_str()));

   const BTSSppInstanceId sppInstance(0);
   const BTSUuid uuid;
   BTSMasInstanceName masName;
   getMasInstanceName(masName, remoteAddress, protocol, masInstance);
   BTSRequestResult result(BTS_REQ_FAILED);

   // check error code
   if((BTS_IPC_SUCCESS == errorCode) || (BTS_IPC_NOT_CONNECTED == errorCode))
   {
      result = BTS_REQ_SUCCESS;
   }
   else if(BTS_IPC_BUSY == errorCode)
   {
#ifdef ENABLE_BUSY
      result = BTS_REQ_BUSY;
#endif
   }
   else
   {
      // other error codes
      // use case: cancel session request is received on BT stack side while session is already destroyed => error is returned => check if object path is available
      bool connected(false);
      BTSObjectPath session;

      if(BTS_PROTO_PIM == protocol)
      {
         connected = _objectPathManagerIf->getObjectPath4Protocol(session, remoteAddress, protocol);
      }
      else if(BTS_PROTO_MSG == protocol)
      {
         connected = _objectPathManagerIf->getObjectPath4Protocol(session, remoteAddress, protocol, masInstance);
      }
      else
      {
         FW_ERRMEM_ASSERT_ALWAYS();
      }

      if(false == connected)
      {
         result = BTS_REQ_SUCCESS;
      }
   }

   // update disconnecting before result
   _callback->disconnecting(bts2IpcMsgList, bts2AppMsgList, messageItem, remoteAddress, protocol, sppInstance, uuid, masInstance, masName);

   // update current connection status
   bool connected(false);
   BTSObjectPath session;

   if(BTS_PROTO_PIM == protocol)
   {
      connected = _objectPathManagerIf->getObjectPath4Protocol(session, remoteAddress, protocol);
   }
   else if(BTS_PROTO_MSG == protocol)
   {
      connected = _objectPathManagerIf->getObjectPath4Protocol(session, remoteAddress, protocol, masInstance);
   }
   else
   {
      FW_ERRMEM_ASSERT_ALWAYS();
   }

   // stop timer
   ObexConnectDisconnectDataGenivi& entry = checkDataList(remoteAddress, protocol, masInstance);
   stopTimer(entry.statusUpdateTimer);

   const BTSConnectionStatus connStatus((true == connected) ? BTS_CONN_CONNECTED : BTS_CONN_DISCONNECTED);
   const BTSDisconnectReason reason(BTS_DISCONNECT_REASON_LAST);
   (void)_callback->updateStatus(bts2IpcMsgList, bts2AppMsgList, messageItem, remoteAddress, protocol, sppInstance, uuid, masInstance, masName, connStatus, reason);

   // forward disconnect result
   _callback->disconnectResult(bts2IpcMsgList, bts2AppMsgList, messageItem, remoteAddress, protocol, sppInstance, uuid, masInstance, masName, result);
}

void ObexConnectDisconnectGenivi::updateDisconnectReason(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSInternalDisconnectReason reason)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   if((BTS_PROTO_PIM != protocol) && (BTS_PROTO_MSG != protocol))
   {
      return;
   }

   ETG_TRACE_USR3((" updateDisconnectReason: address=%s", address.c_str()));

   // optional information, only called in case of successful disconnect

   // forward disconnect reason
   const BTSSppInstanceId sppInstance(0);
   const BTSUuid uuid;
   BTSMasInstanceId masInstance(0);
   BTSMasInstanceName masName;

   // we need matching masInstance + masName
   if(true == findMatchingDisconnectMasInstance(masInstance, masName, *_callback, address, protocol))
   {
      _callback->updateDisconnectReason(address, protocol, sppInstance, uuid, masInstance, masName, convertInternalDisconnectReason2Public(reason));
   }
   else
   {
      // not found due to whatever reason, do not forward
   }
}

void ObexConnectDisconnectGenivi::handleDisconnectInd(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSStatusCode status)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   if((BTS_PROTO_PIM != protocol) && (BTS_PROTO_MSG != protocol))
   {
      return;
   }

   ETG_TRACE_USR3((" handleDisconnectInd: address=%s", address.c_str()));

   // optional information

   // forward disconnect indication, use success or failed as result, disconnect reason was forwarded via updateDisconnectReason() before
   const BTSSppInstanceId sppInstance(0);
   const BTSUuid uuid;
   BTSMasInstanceId masInstance(0);
   BTSMasInstanceName masName;
   const BTSRequestResult result((BTS_STATUS_CODE_SUCCESS == status) ? BTS_REQ_SUCCESS : BTS_REQ_FAILED);

   // we need matching masInstance + masName
   if(true == findMatchingDisconnectMasInstance(masInstance, masName, *_callback, address, protocol))
   {
      _callback->disconnectIndication(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName, result);
   }
   else
   {
      // not found due to whatever reason, do not forward
   }
}

void ObexConnectDisconnectGenivi::setTriggerIgnoreProtocolAdded(IN const bool enable)
{
   _testTriggerIgnoreProtocolAdded = enable;
}

void ObexConnectDisconnectGenivi::handleExtendedTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSTimerId timerId)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   // check timer id
   for(::std::map< BTSProtocolBaseEntry, ObexConnectDisconnectDataGenivi >::iterator it = _connectDisconnectDataList.begin(); it != _connectDisconnectDataList.end(); ++it)
   {
      ObexConnectDisconnectDataGenivi& entry = it->second;

      // check for active timer
      if(true == entry.statusUpdateTimer.compare(timerId))
      {
         // timeout happened for status update (protocol added) => send virtual status update
         const BTSBDAddress& address(it->first.deviceAddress);
         const BTSProtocolId protocol(it->first.protocolId);
         const BTSMasInstanceId masInstance(it->first.masInstanceId);
         BTSMasInstanceName masName;
         getMasInstanceName(masName, address, protocol, masInstance);
         const BTSSppInstanceId sppInstance(0);
         const BTSUuid uuid;

         // forward new connection status
         const BTSConnectionStatus connStatus(BTS_CONN_DISCONNECTED);
         const BTSDisconnectReason reason(BTS_DISCONNECT_REASON_LAST);
         (void)_callback->updateStatus(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName, connStatus, reason);

         return;
      }
   }
}

void ObexConnectDisconnectGenivi::createCreateSessionRequestMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& remoteAddress, IN const BTSBDAddress& localAddress, IN const BTSProtocolId protocolId, IN const BTSMasInstanceId instance) const
{
   Bts2Ipc_CreateSession* msg = ptrNew_Bts2Ipc_CreateSession();
   if(0 != msg)
   {
      msg->setBDAddress(remoteAddress);
      msg->setSourceAddress(localAddress);
      msg->setProtocolId(protocolId);
      msg->setInstanceId(instance);
      msg->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg);
   }
}

void ObexConnectDisconnectGenivi::createRemoveSessionRequestMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& remoteAddress, IN const BTSProtocolId protocolId, IN const BTSMasInstanceId instance, IN const BTSObjectPath& session) const
{
   Bts2Ipc_RemoveSession* msg = ptrNew_Bts2Ipc_RemoveSession();
   if(0 != msg)
   {
      msg->setBDAddress(remoteAddress);
      msg->setProtocolId(protocolId);
      msg->setInstanceId(instance);
      msg->setSession(session);
      msg->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg);
   }
}

void ObexConnectDisconnectGenivi::createCancelSessionRequestMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& remoteAddress, IN const BTSBDAddress& localAddress, IN const BTSProtocolId protocolId, IN const BTSMasInstanceId instance) const
{
   Bts2Ipc_CancelSession* msg = ptrNew_Bts2Ipc_CancelSession();
   if(0 != msg)
   {
      msg->setBDAddress(remoteAddress);
      msg->setSourceAddress(localAddress);
      msg->setProtocolId(protocolId);
      msg->setInstanceId(instance);
      msg->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg);
   }
}

void ObexConnectDisconnectGenivi::checkForDataComplete(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const
{
   // get data entry
   ProtocolManagerData& entry = callback.getEntry(address, protocol, sppInstance, uuid, masInstance, masName);

   // get stored availability information
   const bool oldAvailable(entry.info.getBit(ProtocolManagerData::PROTOCOL_AVAILABLE));

   // get new availability information
   bool newAvailable(true);

   if(false == isDeviceAvailable(address))
   {
      newAvailable = false;
   }
   else if(false == entry.info.getBit(ProtocolManagerData::PROTOCOL_ADDED))
   {
      newAvailable = false;
   }

   // indicate only changed states
   if(oldAvailable != newAvailable)
   {
      ProtocolManagerData::ProtocolAvailableInfo protocolAvailableInfo;
      protocolAvailableInfo.setBit(ProtocolManagerData::PROTOCOL_AVAILABLE_LEVEL_1);
      protocolAvailableInfo.setBit(ProtocolManagerData::PROTOCOL_AVAILABLE_LEVEL_2);

      if(true == newAvailable)
      {
         callback.protocolAvailable(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName, protocolAvailableInfo);
      }
      else
      {
         callback.protocolUnavailable(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName, protocolAvailableInfo);
      }
   }
}

bool ObexConnectDisconnectGenivi::isDeviceAvailable(IN const BTSBDAddress& address) const
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_deviceManagerIf);

   return _deviceManagerIf->isDeviceAvailable(address);
}

bool ObexConnectDisconnectGenivi::findMatchingConnectMasInstance(OUT BTSMasInstanceId& masInstance, OUT BTSMasInstanceName& masName, IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol) const
{
   if(BTS_PROTO_MSG != protocol)
   {
      // masInstance + masName only needed for MAP, for all other protocols we can continue with default values
      return true;
   }

   const ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >& protocolList = callback.getProtocolList();

   // only 1 MAP connect to the same device at the same time is allowed, connect indication happens only in case of ongoing MAP connect
   for(::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = protocolList.begin(); it != protocolList.end(); ++it)
   {
      const BTSProtocolBaseEntry& keyData = it->first;

      if((address == keyData.deviceAddress) && (protocol == keyData.protocolId))
      {
         const ProtocolManagerData& entry = it->second;

         if(0 != entry.sm.getConnectSm())
         {
            // MAP connect is only started after SDP search was successful => MAS instance id and name already set
            masInstance = keyData.masInstanceId;
            getMasInstanceName(masName, address, protocol, masInstance);

            return true;
         }
      }
   }

   return false;
}

bool ObexConnectDisconnectGenivi::findMatchingDisconnectMasInstance(OUT BTSMasInstanceId& masInstance, OUT BTSMasInstanceName& masName, IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol) const
{
   if(BTS_PROTO_MSG != protocol)
   {
      // masInstance + masName only needed for MAP, for all other protocols we can continue with default values
      return true;
   }

   const ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >& protocolList = callback.getProtocolList();

   // disconnect indication happens during local MAP disconnect, remote MAP disconnect, local device disconnect, connection loss

   unsigned int count(0);
   for(::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = protocolList.begin(); it != protocolList.end(); ++it)
   {
      const BTSProtocolBaseEntry& keyData = it->first;

      if((address == keyData.deviceAddress) && (protocol == keyData.protocolId))
      {
         count++;
      }
   }

   if(1 == count)
   {
      for(::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = protocolList.begin(); it != protocolList.end(); ++it)
      {
         const BTSProtocolBaseEntry& keyData = it->first;

         if((address == keyData.deviceAddress) && (protocol == keyData.protocolId))
         {
            masInstance = keyData.masInstanceId;
            getMasInstanceName(masName, address, protocol, masInstance);

            return true;
         }
      }
   }

   return false;
}

void ObexConnectDisconnectGenivi::getMasInstanceName(OUT BTSMasInstanceName& instanceName, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSMasInstanceId instanceId) const
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   _callback->getMasInstanceNameInternal(instanceName, address, protocol, instanceId);
}

bool ObexConnectDisconnectGenivi::isAnyMasInstanceConnected(IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address) const
{
   const ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >& protocolList = callback.getProtocolList();

   // check list
   for(::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = protocolList.begin(); it != protocolList.end(); ++it)
   {
      const BTSProtocolBaseEntry& key = it->first;

      if((address == key.deviceAddress) && (BTS_PROTO_MSG == key.protocolId))
      {
         const ProtocolManagerData& data = it->second;

         if(true == data.connected)
         {
            return true;
         }
      }
   }

   return false;
}

void ObexConnectDisconnectGenivi::checkForStartingStatusTimer(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSMasInstanceId masInstance)
{
   // TODO: check how to stop timer during crash of Evolution stack

   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   /*
    * we have to handle following situation:
    * - MAP/PBAP connect is triggered and sent to Evolution stack
    * - profile connection is successful
    * - but remote phone disconnects profile connection immediately
    * - DBUS method call is answered with method return (success)
    * - but Evolution stack does not emit InterfacesAdded signal that is used as status update internally
    * - to implement a workaround we will start a timer to wait for this signal
    * - after timeout we will continue with failed connection status
    */

   // check if object path is already available for given address + protocol + masInstance
   BTSObjectPath session;
   bool available(false);

   if(BTS_PROTO_PIM == protocol)
   {
      available = _objectPathManagerIf->getObjectPath4Protocol(session, address, protocol);
   }
   else if(BTS_PROTO_MSG == protocol)
   {
      available = _objectPathManagerIf->getObjectPath4Protocol(session, address, protocol, masInstance);
   }
   else
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   if(false == available)
   {
      // start timer
      ObexConnectDisconnectDataGenivi& entry = checkDataList(address, protocol, masInstance);
      startTimer(entry.statusUpdateTimer, _statusUpdateTimeout);
   }
}

void ObexConnectDisconnectGenivi::startTimer(IN ExtendedTimerEntry& timer, IN const BTSTimeValue timeout)
{
   ETG_TRACE_USR3((" startTimer: timeout=%u", timeout));

   FW_ERRMEM_IF_NULL_PTR_RETURN(_timerPoolIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   timer.setTimerPool(_timerPoolIf);
   timer.start(_controlIf, this, timeout);
}

void ObexConnectDisconnectGenivi::stopTimer(IN ExtendedTimerEntry& timer) const
{
   ETG_TRACE_USR3((" stopTimer"));

   timer.stop();

   // do not release timer
}

void ObexConnectDisconnectGenivi::releaseTimer(IN ExtendedTimerEntry& timer) const
{
   ETG_TRACE_USR3((" releaseTimer"));

   timer.release();
}

bool ObexConnectDisconnectGenivi::isTimerActive(IN const ExtendedTimerEntry& timer) const
{
   return timer.isActive();
}

ObexConnectDisconnectDataGenivi& ObexConnectDisconnectGenivi::checkDataList(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSMasInstanceId masInstance)
{
   BTSProtocolBaseEntry entry;
   entry.deviceAddress = address;
   entry.protocolId = protocol;
   entry.masInstanceId = masInstance;

   return _connectDisconnectDataList[entry];
}

} //genivi
} //btstackif
