/**
 * @file LocalBluetoothNameGenivi.cpp
 *
 * @par SW-Component
 * State machine for setting local Bluetooth name
 *
 * @brief Implementation of Genivi setting local Bluetooth name state machine.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of Genivi setting local Bluetooth name state machine.
 */

#include "LocalBluetoothNameGenivi.h"
#include "ILocalBluetoothNameCallback.h"
#include "IBasicControl.h"
#include "FwErrmemPrint.h"
#include "Bts2Ipc_MessageWrapper_GEN.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"

namespace btstackif {
namespace genivi {

LocalBluetoothNameGenivi::LocalBluetoothNameGenivi() :
_callback(0),
_controlIf(0),
_adapterDataReceived(false)
{
}

LocalBluetoothNameGenivi::~LocalBluetoothNameGenivi()
{
   _callback = 0;
   _controlIf = 0;
}

void LocalBluetoothNameGenivi::reset(void)
{
   _adapterDataReceived = false;
}

void LocalBluetoothNameGenivi::setCallback(IN ILocalBluetoothNameCallback* callback)
{
   _callback = callback;

   FW_ERRMEM_ASSERT(0 != _callback);
}

void LocalBluetoothNameGenivi::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_ERRMEM_ASSERT(0 != _controlIf);
}

void LocalBluetoothNameGenivi::setLocalName(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSDeviceName& name)
{
   (void)(bts2AppMsgList);

   Bts2Ipc_SetAdapterAlias* msg = ptrNew_Bts2Ipc_SetAdapterAlias();
   if(0 != msg)
   {
      msg->setAlias(name);
      msg->setResponseMessageFlag(true);
      bts2IpcMsgList.push_back(msg);
   }
}

void LocalBluetoothNameGenivi::sendVirtualLocalNameUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSDeviceName& name, IN const BTSIpcCommonErrorCode errorCode)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   Ipc2Bts_AdapterAliasUpdate* msg = ptrNew_Ipc2Bts_AdapterAliasUpdate();
   if(0 != msg)
   {
      msg->setAlias(name);
      msg->setIpcCommonErrorCode(errorCode);
   }
   _controlIf->sendInternalIpc2BtsMessage(msg);
}

ILocalBluetoothNameRequest* LocalBluetoothNameGenivi::getRequestIf(void)
{
   return this;
}

void LocalBluetoothNameGenivi::handleLocalNameUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDeviceName& name, IN const BTSIpcCommonErrorCode errorCode, IN const bool response)
{
   (void)(response);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   _callback->updateLocalName(bts2IpcMsgList, bts2AppMsgList, messageItem, name, (BTS_IPC_SUCCESS == errorCode));
}

void LocalBluetoothNameGenivi::handleAdapterProperties(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDbusPropertyList& properties, IN const BTSIpcCommonErrorCode errorCode)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   // is triggered by a signal or method return => check error code
   if(BTS_IPC_SUCCESS == errorCode)
   {
      if(true == _adapterDataReceived)
      {
         // adapter data was already received => ignore this message
      }
      else
      {
         // first reception of adapter data => process
         _adapterDataReceived = true;

         // store information
         for(size_t i = 0; i < properties.size(); i++)
         {
            const ::ccdbusif::evolution::AdapterProperty property = (::ccdbusif::evolution::AdapterProperty)properties[i].propEnum;

            switch(property)
            {
               case ::ccdbusif::evolution::ADAPTER_ALIAS:
               {
                  setStackLocalBtName(properties[i].propData.getString());
                  break;
               }
               default:
               {
                  break;
               }
            }
         }
      }
   }
}

void LocalBluetoothNameGenivi::setStackLocalBtName(IN const BTSDeviceName& name)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   _callback->setBtStackName(name);
}

} //genivi
} //btstackif
