/**
 * @file HfpDeviceCapabilitiesGenivi.cpp
 *
 * @par SW-Component
 * State machine for HFP device capabilities
 *
 * @brief Implementation of Genivi HFP device capabilities state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of Genivi HFP device capabilities state machine.
 */

#include "HfpDeviceCapabilitiesGenivi.h"
#include "IHfpDeviceCapabilitiesCallback.h"
#include "IObjectPathManagerGenivi.h"
#include "Bts2Ipc_MessageWrapper_GEN.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"
#include "cc_dbus_if/EvolutionGeniviDbusParser.h"
#include "FwErrmemPrint.h"

namespace btstackif {
namespace genivi {

HfpDeviceCapabilitiesGenivi::HfpDeviceCapabilitiesGenivi() :
_callback(0),
_controlIf(0),
_objectPathManagerIf(0)
{
   _serviceStatusHfpManager.setEnabled(false); // set initially to disabled
   _serviceStatusHfpManager.setName("HfpManager");
   _serviceStatusHfpManager.setInterface((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_HFP_MANAGER);
   _serviceStatusHfpModem.setEnabled(false); // set initially to disabled
   _serviceStatusHfpModem.setName("HfpModem");
   _serviceStatusHfpModem.setInterface((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_HFP_MODEM);
   _serviceStatusHfpHandsfree.setEnabled(false); // set initially to disabled
   _serviceStatusHfpHandsfree.setName("HfpHandsfree");
   _serviceStatusHfpHandsfree.setInterface((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_HFP_HANDSFREE);
}

HfpDeviceCapabilitiesGenivi::~HfpDeviceCapabilitiesGenivi()
{
   _callback = 0;
   _controlIf = 0;
   _objectPathManagerIf = 0;
}

void HfpDeviceCapabilitiesGenivi::reset(void)
{
   if(true == _serviceStatusHfpManager.getEnabled())
   {
      _serviceStatusHfpManager.setAvailability(BTS_DBUS_SERVICE_NOT_AVAILABLE);
      _serviceStatusHfpModem.removeAllServiceInfos();
      _serviceStatusHfpHandsfree.removeAllServiceInfos();
   }

   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);
   _objectPathManagerIf->removeAllModem2ObjectPathMappings();
}

void HfpDeviceCapabilitiesGenivi::setCallback(IN IHfpDeviceCapabilitiesCallback* callback)
{
   _callback = callback;

   FW_ERRMEM_ASSERT(0 != _callback);
}

void HfpDeviceCapabilitiesGenivi::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   _serviceStatusHfpManager.setIpc2BtsSendIf(_controlIf);
   _serviceStatusHfpModem.setIpc2BtsSendIf(_controlIf);
   _serviceStatusHfpHandsfree.setIpc2BtsSendIf(_controlIf);
}

void HfpDeviceCapabilitiesGenivi::getModems(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   (void)(bts2AppMsgList);

   if(true == _serviceStatusHfpManager.getEnabled())
   {
      // HINT: during testing we observed that error "org.bluez.Error.NotReady" + "Resource Not Ready" is returned for GetModems() request; this is uncritical because no modems available during this moment

      Bts2Ipc_GetModems* msg = ptrNew_Bts2Ipc_GetModems();
      if(0 != msg)
      {
         msg->setResponseMessageFlag(true);

         bts2IpcMsgList.push_back(msg);
      }
   }
}

void HfpDeviceCapabilitiesGenivi::getModemProperties(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address)
{
   (void)(bts2AppMsgList);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   if(true == _serviceStatusHfpManager.getEnabled())
   {
      BTSObjectPath objPath;
      if(false == _objectPathManagerIf->getObjectPath4Modem(objPath, address))
      {
         FW_ERRMEM_ASSERT_ALWAYS();
         return;
      }

      Bts2Ipc_GetModemProperties* msg = ptrNew_Bts2Ipc_GetModemProperties();
      if(0 != msg)
      {
         msg->setBDAddress(address);
         msg->setModem(objPath);
         msg->setResponseMessageFlag(true);

         bts2IpcMsgList.push_back(msg);
      }
   }
}

void HfpDeviceCapabilitiesGenivi::getHandsfreeProperties(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address)
{
   (void)(bts2AppMsgList);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   if(true == _serviceStatusHfpManager.getEnabled())
   {
      BTSObjectPath objPath;
      if(false == _objectPathManagerIf->getObjectPath4Modem(objPath, address))
      {
         FW_ERRMEM_ASSERT_ALWAYS();
         return;
      }

      Bts2Ipc_GetHandsfreeProperties* msg = ptrNew_Bts2Ipc_GetHandsfreeProperties();
      if(0 != msg)
      {
         msg->setBDAddress(address);
         msg->setModem(objPath);
         msg->setResponseMessageFlag(true);

         bts2IpcMsgList.push_back(msg);
      }
   }
}

IHfpDeviceCapabilitiesRequest* HfpDeviceCapabilitiesGenivi::getRequestIf(void)
{
   return this;
}

void HfpDeviceCapabilitiesGenivi::setObjectPathManagerIf(IN IObjectPathManagerGenivi* objectPathManager)
{
   _objectPathManagerIf = objectPathManager;

   FW_ERRMEM_ASSERT(0 != _objectPathManagerIf);
}

void HfpDeviceCapabilitiesGenivi::setEnabled(IN const bool enabled, IN const BTSUserMode userMode)
{
   if(true == enabled)
   {
      if((BTS_USER_MODE_CONNECTION == userMode) ||
         (BTS_USER_MODE_CONN_WO_AGENT == userMode) ||
         (BTS_USER_MODE_ALL_WI_AGENT == userMode) ||
         (BTS_USER_MODE_ALL_WO_AGENT == userMode))
      {
         _serviceStatusHfpManager.setEnabled(enabled);
         _serviceStatusHfpModem.setEnabled(enabled);
         _serviceStatusHfpHandsfree.setEnabled(enabled);
      }
   }
   else
   {
      _serviceStatusHfpManager.setEnabled(enabled);
      _serviceStatusHfpModem.setEnabled(enabled);
      _serviceStatusHfpHandsfree.setEnabled(enabled);
   }
}

bool HfpDeviceCapabilitiesGenivi::getEnabled(void) const
{
   return _serviceStatusHfpManager.getEnabled();
}

DbusServiceStatus& HfpDeviceCapabilitiesGenivi::getServiceEntry(void)
{
   return _serviceStatusHfpManager;
}

void HfpDeviceCapabilitiesGenivi::addInterfaces(OUT ::std::vector< BTSDbusInterfaceItem >& dbusInterfaces, IN const BTSUserMode userMode)
{
   if((BTS_USER_MODE_CONNECTION == userMode) ||
      (BTS_USER_MODE_CONN_WO_AGENT == userMode) ||
      (BTS_USER_MODE_ALL_WI_AGENT == userMode) ||
      (BTS_USER_MODE_ALL_WO_AGENT == userMode))
   {
      if(true == _serviceStatusHfpManager.getEnabled())
      {
         ::ccdbusif::evolution::EvolutionGeniviDbusParser evoParser;

         _serviceStatusHfpManager.addServiceInfo(evoParser.getInterface2ObjectPath(::ccdbusif::evolution::IF_MANAGER));
         _serviceStatusHfpManager.setAvailability(BTS_DBUS_SERVICE_WAITING);
         BTSDbusInterfaceItem dbusItem;
         dbusItem.dbusInterface = (BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_HFP_MANAGER;
         dbusInterfaces.push_back(dbusItem);
      }
   }
}

BTSDbusServiceAvailability HfpDeviceCapabilitiesGenivi::getManagerAvailability(void) const
{
   return _serviceStatusHfpManager.getAvailability();
}

void HfpDeviceCapabilitiesGenivi::updateManagerAvailability(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDbusServiceAvailability availability)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   if(true == _serviceStatusHfpManager.getEnabled())
   {
      _serviceStatusHfpManager.setAvailability(availability);

      // indicate availability to higher layer
      if((BTS_DBUS_SERVICE_AVAILABLE == availability) || (BTS_DBUS_SERVICE_NOT_AVAILABLE == availability))
      {
         _callback->indicateManagerServiceAvailability(bts2IpcMsgList, bts2AppMsgList, messageItem, (BTS_DBUS_SERVICE_AVAILABLE == availability));
      }
      // else: intermediate state
   }
}

void HfpDeviceCapabilitiesGenivi::updateModemAvailability(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& objPath, IN const BTSDbusServiceAvailability availability)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   if(true == _serviceStatusHfpManager.getEnabled())
   {
      // given: modem object path
      // needed: address
      BTSBDAddress address;
      if(false == _objectPathManagerIf->getModem4ObjectPath(address, objPath))
      {
         // modem object path is created before proxy is created => therefore this error should never happen
         return;
      }

      _serviceStatusHfpModem.setAvailability(address, availability); // TODO: [low]: check for BTS_DBUS_SERVICE_NOT_AVAILABLE needed

      // indicate availability to higher layer
      if(BTS_DBUS_SERVICE_AVAILABLE == availability)
      {
         _callback->modemAvailable(bts2IpcMsgList, bts2AppMsgList, messageItem, address);
      }
      else if(BTS_DBUS_SERVICE_NOT_AVAILABLE == availability)
      {
         _callback->modemUnavailable(bts2IpcMsgList, bts2AppMsgList, messageItem, address);
      }
      // else: intermediate state
   }
}

void HfpDeviceCapabilitiesGenivi::updateHandsfreeAvailability(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& objPath, IN const BTSDbusServiceAvailability availability)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   if(true == _serviceStatusHfpManager.getEnabled())
   {
      // given: modem object path
      // needed: address
      BTSBDAddress address;
      if(false == _objectPathManagerIf->getModem4ObjectPath(address, objPath))
      {
         // hands-free object path is created before proxy is created => therefore this error should never happen
         return;
      }

      _serviceStatusHfpHandsfree.setAvailability(address, availability); // TODO: [low]: check for BTS_DBUS_SERVICE_NOT_AVAILABLE needed

      // indicate availability to higher layer
      if(BTS_DBUS_SERVICE_AVAILABLE == availability)
      {
         _callback->handsfreeInterfaceAvailable(bts2IpcMsgList, bts2AppMsgList, messageItem, address);
      }
      else if(BTS_DBUS_SERVICE_NOT_AVAILABLE == availability)
      {
         _callback->handsfreeInterfaceUnavailable(bts2IpcMsgList, bts2AppMsgList, messageItem, address);
      }
      // else: intermediate state
   }
}

void HfpDeviceCapabilitiesGenivi::handleAddedHfpInterface(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& objPath, IN const ::ccdbusif::evolution::Interface interface)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   if(true == _serviceStatusHfpManager.getEnabled())
   {
      // handle only hands-free interface
      if(::ccdbusif::evolution::IF_HANDSFREE != interface)
      {
         return;
      }

      // given: modem object path
      // needed: address
      BTSBDAddress address;
      if(false == _objectPathManagerIf->getModem4ObjectPath(address, objPath))
      {
         // related modem object path mapping must be available
         FW_ERRMEM_ASSERT_ALWAYS();
         return;
      }

      // check whether entry is available
      if(false == _serviceStatusHfpHandsfree.isEntryAvailable(address))
      {
         // DBUS service information has to be added
         _serviceStatusHfpHandsfree.addServiceInfo(address, objPath);
         // set to waiting because proxy was created
         _serviceStatusHfpHandsfree.setAvailability(address, BTS_DBUS_SERVICE_WAITING);
      }

      // indicate added hands-free interface to higher layer
      _callback->handsfreeInterfaceAdded(bts2IpcMsgList, bts2AppMsgList, messageItem, address);
   }
}

void HfpDeviceCapabilitiesGenivi::handleRemovedHfpInterface(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& objPath, IN const ::ccdbusif::evolution::Interface interface)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   if(true == _serviceStatusHfpManager.getEnabled())
   {
      // handle only hands-free interface
      if(::ccdbusif::evolution::IF_HANDSFREE != interface)
      {
         return;
      }

      // given: modem object path
      // needed: address
      BTSBDAddress address;
      if(false == _objectPathManagerIf->getModem4ObjectPath(address, objPath))
      {
         // no assert; maybe not available
         return;
      }

      // check whether entry is available
      if(true == _serviceStatusHfpHandsfree.isEntryAvailable(address))
      {
         // DBUS service information has to be removed
         _serviceStatusHfpHandsfree.removeServiceInfo(address);
      }

      // indicate removed hands-free interface to higher layer
      _callback->handsfreeInterfaceRemoved(bts2IpcMsgList, bts2AppMsgList, messageItem, address);
   }
}

void HfpDeviceCapabilitiesGenivi::handleHandsfreeFeaturesUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& modem, IN const BTSHfpSupportedFeatures& features)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   if(true == _serviceStatusHfpManager.getEnabled())
   {
      BTSBDAddress address;
      if(false == _objectPathManagerIf->getModem4ObjectPath(address, modem))
      {
         // related modem object path mapping must be available
         FW_ERRMEM_ASSERT_ALWAYS();
         return;
      }

      // check for received empty property value; this happens if we send a GetHandsfreeProperties() request; result contains an empty array for features property
      // if real value is empty then inband ringtone support is set as false (but no update happens)
      if(0 == features.getData())
      {
         return;
      }

      // forward to higher layer
      BTSHandsfreeAgFeatures agFeatures;

      if(true == features.getBit(::ccdbusif::evolution::HFP_HANDSFREE_FEATURES_VOICE_RECOGNITION)) { agFeatures.setBit(BTS_HFP_AG_FEATURE_VOICE_RECOGNITION); }
      if(true == features.getBit(::ccdbusif::evolution::HFP_HANDSFREE_FEATURES_ATTACH_VOICE_TAG)) { agFeatures.setBit(BTS_HFP_AG_FEATURE_ATTACH_VOICE_TAG); }
      if(true == features.getBit(::ccdbusif::evolution::HFP_HANDSFREE_FEATURES_3WAY)) { agFeatures.setBit(BTS_HFP_AG_FEATURE_3WAY); }
      if(true == features.getBit(::ccdbusif::evolution::HFP_HANDSFREE_FEATURES_ECNR)) { agFeatures.setBit(BTS_HFP_AG_FEATURE_ECNR); }
      if(true == features.getBit(::ccdbusif::evolution::HFP_HANDSFREE_FEATURES_INBAND_RING)) { agFeatures.setBit(BTS_HFP_AG_FEATURE_INBAND_RING); }
      if(true == features.getBit(::ccdbusif::evolution::HFP_HANDSFREE_FEATURES_REJECT_INCOMING_CALL)) { agFeatures.setBit(BTS_HFP_AG_FEATURE_REJECT_INCOMING_CALL); }
      if(true == features.getBit(::ccdbusif::evolution::HFP_HANDSFREE_FEATURES_ENHANCED_CALL_STATUS)) { agFeatures.setBit(BTS_HFP_AG_FEATURE_ENHANCED_CALL_STATUS); }
      if(true == features.getBit(::ccdbusif::evolution::HFP_HANDSFREE_FEATURES_ENHANCED_CALL_CONTROL)) { agFeatures.setBit(BTS_HFP_AG_FEATURE_ENHANCED_CALL_CONTROL); }
      if(true == features.getBit(::ccdbusif::evolution::HFP_HANDSFREE_FEATURES_EXTENDED_ERROR_CODES)) { agFeatures.setBit(BTS_HFP_AG_FEATURE_EXTENDED_ERROR_CODES); }
      if(true == features.getBit(::ccdbusif::evolution::HFP_HANDSFREE_FEATURES_CODEC_NEGOTIATION)) { agFeatures.setBit(BTS_HFP_AG_FEATURE_CODEC_NEGOTIATION); }
      if(true == features.getBit(::ccdbusif::evolution::HFP_HANDSFREE_FEATURES_HF_INDICATORS)) { agFeatures.setBit(BTS_HFP_AG_FEATURE_HF_INDICATORS); }
      if(true == features.getBit(::ccdbusif::evolution::HFP_HANDSFREE_FEATURES_ESCO_S4_AND_T2_SETTING_SUPPORTED)) { agFeatures.setBit(BTS_HFP_AG_FEATURE_ESCO_S4_AND_T2_SETTING_SUPPORTED); }

      _callback->updateHandsfreeFeatures(bts2IpcMsgList, bts2AppMsgList, messageItem, address, agFeatures);
   }
}

void HfpDeviceCapabilitiesGenivi::handleHandsfreeInbandRingingUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& modem, IN const bool inbandRingingEnabled)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   if(true == _serviceStatusHfpManager.getEnabled())
   {
      BTSBDAddress address;
      if(false == _objectPathManagerIf->getModem4ObjectPath(address, modem))
      {
         // related modem object path mapping must be available
         FW_ERRMEM_ASSERT_ALWAYS();
         return;
      }

      _callback->updateHandsfreeInbandRinging(bts2IpcMsgList, bts2AppMsgList, messageItem, address, inbandRingingEnabled);
   }
}

void HfpDeviceCapabilitiesGenivi::handleGetModemsResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode errorCode)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   if(true == _serviceStatusHfpManager.getEnabled())
   {
      _callback->getModemsResult(bts2IpcMsgList, bts2AppMsgList, messageItem, ((BTS_IPC_SUCCESS == errorCode) ? BTS_REQ_SUCCESS : BTS_REQ_FAILED));
   }
}

void HfpDeviceCapabilitiesGenivi::handleGetHandsfreePropertiesResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& modem, IN const BTSIpcCommonErrorCode errorCode)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   if(true == _serviceStatusHfpManager.getEnabled())
   {
      BTSBDAddress address;
      if(false == _objectPathManagerIf->getModem4ObjectPath(address, modem))
      {
         // related modem object path mapping must be available
         FW_ERRMEM_ASSERT_ALWAYS();
         return;
      }

      _callback->getHandsfreePropertiesResult(bts2IpcMsgList, bts2AppMsgList, messageItem, address, ((BTS_IPC_SUCCESS == errorCode) ? BTS_REQ_SUCCESS : BTS_REQ_FAILED));
   }
}

void HfpDeviceCapabilitiesGenivi::handleGetModemPropertiesResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& modem, IN const BTSIpcCommonErrorCode errorCode)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   if(true == _serviceStatusHfpManager.getEnabled())
   {
      BTSBDAddress address;
      if(false == _objectPathManagerIf->getModem4ObjectPath(address, modem))
      {
         // related modem object path mapping must be available
         FW_ERRMEM_ASSERT_ALWAYS();
         return;
      }

      _callback->getModemPropertiesResult(bts2IpcMsgList, bts2AppMsgList, messageItem, address, ((BTS_IPC_SUCCESS == errorCode) ? BTS_REQ_SUCCESS : BTS_REQ_FAILED));
   }
}

void HfpDeviceCapabilitiesGenivi::handleModemAdded(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSObjectPath& modem)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   if(true == _serviceStatusHfpManager.getEnabled())
   {
      // store modem object path mapping
      _objectPathManagerIf->setModem2ObjectPath(address, modem);

      // check whether entry is available
      if(false == _serviceStatusHfpModem.isEntryAvailable(address))
      {
         // DBUS service information has to be added
         _serviceStatusHfpModem.addServiceInfo(address, modem);
         // set to waiting because proxy was created
         _serviceStatusHfpModem.setAvailability(address, BTS_DBUS_SERVICE_WAITING);
      }

      // indicate added modem interface to higher layer
      _callback->modemAdded(bts2IpcMsgList, bts2AppMsgList, messageItem, address);
   }
}

void HfpDeviceCapabilitiesGenivi::handleModemRemoved(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& modem)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   if(true == _serviceStatusHfpManager.getEnabled())
   {
      // given: modem object path
      // needed: address
      BTSBDAddress address;
      if(false == _objectPathManagerIf->getModem4ObjectPath(address, modem))
      {
         // no assert; maybe not available
         return;
      }

      // check whether entry is available
      if(true == _serviceStatusHfpModem.isEntryAvailable(address))
      {
         // DBUS service information has to be removed
         _serviceStatusHfpModem.removeServiceInfo(address);
      }

      // remove modem object path mapping
      _objectPathManagerIf->removeModem2ObjectPath(modem);

      // indicate removed modem interface to higher layer
      _callback->modemRemoved(bts2IpcMsgList, bts2AppMsgList, messageItem, address);
   }
}

void HfpDeviceCapabilitiesGenivi::handleModemInterfacesUpdate(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& modem, IN const BTSDbusInterfaceList& interfaces)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(modem);
   (void)(interfaces);

   // object path mapping for hands-free interfaces is handled by handleAddedHfpInterface()/handleRemovedHfpInterface()
}

} //genivi
} //btstackif
