/**
 * @file HfpAvpConnectDisconnectGenivi.cpp
 *
 * @par SW-Component
 * State machine for HFP/AVP connect/disconnect
 *
 * @brief Implementation of Genivi HFP/AVP connect/disconnect state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of Genivi HFP/AVP connect/disconnect state machine.
 */

#include "HfpAvpConnectDisconnectGenivi.h"
#include "IProtocolManagerCallback.h"
#include "ProtocolManagerData.h"
#include "IBasicControl.h"
#include "IDeviceManager.h"
#include "IServiceSearch.h"
#include "IObjectPathManagerGenivi.h"
#include "Bts2Ipc_MessageWrapper_GEN.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"
#include "BtsUtils.h"
#include "EvolutionGeniviUtils.h"
#include "cc_dbus_if/EvolutionGeniviDbusParser.h"
#include "FwErrmemPrint.h"
#include "TraceClasses.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/HfpAvpConnectDisconnectGenivi.cpp.trc.h"
#endif
#endif

// if enabled connect and disconnect SMs shall be updated and tested
// #define ENABLE_BUSY
#undef ENABLE_BUSY

namespace btstackif {
namespace genivi {

HfpAvpConnectDisconnectGenivi::HfpAvpConnectDisconnectGenivi() :
_callback(0),
_controlIf(0),
_timerPoolIf(0),
_deviceManagerIf(0),
_serviceSearchIf(0),
_objectPathManagerIf(0),
_serviceStatusService(),
_serviceStatusSerial(),
_connectedServicesList(),
_tokenList(),
_connectDisconnectDataList(),
_pendingSerialObjectPathEntries(),
_pendingSerialAvailabilityEntries(),
_serviceSppServiceTimeout(5000)
{
   _serviceStatusService.setEnabled(true); // only used if any protocol connection is created
   _serviceStatusService.setName("Service");
   _serviceStatusService.setInterface((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_SERVICE);
   _serviceStatusSerial.setEnabled(true); // only used if any protocol connection is created
   _serviceStatusSerial.setName("Serial");
   _serviceStatusSerial.setInterface((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_SERIAL);
}

HfpAvpConnectDisconnectGenivi::~HfpAvpConnectDisconnectGenivi()
{
   _callback = 0;
   _controlIf = 0;
   _timerPoolIf = 0;
   _deviceManagerIf = 0;
   _serviceSearchIf = 0;
   _objectPathManagerIf = 0;
}

void HfpAvpConnectDisconnectGenivi::reset(void)
{
   _serviceStatusService.removeAllServiceInfos();
   _serviceStatusSerial.removeAllServiceInfos();
   _connectedServicesList.clear();
   _tokenList.clear();
   // stop and release all timer
   for(::std::map< BTSProtocolBaseEntry, HfpAvpConnectDisconnectDataGenivi >::iterator it = _connectDisconnectDataList.begin(); it != _connectDisconnectDataList.end(); ++it)
   {
      releaseTimer(it->second.serviceSppServiceTimer);
   }
   _connectDisconnectDataList.clear();
   _pendingSerialObjectPathEntries.clear();
   _pendingSerialAvailabilityEntries.clear();

   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);
   _objectPathManagerIf->removeAllProtocol2ObjectPathMappings();
}

void HfpAvpConnectDisconnectGenivi::setCallback(IN IProtocolManagerCallback* callback)
{
   _callback = callback;

   FW_ERRMEM_ASSERT(0 != _callback);
}

void HfpAvpConnectDisconnectGenivi::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   _serviceStatusService.setIpc2BtsSendIf(_controlIf);
   _serviceStatusSerial.setIpc2BtsSendIf(_controlIf);

   _controlIf->addTTYMonitorHandler(this, "/dev/bvd_tty");
}

void HfpAvpConnectDisconnectGenivi::setTimerPoolIf(IN ITimerPool* timerPool)
{
   _timerPoolIf = timerPool;

   FW_ERRMEM_ASSERT(0 != _timerPoolIf);
}

void HfpAvpConnectDisconnectGenivi::prepareProtocolConnect(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   if((BTS_PROTO_HFP == protocol) || (BTS_PROTO_AVP == protocol))
   {
      // nothing to do
   }
   else if(BTS_PROTO_SPP == protocol)
   {
      // simulate some states: protocol added + status updated
      // DBUS service availability is not needed
      BTSHandleIpc2BtsMessageItem messageItem;
      _callback->protocolAdded(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName);
      FW_ERRMEM_ASSERT(false == messageItem.deleteMessage);
      FW_ERRMEM_ASSERT(0 == messageItem.message);

      // initial status is always disconnected

      // check if all data is complete
      checkForDataComplete(bts2IpcMsgList, bts2AppMsgList, messageItem, *_callback, address, protocol, sppInstance, uuid, masInstance, masName);
      FW_ERRMEM_ASSERT(false == messageItem.deleteMessage);
      FW_ERRMEM_ASSERT(0 == messageItem.message);
   }
   else
   {
      FW_ERRMEM_ASSERT_ALWAYS();
   }
}

void HfpAvpConnectDisconnectGenivi::setDeviceAvailability(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const bool available)
{
   (void)(available);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   checkForDataComplete(bts2IpcMsgList, bts2AppMsgList, messageItem, *_callback, address, protocol, sppInstance, uuid, masInstance, masName);
}

bool HfpAvpConnectDisconnectGenivi::isServiceAvailable(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const
{
   (void)(address);
   (void)(protocol);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masInstance);
   (void)(masName);

   // all needed DBUS services are available (DBUS server application is up and running)
   return true;
}

void HfpAvpConnectDisconnectGenivi::waitForServiceAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(address);
   (void)(protocol);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masInstance);
   (void)(masName);

   // isServiceAvailable() was answered with true, therefore this function shall never be called
   FW_ERRMEM_ASSERT_ALWAYS();
}

bool HfpAvpConnectDisconnectGenivi::isServiceSearchNeeded(OUT BTSSearchType& searchType, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_objectPathManagerIf);

   (void)(sppInstance);
   (void)(masInstance);
   (void)(masName);

   if((BTS_PROTO_HFP == protocol) || (BTS_PROTO_AVP == protocol))
   {
      // not needed
   }
   else if(BTS_PROTO_SPP == protocol)
   {
      // if object path is available then we can continue without service search
      BTSObjectPath service;
      if(false == _objectPathManagerIf->getObjectPath4Protocol(service, address, protocol, uuid))
      {
         searchType = BTS_SEARCH_SPP;
         return true;
      }
      else
      {
         // object path is available => service search not needed
      }
   }
   else
   {
      FW_ERRMEM_ASSERT_ALWAYS();
   }

   return false;
}

bool HfpAvpConnectDisconnectGenivi::isSettingUuidNeeded(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const
{
   (void)(address);
   (void)(protocol);
   (void)(sppInstance);
   (void)(uuid);
   (void)(masInstance);
   (void)(masName);

   // not needed
   return false;
}

void HfpAvpConnectDisconnectGenivi::waitForProtocolAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(sppInstance);
   (void)(masInstance);
   (void)(masName);

   if((BTS_PROTO_HFP == protocol) || (BTS_PROTO_AVP == protocol))
   {
      // not necessary
      FW_ERRMEM_ASSERT_ALWAYS();
   }
   else if(BTS_PROTO_SPP == protocol)
   {
      HfpAvpConnectDisconnectDataGenivi& entry = checkDataList(address, protocol, uuid);

      // start timer if not active
      if(false == isTimerActive(entry.serviceSppServiceTimer))
      {
         startTimer(entry.serviceSppServiceTimer, _serviceSppServiceTimeout);
      }
   }
   else
   {
      FW_ERRMEM_ASSERT_ALWAYS();
   }
}

void HfpAvpConnectDisconnectGenivi::connect(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName)
{
   (void)(bts2AppMsgList);
   (void)(sppInstance);
   (void)(masInstance);
   (void)(masName);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   BTSObjectPath service;

   if((BTS_PROTO_HFP == protocol) || (BTS_PROTO_AVP == protocol))
   {
      if(false == _objectPathManagerIf->getObjectPath4Protocol(service, address, protocol))
      {
         FW_ERRMEM_ASSERT_ALWAYS();
         return;
      }
   }
   else if(BTS_PROTO_SPP == protocol)
   {
      if(false == _objectPathManagerIf->getObjectPath4Protocol(service, address, protocol, uuid))
      {
         FW_ERRMEM_ASSERT_ALWAYS();
         return;
      }
   }
   else
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   createConnectMsg(bts2IpcMsgList, address, protocol, uuid, service);
}

void HfpAvpConnectDisconnectGenivi::disconnect(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const bool pauseBtStreaming)
{
   (void)(bts2AppMsgList);
   (void)(sppInstance);
   (void)(masInstance);
   (void)(masName);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_deviceManagerIf);

   BTSObjectPath service;

   if((BTS_PROTO_HFP == protocol) || (BTS_PROTO_AVP == protocol))
   {
      if(false == _objectPathManagerIf->getObjectPath4Protocol(service, address, protocol))
      {
         FW_ERRMEM_ASSERT_ALWAYS();
         return;
      }

      if(BTS_PROTO_AVP == protocol)
      {
         BTSObjectPath objPath;
         if(false == _objectPathManagerIf->getObjectPath4Address(objPath, address))
         {
            // should never happen
            // FW_NORMAL_ASSERT_ALWAYS();
            return;
         }

         // if requested value already set, then ignore
         if(pauseBtStreaming != _deviceManagerIf->getEnableAvpPause(address))
         {
            createSetEnableAvpPauseMsg(bts2IpcMsgList, address, objPath, pauseBtStreaming);
         }
      }
   }
   else if(BTS_PROTO_SPP == protocol)
   {
      if(false == _objectPathManagerIf->getObjectPath4Protocol(service, address, protocol, uuid))
      {
         FW_ERRMEM_ASSERT_ALWAYS();
         return;
      }
   }
   else
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   createDisconnectMsg(bts2IpcMsgList, address, protocol, uuid, service);
}

void HfpAvpConnectDisconnectGenivi::cancel(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName)
{
   (void)(bts2AppMsgList);
   (void)(sppInstance);
   (void)(masInstance);
   (void)(masName);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   BTSObjectPath service;

   if((BTS_PROTO_HFP == protocol) || (BTS_PROTO_AVP == protocol))
   {
      if(false == _objectPathManagerIf->getObjectPath4Protocol(service, address, protocol))
      {
         FW_ERRMEM_ASSERT_ALWAYS();
         return;
      }
   }
   else if(BTS_PROTO_SPP == protocol)
   {
      if(false == _objectPathManagerIf->getObjectPath4Protocol(service, address, protocol, uuid))
      {
         FW_ERRMEM_ASSERT_ALWAYS();
         return;
      }
   }
   else
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   createDisconnectMsg(bts2IpcMsgList, address, protocol, uuid, service);
}

void HfpAvpConnectDisconnectGenivi::accept(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const bool accept)
{
   (void)(sppInstance);
   (void)(masInstance);
   (void)(masName);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   if((BTS_PROTO_HFP == protocol) || (BTS_PROTO_AVP == protocol) || (BTS_PROTO_SPP == protocol))
   {
      act_t token(0);

      if(true == getAndRemoveToken(token, address, protocol, uuid))
      {
         const BTSConfirmationMode mode((true == accept) ? BTS_CONFIRM_ACCEPT : BTS_CONFIRM_REJECT);
         createAuthorizeServiceResponseMsg(bts2IpcMsgList, address, token, mode);

         if(BTS_CONFIRM_REJECT == mode)
         {
            /*
             * follow scenario was observed:
             * - reject is sent immediately
             * - no Connecting is reported as Service State
             * - in this case we have 2 consecutive Service State updates with Disconnected
             * - this would cause ignoring second Disconnected in function handleServiceState()
             * - this would cause a remaining active state machine that blocks other requests
             * => set Connecting in case of reject
             */
            BTSHandleIpc2BtsMessageItem tmpMessageItem;
            bool connectFailed;
            BTSBDAddress failedAddress;
            BTSObjectPath service;
            if((BTS_PROTO_HFP == protocol) || (BTS_PROTO_AVP == protocol))
            {
               if(false == _objectPathManagerIf->getObjectPath4Protocol(service, address, protocol))
               {
                  FW_ERRMEM_ASSERT_ALWAYS();
                  return;
               }
            }
            else
            {
               if(false == _objectPathManagerIf->getObjectPath4Protocol(service, address, protocol, uuid))
               {
                  FW_ERRMEM_ASSERT_ALWAYS();
                  return;
               }
            }

            handleServiceState(bts2IpcMsgList, bts2AppMsgList, tmpMessageItem, connectFailed, failedAddress, service, BTS_SERVICE_STATE_CONNECTING, BTS_IPC_SUCCESS);
         }
      }
      else
      {
         // token not available - what happened?
         FW_ERRMEM_ASSERT_ALWAYS();
      }
   }
   else
   {
      FW_ERRMEM_ASSERT_ALWAYS();
   }
}

void HfpAvpConnectDisconnectGenivi::sendVirtualFailedConnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSIpcCommonErrorCode errorCode) const
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(sppInstance);
   (void)(masInstance);
   (void)(masName);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   BTSObjectPath service;

   if((BTS_PROTO_HFP == protocol) || (BTS_PROTO_AVP == protocol))
   {
      if(false == _objectPathManagerIf->getObjectPath4Protocol(service, address, protocol))
      {
         FW_ERRMEM_ASSERT_ALWAYS();
         return;
      }
   }
   else if(BTS_PROTO_SPP == protocol)
   {
      if(false == _objectPathManagerIf->getObjectPath4Protocol(service, address, protocol, uuid))
      {
         FW_ERRMEM_ASSERT_ALWAYS();
         return;
      }
   }
   else
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   Ipc2Bts_ConnectResult* msg = ptrNew_Ipc2Bts_ConnectResult();
   if(0 != msg)
   {
      msg->setService(service);
      msg->setIpcCommonErrorCode(errorCode);
   }

   _controlIf->sendInternalIpc2BtsMessage(msg, true);
}

void HfpAvpConnectDisconnectGenivi::sendVirtualFailedDisconnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSIpcCommonErrorCode errorCode) const
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(sppInstance);
   (void)(masInstance);
   (void)(masName);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   BTSObjectPath service;

   if((BTS_PROTO_HFP == protocol) || (BTS_PROTO_AVP == protocol))
   {
      if(false == _objectPathManagerIf->getObjectPath4Protocol(service, address, protocol))
      {
         FW_ERRMEM_ASSERT_ALWAYS();
         return;
      }
   }
   else if(BTS_PROTO_SPP == protocol)
   {
      if(false == _objectPathManagerIf->getObjectPath4Protocol(service, address, protocol, uuid))
      {
         FW_ERRMEM_ASSERT_ALWAYS();
         return;
      }
   }
   else
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   Ipc2Bts_DisconnectResult* msg = ptrNew_Ipc2Bts_DisconnectResult();
   if(0 != msg)
   {
      msg->setService(service);
      msg->setIpcCommonErrorCode(errorCode);
   }

   _controlIf->sendInternalIpc2BtsMessage(msg, true);
}

void HfpAvpConnectDisconnectGenivi::sendVirtualConnectedUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const bool connected, IN const BTSIpcCommonErrorCode errorCode)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(sppInstance);
   (void)(masInstance);
   (void)(masName);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   BTSObjectPath service;

   if((BTS_PROTO_HFP == protocol) || (BTS_PROTO_AVP == protocol))
   {
      if(false == _objectPathManagerIf->getObjectPath4Protocol(service, address, protocol))
      {
         FW_ERRMEM_ASSERT_ALWAYS();
         return;
      }
   }
   else if(BTS_PROTO_SPP == protocol)
   {
      if(false == _objectPathManagerIf->getObjectPath4Protocol(service, address, protocol, uuid))
      {
         FW_ERRMEM_ASSERT_ALWAYS();
         return;
      }
   }
   else
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   Ipc2Bts_ServiceStateUpdate* msg1 = ptrNew_Ipc2Bts_ServiceStateUpdate();
   if(0 != msg1)
   {
      msg1->setService(service);
      if(true == connected)
      {
         msg1->setServiceState(BTS_SERVICE_STATE_CONNECTED);
      }
      else
      {
         msg1->setServiceState(BTS_SERVICE_STATE_DISCONNECTED);
      }
      msg1->setIpcCommonErrorCode(errorCode);
   }

   _controlIf->sendInternalIpc2BtsMessage(msg1, true);

   if(true == connected)
   {
      Ipc2Bts_ConnectResult* msg2 = ptrNew_Ipc2Bts_ConnectResult();
      if(0 != msg2)
      {
         msg2->setService(service);
         msg2->setIpcCommonErrorCode(errorCode);
      }

      _controlIf->sendInternalIpc2BtsMessage(msg2, true);
   }
   else
   {
      Ipc2Bts_DisconnectResult* msg2 = ptrNew_Ipc2Bts_DisconnectResult();
      if(0 != msg2)
      {
         msg2->setService(service);
         msg2->setIpcCommonErrorCode(errorCode);
      }

      _controlIf->sendInternalIpc2BtsMessage(msg2, true);
   }
}

IProtocolManagerRequest* HfpAvpConnectDisconnectGenivi::getRequestIf(void)
{
   return this;
}

void HfpAvpConnectDisconnectGenivi::setDeviceManagerIf(IN IDeviceManager* deviceManagerIf)
{
   _deviceManagerIf = deviceManagerIf;

   FW_ERRMEM_ASSERT(0 != _deviceManagerIf);
}

void HfpAvpConnectDisconnectGenivi::setServiceSearchIf(IN IServiceSearch* serviceSearchIf)
{
   _serviceSearchIf = serviceSearchIf;

   FW_ERRMEM_ASSERT(0 != _serviceSearchIf);
}

void HfpAvpConnectDisconnectGenivi::setObjectPathManagerIf(IN IObjectPathManagerGenivi* objectPathManager)
{
   _objectPathManagerIf = objectPathManager;

   FW_ERRMEM_ASSERT(0 != _objectPathManagerIf);
}

void HfpAvpConnectDisconnectGenivi::protocolAdded(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDbusPropertyList& properties)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   ETG_TRACE_USR3((" protocolAdded"));

   // is triggered by a signal
   ::ccdbusif::evolution::EvolutionGeniviDbusParser evoParser;
   BTSServiceState state(BTS_SERVICE_STATE_DISCONNECTED);
   BTSProtocolId protocol(BTS_PROTO_LAST);
   BTSObjectPath device;
   ::std::string convertedUuid;

   // store information
   for(size_t i = 0; i < properties.size(); i++)
   {
      const ::ccdbusif::evolution::ServiceProperty property = (::ccdbusif::evolution::ServiceProperty)properties[i].propEnum;

      switch(property)
      {
         case ::ccdbusif::evolution::SERVICE_REMOTEUUID:
         {
            // protocol = convertUuid2Protocol(evoParser.getUuid2Enum(properties[i].propData.getString()));
            protocol = convertSupportedService2Protocol(convertUuid2SupportedService(convertedUuid, properties[i].propData.getString()));
            break;
         }
         case ::ccdbusif::evolution::SERVICE_STATE:
         {
            state = convertServiceState2InternalValue(evoParser.convertString2ServiceState(properties[i].propData.getString()));
            break;
         }
         case ::ccdbusif::evolution::SERVICE_DEVICE:
         {
            device = properties[i].propData.getString();
            break;
         }
         default:
         {
            break;
         }
      }
   }

   // check for valid protocol
   if((BTS_PROTO_HFP == protocol) || (BTS_PROTO_AVP == protocol) || (BTS_PROTO_SPP == protocol))
   {
      // device mapping shall exist
      BTSBDAddress address;
      if(true == _objectPathManagerIf->getAddress4ObjectPath(address, device))
      {
         // store connection status
         const BTSSppInstanceId sppInstance(0);
         const BTSUuid uuid(convertedUuid);
         const BTSMasInstanceId masInstance(0);
         const BTSMasInstanceName masName;
         setConnectionStatus(*_callback, address, protocol, sppInstance, uuid, masInstance, masName, state);

         // protocol is not connected during this step (maybe state is BTS_SERVICE_STATE_CONNECTING)
         protocolAdded(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, uuid, state);
      }
      else
      {
         // device object path is created before service object path => therefore this error should never happen
         FW_ERRMEM_ASSERT_ALWAYS();
      }
   }
   else
   {
      // unexpected protocol
      FW_ERRMEM_ASSERT_ALWAYS();
   }
}

void HfpAvpConnectDisconnectGenivi::protocolAdded(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& deviceObjectPath, IN const BTSObjectPath& protocolObjectPath, IN const BTSProtocolId protocol, IN const BTSUuid& uuid)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   ETG_TRACE_USR3((" protocolAdded: protocol=%d deviceObjectPath=%40s protocolObjectPath=%s", protocol, deviceObjectPath.c_str(), protocolObjectPath.c_str()));

   BTSBDAddress address;

   // service object path was created
   if((BTS_PROTO_HFP == protocol) || (BTS_PROTO_AVP == protocol) || (BTS_PROTO_SPP == protocol))
   {
      // given: device object path, service object, protocol
      // needed: address
      if(false == _objectPathManagerIf->getAddress4ObjectPath(address, deviceObjectPath))
      {
         // device object path is created before service object path => therefore this error should never happen
         FW_ERRMEM_ASSERT_ALWAYS();
         return;
      }
   }
   else
   {
      // protocol not supported by this SM
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   if((BTS_PROTO_HFP == protocol) || (BTS_PROTO_AVP == protocol))
   {
      // store mapping
      _objectPathManagerIf->setProtocol2ObjectPath(address, protocol, protocolObjectPath);
      // DBUS service information has to be added
      _serviceStatusService.addServiceInfo(address, protocol, protocolObjectPath);
      // set to waiting because proxy was created
      _serviceStatusService.setAvailability(address, protocol, BTS_DBUS_SERVICE_WAITING);
   }
   else
   {
      // store mapping
      _objectPathManagerIf->setProtocol2ObjectPath(address, protocol, uuid, protocolObjectPath);
      // DBUS service information has to be added
      _serviceStatusService.addServiceInfo(address, protocol, uuid, protocolObjectPath);
      // set to waiting because proxy was created
      _serviceStatusService.setAvailability(address, protocol, uuid, BTS_DBUS_SERVICE_WAITING);
      // check for pending serial object path
      checkPendingSerialObjectPath(protocolObjectPath);
      // check for pending serial availability
      checkPendingSerialAvailability(protocolObjectPath);
   }

   // indicate that protocol was added
   protocolAdded(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, uuid, BTS_SERVICE_STATE_DISCONNECTED);
}

void HfpAvpConnectDisconnectGenivi::protocolAdded(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSServiceState state)
{
   (void)(state);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   ETG_TRACE_USR3((" protocolAdded: protocol=%d address=%12s uuid=%s", protocol, address.c_str(), uuid.c_str()));

   // consider that this function can be called more than once
   // doubled calls are handled within protocolAdded()
   const BTSSppInstanceId sppInstance(0);
   const BTSMasInstanceId masInstance(0);
   const BTSMasInstanceName masName;
   _callback->protocolAdded(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName);

   // check if all data is complete
   checkForDataComplete(bts2IpcMsgList, bts2AppMsgList, messageItem, *_callback, address, protocol, sppInstance, uuid, masInstance, masName);
}

void HfpAvpConnectDisconnectGenivi::protocolRemoved(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& protocolObjectPath)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   // service object path was removed
   BTSBDAddress address;
   BTSProtocolId protocol;
   BTSUuid uuid;
   if(false == _objectPathManagerIf->getProtocol4ObjectPath(address, protocol, uuid, protocolObjectPath))
   {
      // ignore
      return;
   }

   if((BTS_PROTO_HFP != protocol) && (BTS_PROTO_AVP != protocol) && (BTS_PROTO_SPP != protocol))
   {
      // ignore
      return;
   }

   ETG_TRACE_USR3((" protocolRemoved: protocolObjectPath=%s", protocolObjectPath.c_str()));

   // DBUS service information has to be removed
   if((BTS_PROTO_HFP == protocol) || (BTS_PROTO_AVP == protocol))
   {
      _serviceStatusService.removeServiceInfo(address, protocol);
   }
   else
   {
      _serviceStatusService.removeServiceInfo(address, protocol, uuid);
   }
   // remove related entry in protocol manager
   const BTSSppInstanceId sppInstance(0);
   const BTSMasInstanceId masInstance(0);
   const BTSMasInstanceName masName;
   _callback->protocolRemoved(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName);

   _objectPathManagerIf->removeProtocol2ObjectPath(protocolObjectPath);
}

void HfpAvpConnectDisconnectGenivi::updateDbusServiceAvailability(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& service, IN const BTSDbusServiceAvailability availability)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   ETG_TRACE_USR3((" updateDbusServiceAvailability: service=%s", service.c_str()));

   BTSBDAddress address;
   BTSProtocolId protocol;
   BTSUuid uuid;
   if(false == _objectPathManagerIf->getProtocol4ObjectPath(address, protocol, uuid, service))
   {
      // service object path is created before proxy is created => therefore this error should never happen
      return;
   }

   if((BTS_PROTO_HFP != protocol) && (BTS_PROTO_AVP != protocol) && (BTS_PROTO_SPP != protocol))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   // set availability
   if((BTS_PROTO_HFP == protocol) || (BTS_PROTO_AVP == protocol))
   {
      _serviceStatusService.setAvailability(address, protocol, availability);
   }
   else
   {
      _serviceStatusService.setAvailability(address, protocol, uuid, availability);
   }
   const bool available(BTS_DBUS_SERVICE_AVAILABLE == availability);

   // TODO: [low]: check for BTS_DBUS_SERVICE_NOT_AVAILABLE needed

   const BTSSppInstanceId sppInstance(0);
   const BTSMasInstanceId masInstance(0);
   const BTSMasInstanceName masName;

   // store availability
   setDbusServiceAvailability(*_callback, address, protocol, sppInstance, uuid, masInstance, masName, available);

   // check if all data is complete
   checkForDataComplete(bts2IpcMsgList, bts2AppMsgList, messageItem, *_callback, address, protocol, sppInstance, uuid, masInstance, masName);

   // request major data in case of service is available
   if(true == available)
   {
      createGetServiceStateMsg(bts2IpcMsgList, address, protocol, uuid, service);
   }
}

bool HfpAvpConnectDisconnectGenivi::handleCancelRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool sent, IN const act_t token)
{
   (void)(bts2AppMsgList);
   (void)(messageItem);

   /*
    * Cancel request will be triggered by Evolution in following scenarios:
    * - timeout AutorizeService: 10s => will be handled by failed connect
    */

   // but ensure that answer is sent
   if(false == sent)
   {
      createCancelRes(bts2IpcMsgList, token);
   }

   return true;
}

bool HfpAvpConnectDisconnectGenivi::handleCancelRequestRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const bool sent, IN const act_t token)
{
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(device);

   /*
    * Cancel request will be triggered by Evolution in following scenarios:
    * - timeout AutorizeService: 10s => will be handled by failed connect
    */

   // but ensure that answer is sent
   if(false == sent)
   {
      createCancelRequestRes(bts2IpcMsgList, token);
   }

   return true;
}

void HfpAvpConnectDisconnectGenivi::updateConnectedService(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSServiceFunction& connectedService)
{
   (void)(messageItem);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   ETG_TRACE_USR3((" updateConnectedService: device=%s", device.c_str()));

   BTSBDAddress address;
   if(false == _objectPathManagerIf->getAddress4ObjectPath(address, device))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   BTSServiceFunction& entry = checkConnectedServicesList(address);

   if(entry.getData() == connectedService.getData())
   {
      // no change
      return;
   }

   // at the moment we only need the information about connected AVRCP
   if((false == entry.getBit(BTS_SRV_FUNC_AVRCP)) && (true == connectedService.getBit(BTS_SRV_FUNC_AVRCP)))
   {
      checkForAvrcpVersionRequest(bts2IpcMsgList, bts2AppMsgList, address, BTS_PROTO_AVP);
   }

   // update entry
   entry.setData(connectedService.getData());
}

void HfpAvpConnectDisconnectGenivi::removeConnectedService(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   ETG_TRACE_USR3((" removeConnectedService: device=%s", device.c_str()));

   BTSBDAddress address;
   if(false == _objectPathManagerIf->getAddress4ObjectPath(address, device))
   {
      return;
   }

   removeConnectedServicesInfo(address);
}

bool HfpAvpConnectDisconnectGenivi::handleRemoteConnect(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const act_t token)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_callback);
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_objectPathManagerIf);

   if((BTS_PROTO_HFP != protocol) && (BTS_PROTO_AVP != protocol) && (BTS_PROTO_SPP != protocol))
   {
      return false;
   }

   ETG_TRACE_USR3((" handleRemoteConnect: device=%s", device.c_str()));

   BTSBDAddress address;
   if(false == _objectPathManagerIf->getAddress4ObjectPath(address, device))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return false;
   }

   const BTSSppInstanceId sppInstance(0);
   const BTSMasInstanceId masInstance(0);
   const BTSMasInstanceName masName;

   // store token
   storeToken(address, protocol, uuid, token);

   // check if protocol information is complete
   if(true == _callback->isProtocolAvailableInternal(address, protocol, uuid, masName, true))
   {
      // forward directly
      _callback->remoteConnectRequest(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName);

      if(BTS_SERVICE_STATE_CONNECTING == getConnectionStatus(*_callback, address, protocol, sppInstance, uuid, masInstance, masName))
      {
         _callback->connecting(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName);
      }
   }
   else
   {
      // add to pending list
      addPending(*_callback, address, protocol, sppInstance, uuid, masInstance, masName);
   }

   return true;
}

void HfpAvpConnectDisconnectGenivi::createAuthorizeServiceResponse(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSObjectPath& device, IN const act_t token, IN const BTSConfirmationMode mode, IN const bool responseFlag) const
{
   (void)(responseFlag);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   BTSBDAddress address;
   if(false == _objectPathManagerIf->getAddress4ObjectPath(address, device))
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   createAuthorizeServiceResponseMsg(bts2IpcMsgList, address, token, mode);
}

void HfpAvpConnectDisconnectGenivi::handleConnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, OUT bool& connectFailed, OUT BTSBDAddress& failedAddress, IN const BTSObjectPath& service, IN const BTSIpcCommonErrorCode errorCode)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   ETG_TRACE_USR3((" handleConnectResult: service=%s", service.c_str()));

   BTSBDAddress address;
   BTSProtocolId protocol;
   BTSUuid uuid;
   if(false == _objectPathManagerIf->getProtocol4ObjectPath(address, protocol, uuid, service))
   {
      // should never happen
      return;
   }

   const BTSSppInstanceId sppInstance(0);
   const BTSMasInstanceId masInstance(0);
   const BTSMasInstanceName masName;
   BTSRequestResult result(BTS_REQ_FAILED);

   // check for connect status
   checkConnectStatus(address, protocol, sppInstance, uuid, masInstance, masName, errorCode);

   // check error code
   if(BTS_IPC_SUCCESS == errorCode)
   {
      result = BTS_REQ_SUCCESS;
   }
   else if(BTS_IPC_BUSY == errorCode)
   {
#ifdef ENABLE_BUSY
      result = BTS_REQ_BUSY;
#endif
   }
   else if(BTS_IPC_RETRY_ABORTED == errorCode)
   {
      result = BTS_REQ_CONNECT_ABORTED;
   }
   else
   {
      // other error codes
   }

   // forward connect result
   if(true == _callback->connectResult(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName, result))
   {
      connectFailed = true;
      failedAddress = address;
   }
}

void HfpAvpConnectDisconnectGenivi::updateConnectStatus(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSStatusCode status)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   if((BTS_PROTO_HFP != protocol) && (BTS_PROTO_AVP != protocol) && (BTS_PROTO_SPP != protocol))
   {
      return;
   }

   ETG_TRACE_USR3((" updateConnectStatus: address=%s", address.c_str()));

   BTSUuid uuid;

   if(BTS_PROTO_SPP == protocol)
   {
      // this function is called because of BtApplDeviceConnectInd signal but this signal does only provide an SPP instance instead of SPP UUID
      // get a match of SPP instance and SPP UUID based on address and protocol (SPP) because we do only 1 SPP connect to the same device at the same time

      BTSUuid tmpUuid;
      if(true == findMatchingSppConnectEntry(tmpUuid, *_callback, address))
      {
         uuid = tmpUuid;
      }
      else
      {
         return;
      }
   }

   // optional information, contains e.g. page timeout information

   // forward only in case of failed connect
   if(BTS_STATUS_CODE_SUCCESS == status)
   {
      return;
   }

   // forward failed connect reason
   const BTSSppInstanceId sppInstance(0);
   const BTSMasInstanceId masInstance(0);
   const BTSMasInstanceName masName;
   BTSDisconnectReason reason(BTS_DISCONNECT_REASON_PROTO_CONNECT_FAILED);

   // check for page timeout, aborted, missing key
   if(BTS_STATUS_CODE_PAGE_TIMEOUT == status)
   {
      reason = BTS_DISCONNECT_REASON_OUT_OF_RANGE; // e.g. device is out of range
   }
   else if(BTS_STATUS_CODE_ABORTED == status)
   {
      reason = BTS_DISCONNECT_REASON_PROTO_CONNECT_ABORTED; // e.g. connect was aborted/cancelled
   }
   else if(BTS_STATUS_CODE_PIN_OR_KEY_MISSING == status)
   {
      reason = BTS_DISCONNECT_REASON_MISSING_LINK_KEY; // e.g. pairing information on BT device was removed
   }
   else if(BTS_STATUS_CODE_SERVICE_NOTSUPPORTED == status)
   {
      reason = BTS_DISCONNECT_REASON_CONFLICT_DIFFERENT_DEVICE; // e.g. local AVP connect during remote AVP connect from different device, AVP resource already in use (max 1 instance)
   }

   _callback->updateFailedConnectReason(address, protocol, sppInstance, uuid, masInstance, masName, reason);
}

void HfpAvpConnectDisconnectGenivi::handleConnectInd(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSStatusCode status)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   if((BTS_PROTO_HFP != protocol) && (BTS_PROTO_AVP != protocol) && (BTS_PROTO_SPP != protocol))
   {
      return;
   }

   ETG_TRACE_USR3((" handleConnectInd: address=%s", address.c_str()));

   BTSUuid uuid;

   if(BTS_PROTO_SPP == protocol)
   {
      // this function is called because of BtApplDeviceConnectInd signal but this signal does only provide an SPP instance instead of SPP UUID
      // get a match of SPP instance and SPP UUID based on address and protocol (SPP) because we do only 1 SPP connect to the same device at the same time

      BTSUuid tmpUuid;
      if(true == findMatchingSppConnectEntry(tmpUuid, *_callback, address))
      {
         uuid = tmpUuid;
      }
      else
      {
         return;
      }
   }

   // optional information

   // forward connect indication, use success or failed as result, reason for failed connect was forwarded via updateFailedConnectReason() before
   const BTSSppInstanceId sppInstance(0);
   const BTSMasInstanceId masInstance(0);
   const BTSMasInstanceName masName;
   const BTSRequestResult result((BTS_STATUS_CODE_SUCCESS == status) ? BTS_REQ_SUCCESS : BTS_REQ_FAILED);

   _callback->connectIndication(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName, result);
}

void HfpAvpConnectDisconnectGenivi::handleServiceState(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, OUT bool& connectFailed, OUT BTSBDAddress& failedAddress, IN const BTSObjectPath& service, IN const BTSServiceState serviceState, IN const BTSIpcCommonErrorCode errorCode)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   ETG_TRACE_USR3((" handleServiceState: service=%s", service.c_str()));

   BTSBDAddress address;
   BTSProtocolId protocol;
   BTSUuid uuid;
   if(false == _objectPathManagerIf->getProtocol4ObjectPath(address, protocol, uuid, service))
   {
      // should never happen
      return;
   }

   const BTSSppInstanceId sppInstance(0);
   const BTSMasInstanceId masInstance(0);
   const BTSMasInstanceName masName;

   // get previous connection status
   const BTSServiceState prevState(getConnectionStatus(*_callback, address, protocol, sppInstance, uuid, masInstance, masName));

   BTSServiceState newState(serviceState);

   // handle only in success case (ignore in case of get request failed (METHOD_RETURN_TIMEOUT); initial state is DISCONNECTED) TODO: [low]: check for other property updates too
   if(BTS_IPC_SUCCESS != errorCode)
   {
      newState = prevState;
   }

   // store connection status
   setConnectionStatus(*_callback, address, protocol, sppInstance, uuid, masInstance, masName, newState);

   // check if all data is complete
   checkForDataComplete(bts2IpcMsgList, bts2AppMsgList, messageItem, *_callback, address, protocol, sppInstance, uuid, masInstance, masName);

   // forward connecting and disconnecting
   forwardIntermediateState(bts2IpcMsgList, bts2AppMsgList, messageItem, *_callback, address, protocol, sppInstance, uuid, masInstance, masName, newState);

   // compare old and new service state
   if(prevState == newState)
   {
      // same state => ignore
      return;
   }

   bool process(false);
   bool connected(false);

   switch(newState)
   {
      case BTS_SERVICE_STATE_DISCONNECTED:
         process = true;
         connected = false;
         break;
      case BTS_SERVICE_STATE_CONNECTING:
         // ignore because intermediate state
         break;
      case BTS_SERVICE_STATE_CONNECTED:
         process = true;
         connected = true;
         break;
      case BTS_SERVICE_STATE_DISCONNECTING:
         // ignore because intermediate state
         break;
      case BTS_SERVICE_STATE_ERROR:
         // currently we are not sure what this state really means

         // observed during testing: link loss situation with Samsung Galaxy S2
         // => check current device connection state from stack
         if(true == isDeviceConnected(address))
         {
            // stack value indicates connected => not sure what to do
            FW_ERRMEM_ASSERT_ALWAYS();
         }
         else
         {
            // stack value indicates disconnected => handle as disconnected
            process = true;
            connected = false;
         }
         break;
      case BTS_SERVICE_STATE_LAST:
      default:
         FW_ERRMEM_ASSERT_ALWAYS();
         break;
   }

   if(false == process)
   {
      // nothing to do
      return;
   }

   // check for SPP connected
   if((true == connected) && (BTS_PROTO_SPP == protocol))
   {
      // check for indicated character device name
      checkForAvailableDeviceName(*_callback, address, protocol, sppInstance, uuid, masInstance, masName);
   }

   // forward new connection status
   const BTSConnectionStatus connStatus((true == connected) ? BTS_CONN_CONNECTED : BTS_CONN_DISCONNECTED);
   const BTSDisconnectReason reason(BTS_DISCONNECT_REASON_LAST);
   if(true == _callback->updateStatus(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName, connStatus, reason))
   {
      connectFailed = true;
      failedAddress = address;
   }
}

void HfpAvpConnectDisconnectGenivi::handleDisconnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& service, IN const BTSIpcCommonErrorCode errorCode)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   ETG_TRACE_USR3((" handleDisconnectResult: service=%s", service.c_str()));

   BTSBDAddress address;
   BTSProtocolId protocol;
   BTSUuid uuid;
   if(false == _objectPathManagerIf->getProtocol4ObjectPath(address, protocol, uuid, service))
   {
      // should never happen
      return;
   }

   const BTSSppInstanceId sppInstance(0);
   const BTSMasInstanceId masInstance(0);
   const BTSMasInstanceName masName;
   BTSRequestResult result(BTS_REQ_FAILED);

   // check error code
   if((BTS_IPC_SUCCESS == errorCode) || (BTS_IPC_NOT_CONNECTED == errorCode))
   {
      result = BTS_REQ_SUCCESS;
   }
   else if(BTS_IPC_BUSY == errorCode)
   {
#ifdef ENABLE_BUSY
      result = BTS_REQ_BUSY;
#endif
   }
   else if(BTS_IPC_RETRY_ABORTED == errorCode)
   {
      result = BTS_REQ_DISCONNECT_ABORTED;
   }
   else
   {
      // other error codes
   }

   // forward disconnect result
   _callback->disconnectResult(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName, result);
}

void HfpAvpConnectDisconnectGenivi::updateDisconnectReason(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSInternalDisconnectReason reason)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   if((BTS_PROTO_HFP != protocol) && (BTS_PROTO_AVP != protocol) && (BTS_PROTO_SPP != protocol))
   {
      return;
   }

   ETG_TRACE_USR3((" updateDisconnectReason: address=%s", address.c_str()));

   BTSUuid uuid;

   if(BTS_PROTO_SPP == protocol)
   {
      // this function is called because of BtApplDeviceDisconnectInd signal but this signal does only provide an SPP instance instead of SPP UUID
      // get a match of SPP instance and SPP UUID based on address and protocol (SPP) because we do only 1 SPP disconnect to the same device at the same time

      BTSUuid tmpUuid;
      if(true == findMatchingSppDisconnectEntry(tmpUuid, *_callback, address))
      {
         uuid = tmpUuid;
      }
      else
      {
         return;
      }
   }

   // optional information, only called in case of successful disconnect

   // check for exception
   if(BTS_INT_DISCONNECT_REASON_EXCEPTION == reason)
   {
      // forward to device manager
      informDeviceManagerAboutException(address);
   }

   // forward disconnect reason
   const BTSSppInstanceId sppInstance(0);
   const BTSMasInstanceId masInstance(0);
   const BTSMasInstanceName masName;

   _callback->updateDisconnectReason(address, protocol, sppInstance, uuid, masInstance, masName, convertInternalDisconnectReason2Public(reason));
}

void HfpAvpConnectDisconnectGenivi::handleDisconnectInd(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSStatusCode status)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   if((BTS_PROTO_HFP != protocol) && (BTS_PROTO_AVP != protocol) && (BTS_PROTO_SPP != protocol))
   {
      return;
   }

   ETG_TRACE_USR3((" handleDisconnectInd: address=%s", address.c_str()));

   BTSUuid uuid;

   if(BTS_PROTO_SPP == protocol)
   {
      // this function is called because of BtApplDeviceDisconnectInd signal but this signal does only provide an SPP instance instead of SPP UUID
      // get a match of SPP instance and SPP UUID based on address and protocol (SPP) because we do only 1 SPP disconnect to the same device at the same time

      BTSUuid tmpUuid;
      if(true == findMatchingSppDisconnectEntry(tmpUuid, *_callback, address))
      {
         uuid = tmpUuid;
      }
      else
      {
         return;
      }
   }

   // optional information

   // forward disconnect indication, use success or failed as result, disconnect reason was forwarded via updateDisconnectReason() before
   const BTSSppInstanceId sppInstance(0);
   const BTSMasInstanceId masInstance(0);
   const BTSMasInstanceName masName;
   const BTSRequestResult result((BTS_STATUS_CODE_SUCCESS == status) ? BTS_REQ_SUCCESS : BTS_REQ_FAILED);

   _callback->disconnectIndication(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName, result);
}

void HfpAvpConnectDisconnectGenivi::serialAdded(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& serialObjectPath)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   ETG_TRACE_USR3((" serialAdded: serialObjectPath=%s", serialObjectPath.c_str()));

   // object path is the same as for service interface
   BTSBDAddress address;
   BTSProtocolId protocol;
   BTSUuid uuid;
   if(false == _objectPathManagerIf->getProtocol4ObjectPath(address, protocol, uuid, serialObjectPath))
   {
      // note: it can happen that serial interface is added before service interface
      storePendingSerialObjectPath(serialObjectPath);
      return;
   }

   HfpAvpConnectDisconnectDataGenivi& entry = checkDataList(address, protocol, uuid);

   entry.serialIfAdded = true;
   entry.serialIfAvailable = false;

   // DBUS service information has to be added
   _serviceStatusSerial.addServiceInfo(address, protocol, uuid, serialObjectPath);
   // set to waiting because proxy was created
   _serviceStatusSerial.setAvailability(address, protocol, uuid, BTS_DBUS_SERVICE_WAITING);
}

void HfpAvpConnectDisconnectGenivi::serialRemoved(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& serialObjectPath)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   ETG_TRACE_USR3((" serialRemoved: serialObjectPath=%s", serialObjectPath.c_str()));

   // object path is the same as for service interface
   BTSBDAddress address;
   BTSProtocolId protocol;
   BTSUuid uuid;
   if(false == _objectPathManagerIf->getProtocol4ObjectPath(address, protocol, uuid, serialObjectPath))
   {
      // should never happen TODO: check during testing
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   HfpAvpConnectDisconnectDataGenivi& entry = checkDataList(address, protocol, uuid);

   entry.serialIfAdded = false;
   entry.serialIfAvailable = false;

   // DBUS service information has to be removed
   _serviceStatusSerial.removeServiceInfo(address, protocol, uuid);
}

void HfpAvpConnectDisconnectGenivi::updateDbusServiceSerialAvailability(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& service, IN const BTSDbusServiceAvailability availability)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);

   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   ETG_TRACE_USR3((" updateDbusServiceSerialAvailability: service=%s", service.c_str()));

   // object path is the same as for service interface
   BTSBDAddress address;
   BTSProtocolId protocol;
   BTSUuid uuid;
   if(false == _objectPathManagerIf->getProtocol4ObjectPath(address, protocol, uuid, service))
   {
      // note: it can happen that serial interface is added before service interface
      storePendingSerialAvailability(service, availability);
      return;
   }

   HfpAvpConnectDisconnectDataGenivi& entry = checkDataList(address, protocol, uuid);

   if(BTS_DBUS_SERVICE_AVAILABLE == availability)
   {
      entry.serialIfAdded = true;
      entry.serialIfAvailable = true;
   }
   else
   {
      // do not modify entry.serialIfAdded
      entry.serialIfAvailable = false;
   }

   // set availability
   _serviceStatusSerial.setAvailability(address, protocol, uuid, availability);
}

void HfpAvpConnectDisconnectGenivi::updateCharacterDeviceName(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& service, IN const BTSDeviceName& deviceName)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   ETG_TRACE_USR3((" updateCharacterDeviceName: deviceName=%20s service=%s", deviceName.c_str(), service.c_str()));

   // object path is the same as for service interface
   BTSBDAddress address;
   BTSProtocolId protocol;
   BTSUuid uuid;
   if(false == _objectPathManagerIf->getProtocol4ObjectPath(address, protocol, uuid, service))
   {
      // should never happen
      FW_ERRMEM_ASSERT_ALWAYS();
      return;
   }

   const BTSSppInstanceId sppInstance(0);
   const BTSMasInstanceId masInstance(0);
   const BTSMasInstanceName masName;
   BTSDeviceName virtualDeviceName("/dev/");
   virtualDeviceName += deviceName;
   _callback->updateVirtualDeviceName(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName, virtualDeviceName);
}

void HfpAvpConnectDisconnectGenivi::characterDeviceAddedRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDeviceName& deviceName, IN const bool added)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   ETG_TRACE_USR3((" characterDeviceAddedRemoved: deviceName=%20s added=%d", deviceName.c_str(), added));

   _callback->characterDeviceAddedRemoved(bts2IpcMsgList, bts2AppMsgList, messageItem, deviceName, added);
}

void HfpAvpConnectDisconnectGenivi::handleExtendedTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSTimerId timerId)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   // check timer id
   for(::std::map< BTSProtocolBaseEntry, HfpAvpConnectDisconnectDataGenivi >::iterator it = _connectDisconnectDataList.begin(); it != _connectDisconnectDataList.end(); ++it)
   {
      HfpAvpConnectDisconnectDataGenivi& entry = it->second;

      // check for active timer
      if(true == entry.serviceSppServiceTimer.compare(timerId))
      {
         // timeout happened for availability of SPP service service
         const BTSSppInstanceId sppInstance(0);
         const BTSMasInstanceId masInstance(0);
         const BTSMasInstanceName masName;
         const BTSProtocolBaseEntry& keyData(it->first);

         _callback->waitingForProtocolResult(bts2IpcMsgList, bts2AppMsgList, messageItem, keyData.deviceAddress, keyData.protocolId, sppInstance, keyData.sppUuid, masInstance, masName, BTS_REQ_FAILED);

         return;
      }
   }
}

void HfpAvpConnectDisconnectGenivi::handleAddedRemovedCharacterDevice(IN const BTSDeviceName& deviceName, IN const bool added)
{
   // NOTE: this function is called from different thread => create only an internal event for further processing

   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   Ipc2Bts_CharacterDeviceAddedRemoved* msg = ptrNew_Ipc2Bts_CharacterDeviceAddedRemoved();
   if(0 != msg)
   {
      msg->setDeviceName(deviceName);
      msg->setAdded(added);
      msg->setIpcCommonErrorCode(BTS_IPC_SUCCESS);
   }

   _controlIf->sendInternalIpc2BtsMessage(msg, true);
}

void HfpAvpConnectDisconnectGenivi::createCancelRes(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const act_t token) const
{
   Bts2Ipc_CancelRes* msg = ptrNew_Bts2Ipc_CancelRes();
   if(0 != msg)
   {
      msg->setDbusToken(token);
      msg->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg);
   }
}

void HfpAvpConnectDisconnectGenivi::createCancelRequestRes(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const act_t token) const
{
   Bts2Ipc_CancelRequestRes* msg = ptrNew_Bts2Ipc_CancelRequestRes();
   if(0 != msg)
   {
      msg->setDbusToken(token);
      msg->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg);
   }
}

void HfpAvpConnectDisconnectGenivi::createGetServiceStateMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSObjectPath& service) const
{
   Bts2Ipc_GetServiceState* msg = ptrNew_Bts2Ipc_GetServiceState();
   if(0 != msg)
   {
      msg->setBDAddress(address);
      msg->setProtocol(protocol);
      msg->setUuid(uuid);
      msg->setService(service);
      msg->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg);
   }
}

void HfpAvpConnectDisconnectGenivi::createAuthorizeServiceResponseMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& address, IN const act_t token, IN const BTSConfirmationMode mode) const
{
   Bts2Ipc_AuthorizeServiceRes* msg = ptrNew_Bts2Ipc_AuthorizeServiceRes();
   if(0 != msg)
   {
      msg->setBDAddress(address);
      msg->setAccept(mode);
      msg->setDbusToken(token);
      msg->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg);
   }
}

void HfpAvpConnectDisconnectGenivi::createConnectMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& deviceAddress, IN const BTSProtocolId protocolId, IN const BTSUuid& uuid, IN const BTSObjectPath& service) const
{
   Bts2Ipc_Connect* msg = ptrNew_Bts2Ipc_Connect();
   if(0 != msg)
   {
      msg->setBDAddress(deviceAddress);
      msg->setProtocol(protocolId);
      msg->setUuid(uuid);
      msg->setService(service);
      msg->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg);
   }
}

void HfpAvpConnectDisconnectGenivi::createDisconnectMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& deviceAddress, IN const BTSProtocolId protocolId, IN const BTSUuid& uuid, IN const BTSObjectPath& service) const
{
   Bts2Ipc_Disconnect* msg = ptrNew_Bts2Ipc_Disconnect();
   if(0 != msg)
   {
      msg->setBDAddress(deviceAddress);
      msg->setProtocol(protocolId);
      msg->setUuid(uuid);
      msg->setService(service);
      msg->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg);
   }
}

void HfpAvpConnectDisconnectGenivi::createSetEnableAvpPauseMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSObjectPath& device, IN const bool enableAvpPause) const
{
   Bts2Ipc_SetDeviceEnableAvpPause* msg = ptrNew_Bts2Ipc_SetDeviceEnableAvpPause();
   if(0 != msg)
   {
      msg->setBDAddress(address);
      msg->setDevice(device);
      msg->setEnableAvpPause(enableAvpPause);
      msg->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg);
   }
}

void HfpAvpConnectDisconnectGenivi::forwardIntermediateState(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSServiceState serviceState) const
{
   if(BTS_SERVICE_STATE_CONNECTING == serviceState)
   {
      callback.connecting(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName);
   }
   else if(BTS_SERVICE_STATE_DISCONNECTING == serviceState)
   {
      callback.disconnecting(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName);
   }
}

void HfpAvpConnectDisconnectGenivi::checkForDataComplete(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName)
{
   if((BTS_PROTO_HFP == protocol) || (BTS_PROTO_AVP == protocol))
   {
      // keep existing handling for HFP/AVP
      checkForDataCompleteHfpAvp(bts2IpcMsgList, bts2AppMsgList, messageItem, callback, address, protocol, sppInstance, uuid, masInstance, masName);
   }
   else if(BTS_PROTO_SPP == protocol)
   {
      // modified handling for SPP
      checkForDataCompleteSpp(bts2IpcMsgList, bts2AppMsgList, messageItem, callback, address, protocol, sppInstance, uuid, masInstance, masName);
   }
   else
   {
      FW_ERRMEM_ASSERT_ALWAYS();
   }
}

void HfpAvpConnectDisconnectGenivi::checkForDataCompleteHfpAvp(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName)
{
   // get data entry
   ProtocolManagerData& entry = callback.getEntry(address, protocol, sppInstance, uuid, masInstance, masName);

   // get stored availability information
   const bool oldAvailable(entry.info.getBit(ProtocolManagerData::PROTOCOL_AVAILABLE));

   // get new availability information
   bool newAvailable(true);

   if(false == isDeviceAvailable(address))
   {
      newAvailable = false;
   }
   else if(false == entry.info.getBit(ProtocolManagerData::PROTOCOL_ADDED))
   {
      newAvailable = false;
   }
   else if(false == getDbusServiceAvailability(entry.secondaryInfo))
   {
      newAvailable = false;
   }
   else if(false == entry.info.getBit(ProtocolManagerData::STATUS_UPDATED))
   {
      newAvailable = false;
   }

   // indicate only changed states
   if(oldAvailable != newAvailable)
   {
      ProtocolManagerData::ProtocolAvailableInfo protocolAvailableInfo;
      protocolAvailableInfo.setBit(ProtocolManagerData::PROTOCOL_AVAILABLE_LEVEL_1);
      protocolAvailableInfo.setBit(ProtocolManagerData::PROTOCOL_AVAILABLE_LEVEL_2);

      if(true == newAvailable)
      {
         callback.protocolAvailable(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName, protocolAvailableInfo);

         checkPendingConnectionRequest(bts2IpcMsgList, bts2AppMsgList, messageItem, callback, address, protocol, sppInstance, uuid, masInstance, masName);
      }
      else
      {
         callback.protocolUnavailable(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName, protocolAvailableInfo);
      }
   }
}

void HfpAvpConnectDisconnectGenivi::checkForDataCompleteSpp(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName)
{
   // get data entry
   ProtocolManagerData& entry = callback.getEntry(address, protocol, sppInstance, uuid, masInstance, masName);

   // get stored availability information
   const bool oldAvailableL1(entry.info.getBit(ProtocolManagerData::PROTOCOL_AVAILABLE));
   const bool oldAvailableL2(entry.info.getBit(ProtocolManagerData::PROTOCOL_AVAILABLE_L2));

   // get new availability information
   bool newAvailableL1(true);
   bool newAvailableL2(true);

   if(false == isDeviceAvailable(address))
   {
      newAvailableL1 = false;
      newAvailableL2 = false;
   }
   else if(false == entry.info.getBit(ProtocolManagerData::PROTOCOL_ADDED))
   {
      newAvailableL1 = false;
      newAvailableL2 = false;
   }
   else if(false == getDbusServiceAvailability(entry.secondaryInfo))
   {
      newAvailableL2 = false;
   }
   else if(false == entry.info.getBit(ProtocolManagerData::STATUS_UPDATED))
   {
      newAvailableL2 = false;
   }

   // indicate only changed states
   if(oldAvailableL1 != newAvailableL1)
   {
      ProtocolManagerData::ProtocolAvailableInfo protocolAvailableInfo;
      protocolAvailableInfo.setBit(ProtocolManagerData::PROTOCOL_AVAILABLE_LEVEL_1);

      if(true == newAvailableL1)
      {
         callback.protocolAvailable(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName, protocolAvailableInfo);
      }
      else
      {
         callback.protocolUnavailable(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName, protocolAvailableInfo);
      }
   }

   // indicate only changed states
   if(oldAvailableL2 != newAvailableL2)
   {
      ProtocolManagerData::ProtocolAvailableInfo protocolAvailableInfo;
      protocolAvailableInfo.setBit(ProtocolManagerData::PROTOCOL_AVAILABLE_LEVEL_2);

      if(true == newAvailableL2)
      {
         callback.protocolAvailable(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName, protocolAvailableInfo);

         HfpAvpConnectDisconnectDataGenivi& entry = checkDataList(address, protocol, uuid);

         // check for active timer
         if(true == isTimerActive(entry.serviceSppServiceTimer))
         {
            stopTimer(entry.serviceSppServiceTimer);

            // SPP service service is available now
            const BTSSppInstanceId sppInstance(0);
            const BTSMasInstanceId masInstance(0);
            const BTSMasInstanceName masName;

            callback.waitingForProtocolResult(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName, BTS_REQ_SUCCESS);
         }

         checkPendingConnectionRequest(bts2IpcMsgList, bts2AppMsgList, messageItem, callback, address, protocol, sppInstance, uuid, masInstance, masName);
      }
      else
      {
         callback.protocolUnavailable(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName, protocolAvailableInfo);
      }
   }
}

void HfpAvpConnectDisconnectGenivi::setConnectionStatus(IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSServiceState state)
{
   ProtocolManagerData& entry = callback.getEntry(address, protocol, sppInstance, uuid, masInstance, masName);

   setConnectionStatus(entry.secondaryInfo, state);

   // mark connection status as available
   entry.info.setBit(ProtocolManagerData::STATUS_UPDATED);
}

void HfpAvpConnectDisconnectGenivi::setConnectionStatus(OUT unsigned int& secondaryData, IN const BTSServiceState state) const
{
   BTSServiceState tmpState(state);

   if(BTS_SERVICE_STATE_LAST <= tmpState)
   {
      FW_ERRMEM_ASSERT_ALWAYS();
      tmpState = BTS_SERVICE_STATE_DISCONNECTED;
   }

   // use bits 0 .. 3
   const unsigned int stateMask(0x0F);
   const unsigned int stateValue((const unsigned int)tmpState);

   secondaryData &= ~stateMask;
   secondaryData |= (stateValue & stateMask);
}

BTSServiceState HfpAvpConnectDisconnectGenivi::getConnectionStatus(IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const
{
   ProtocolManagerData& entry = callback.getEntry(address, protocol, sppInstance, uuid, masInstance, masName);

   return getConnectionStatus(entry.secondaryInfo);
}

BTSServiceState HfpAvpConnectDisconnectGenivi::getConnectionStatus(IN const unsigned int secondaryData) const
{
   // use bits 0 .. 3
   const unsigned int stateMask(0x0F);

   return (BTSServiceState)(secondaryData & stateMask);
}

void HfpAvpConnectDisconnectGenivi::setDbusServiceAvailability(IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const bool available)
{
   ProtocolManagerData& entry = callback.getEntry(address, protocol, sppInstance, uuid, masInstance, masName);

   setDbusServiceAvailability(entry.secondaryInfo, available);
}

void HfpAvpConnectDisconnectGenivi::setDbusServiceAvailability(OUT unsigned int& secondaryData, IN const bool available) const
{
   // use bit 4
   const unsigned int stateMask(0x010);
   const unsigned int stateValue((const unsigned int)available);

   secondaryData &= ~stateMask;
   secondaryData |= ((stateValue << 4) & stateMask);
}

bool HfpAvpConnectDisconnectGenivi::getDbusServiceAvailability(IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const
{
   ProtocolManagerData& entry = callback.getEntry(address, protocol, sppInstance, uuid, masInstance, masName);

   return getDbusServiceAvailability(entry.secondaryInfo);
}

bool HfpAvpConnectDisconnectGenivi::getDbusServiceAvailability(IN const unsigned int secondaryData) const
{
   // use bit 4
   const unsigned int stateMask(0x010);

   return (stateMask == (secondaryData & stateMask));
}

bool HfpAvpConnectDisconnectGenivi::isDeviceAvailable(IN const BTSBDAddress& address) const
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_deviceManagerIf);

   return _deviceManagerIf->isDeviceAvailable(address);
}

bool HfpAvpConnectDisconnectGenivi::isDeviceConnected(IN const BTSBDAddress& address) const
{
   FW_ERRMEM_IF_NULL_PTR_RETURN_FALSE(_deviceManagerIf);

   return _deviceManagerIf->isDeviceConnected(address, true);
}

void HfpAvpConnectDisconnectGenivi::informDeviceManagerAboutException(IN const BTSBDAddress& address) const
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_deviceManagerIf);

   _deviceManagerIf->indicateDisconnectByException(address);
}

BTSServiceFunction& HfpAvpConnectDisconnectGenivi::checkConnectedServicesList(IN const BTSBDAddress& address)
{
   ::std::map< BTSBDAddress, BTSServiceFunction >::iterator it = _connectedServicesList.find(address);
   if(_connectedServicesList.end() == it)
   {
      // add new entry
      ETG_TRACE_USR3((" checkConnectedServicesList: _connectedServicesList.size()=%u", (1 + _connectedServicesList.size())));
      return _connectedServicesList[address];
   }
   else
   {
      // continue with existing entry
      return it->second;
   }
}

void HfpAvpConnectDisconnectGenivi::removeConnectedServicesInfo(IN const BTSBDAddress& address)
{
   ::std::map< BTSBDAddress, BTSServiceFunction >::iterator it = _connectedServicesList.find(address);
   if(_connectedServicesList.end() != it)
   {
      _connectedServicesList.erase(it);
   }

   ETG_TRACE_USR3((" removeConnectedServicesInfo: _connectedServicesList.size()=%u", _connectedServicesList.size()));
}

void HfpAvpConnectDisconnectGenivi::checkForAvrcpVersionRequest(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocolId)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_serviceSearchIf);

   if(BTS_PROTO_AVP != protocolId)
   {
      return;
   }

   _serviceSearchIf->requestServiceVersion(bts2IpcMsgList, bts2AppMsgList, address, protocolId);
}


void HfpAvpConnectDisconnectGenivi::storeToken(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const act_t token)
{
   BTSProtocolBaseEntry entry;
   entry.deviceAddress = address;
   entry.protocolId = protocol;
   entry.sppUuid = uuid;

   ::std::map< BTSProtocolBaseEntry, act_t >::iterator it = _tokenList.find(entry);
   if(_tokenList.end() != it)
   {
      // entry is already stored => continue with latest request
      _tokenList[entry] = token;
   }
   else
   {
      // add entry
      _tokenList[entry] = token;
   }
}

bool HfpAvpConnectDisconnectGenivi::getAndRemoveToken(OUT act_t& token, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid)
{
   BTSProtocolBaseEntry entry;
   entry.deviceAddress = address;
   entry.protocolId = protocol;
   entry.sppUuid = uuid;

   ::std::map< BTSProtocolBaseEntry, act_t >::iterator it = _tokenList.find(entry);
   if(_tokenList.end() != it)
   {
      // entry is available
      token = it->second;

      _tokenList.erase(it);

      return true;
   }
   else
   {
      // entry is not available
   }

   return false;
}

void HfpAvpConnectDisconnectGenivi::addPending(IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName)
{
   ProtocolManagerData& entry = callback.getEntry(address, protocol, sppInstance, uuid, masInstance, masName);

   entry.remotePending = true;
}

void HfpAvpConnectDisconnectGenivi::checkPendingConnectionRequest(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const
{
   // data for device + protocol + UUID entry is now complete (available)
   // check for pending entry
   ProtocolManagerData& entry = callback.getEntry(address, protocol, sppInstance, uuid, masInstance, masName);

   if(true == entry.remotePending)
   {
      // forward
      callback.remoteConnectRequest(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName);

      if(BTS_SERVICE_STATE_CONNECTING == getConnectionStatus(entry.secondaryInfo))
      {
         callback.connecting(bts2IpcMsgList, bts2AppMsgList, messageItem, address, protocol, sppInstance, uuid, masInstance, masName);
      }
   }
}

void HfpAvpConnectDisconnectGenivi::startTimer(IN ExtendedTimerEntry& timer, IN const BTSTimeValue timeout)
{
   ETG_TRACE_USR3((" startTimer: timeout=%u", timeout));

   FW_ERRMEM_IF_NULL_PTR_RETURN(_timerPoolIf);
   FW_ERRMEM_IF_NULL_PTR_RETURN(_controlIf);

   timer.setTimerPool(_timerPoolIf);
   timer.start(_controlIf, this, timeout);
}

void HfpAvpConnectDisconnectGenivi::stopTimer(IN ExtendedTimerEntry& timer) const
{
   ETG_TRACE_USR3((" stopTimer"));

   timer.stop();

   // do not release timer
}

void HfpAvpConnectDisconnectGenivi::releaseTimer(IN ExtendedTimerEntry& timer) const
{
   ETG_TRACE_USR3((" releaseTimer"));

   timer.release();
}

bool HfpAvpConnectDisconnectGenivi::isTimerActive(IN const ExtendedTimerEntry& timer) const
{
   return timer.isActive();
}

HfpAvpConnectDisconnectDataGenivi& HfpAvpConnectDisconnectGenivi::checkDataList(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid)
{
   BTSProtocolBaseEntry entry;
   entry.deviceAddress = address;
   entry.protocolId = protocol;
   entry.sppUuid = uuid;

   return _connectDisconnectDataList[entry];
}

bool HfpAvpConnectDisconnectGenivi::findMatchingSppConnectEntry(OUT BTSUuid& uuid, IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address) const
{
   const ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >& protocolList = callback.getProtocolList();

   // check list
   for(::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = protocolList.begin(); it != protocolList.end(); ++it)
   {
      const BTSProtocolBaseEntry& keyData = it->first;

      if((address == keyData.deviceAddress) && (BTS_PROTO_SPP == keyData.protocolId))
      {
         const ProtocolManagerData& entry = it->second;

         if(0 != entry.sm.getConnectSm())
         {
            // we do only 1 SPP connect to the same device at the same time => entry found
            uuid = keyData.sppUuid;

            return true;
         }
      }
   }

   return false;
}

bool HfpAvpConnectDisconnectGenivi::findMatchingSppDisconnectEntry(OUT BTSUuid& uuid, IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address) const
{
   const ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >& protocolList = callback.getProtocolList();

   // disconnect indication happens during local SPP disconnect, remote SPP disconnect, local device disconnect, connection loss

   unsigned int count(0);
   for(::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = protocolList.begin(); it != protocolList.end(); ++it)
   {
      const BTSProtocolBaseEntry& keyData = it->first;

      if((address == keyData.deviceAddress) && (BTS_PROTO_SPP == keyData.protocolId))
      {
         count++;
      }
   }

   if(1 == count)
   {
      for(::std::map< BTSProtocolBaseEntry, ProtocolManagerData >::const_iterator it = protocolList.begin(); it != protocolList.end(); ++it)
      {
         const BTSProtocolBaseEntry& keyData = it->first;

         if((address == keyData.deviceAddress) && (BTS_PROTO_SPP == keyData.protocolId))
         {
            uuid = keyData.sppUuid;

            return true;
         }
      }
   }

   return false;
}

void HfpAvpConnectDisconnectGenivi::checkForAvailableDeviceName(IN IProtocolManagerCallback& callback, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) const
{
   ProtocolManagerData& entry = callback.getEntry(address, protocol, sppInstance, uuid, masInstance, masName);

   if(true == entry.rfCommDevice.empty())
   {
      // character device name was not indicated until now => bug in BT stack
      FW_ERRMEM_ASSERT_ALWAYS();
   }
}

void HfpAvpConnectDisconnectGenivi::storePendingSerialObjectPath(IN const BTSObjectPath& serialObjectPath)
{
   _pendingSerialObjectPathEntries.push_back(serialObjectPath);
}

void HfpAvpConnectDisconnectGenivi::checkPendingSerialObjectPath(IN const BTSObjectPath& serialObjectPath)
{
   for(::std::vector< BTSObjectPath >::iterator it = _pendingSerialObjectPathEntries.begin(); it != _pendingSerialObjectPathEntries.end(); ++it)
   {
      if(serialObjectPath == *it)
      {
         ETG_TRACE_USR3((" checkPendingSerialObjectPath: serialObjectPath=%s", serialObjectPath.c_str()));

         // object path is the same as for service interface
         BTSBDAddress address;
         BTSProtocolId protocol;
         BTSUuid uuid;
         if(false == _objectPathManagerIf->getProtocol4ObjectPath(address, protocol, uuid, serialObjectPath))
         {
            // should never happen
            FW_ERRMEM_ASSERT_ALWAYS();
            return;
         }

         HfpAvpConnectDisconnectDataGenivi& entry = checkDataList(address, protocol, uuid);

         entry.serialIfAdded = true;
         entry.serialIfAvailable = false;

         // DBUS service information has to be added
         _serviceStatusSerial.addServiceInfo(address, protocol, uuid, serialObjectPath);
         // set to waiting because proxy was created
         _serviceStatusSerial.setAvailability(address, protocol, uuid, BTS_DBUS_SERVICE_WAITING);

         _pendingSerialObjectPathEntries.erase(it);

         return;
      }
   }
}

void HfpAvpConnectDisconnectGenivi::storePendingSerialAvailability(IN const BTSObjectPath& serialObjectPath, IN const BTSDbusServiceAvailability availability)
{
   _pendingSerialAvailabilityEntries[serialObjectPath] = availability;
}

void HfpAvpConnectDisconnectGenivi::checkPendingSerialAvailability(IN const BTSObjectPath& serialObjectPath)
{
   ::std::map< BTSObjectPath, BTSDbusServiceAvailability >::iterator it = _pendingSerialAvailabilityEntries.find(serialObjectPath);
   if(_pendingSerialAvailabilityEntries.end() != it)
   {
      ETG_TRACE_USR3((" checkPendingSerialAvailability: serialObjectPath=%s", serialObjectPath.c_str()));

      // object path is the same as for service interface
      BTSBDAddress address;
      BTSProtocolId protocol;
      BTSUuid uuid;
      if(false == _objectPathManagerIf->getProtocol4ObjectPath(address, protocol, uuid, serialObjectPath))
      {
         // should never happen
         FW_ERRMEM_ASSERT_ALWAYS();
         return;
      }

      HfpAvpConnectDisconnectDataGenivi& entry = checkDataList(address, protocol, uuid);

      const BTSDbusServiceAvailability availability(it->second);

      if(BTS_DBUS_SERVICE_AVAILABLE == availability)
      {
         entry.serialIfAdded = true;
         entry.serialIfAvailable = true;
      }
      else
      {
         // do not modify entry.serialIfAdded
         entry.serialIfAvailable = false;
      }

      // set availability
      _serviceStatusSerial.setAvailability(address, protocol, uuid, availability);

      _pendingSerialAvailabilityEntries.erase(it);
   }
}

void HfpAvpConnectDisconnectGenivi::checkConnectStatus(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSIpcCommonErrorCode errorCode)
{
   FW_ERRMEM_IF_NULL_PTR_RETURN(_callback);

   // check error code
   if(BTS_IPC_PAIRING_MISSING_PIN == errorCode)
   {
      _callback->updateFailedConnectReason(address, protocol, sppInstance, uuid, masInstance, masName, BTS_DISCONNECT_REASON_MISSING_LINK_KEY);
   }
   else if(BTS_IPC_CONNECTION_CONFLICT_SAME_DEVICE == errorCode)
   {
      _callback->updateFailedConnectReason(address, protocol, sppInstance, uuid, masInstance, masName, BTS_DISCONNECT_REASON_CONFLICT_SAME_DEVICE);
   }
   else
   {
      // ignore all other cases
   }
}

} //genivi
} //btstackif
