/**
 * @file DeviceManagerGenivi.cpp
 *
 * @par SW-Component
 * State machine for device manager
 *
 * @brief Implementation of Genivi device manager state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of Genivi device manager state machine.
 */

#include "DeviceManagerGenivi.h"
#include "IDeviceManagerCallback.h"
#include "IObjectPathManagerGenivi.h"
#include "IBasicControl.h"
#include "DeviceManagerData.h"
#include "FwAssert.h"
#include "FwBluetoothStringUtils.h"
#include "EvolutionGeniviUtils.h"
#include "Bts2Ipc_MessageWrapper_GEN.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"
#include "TraceClasses.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/DeviceManagerGenivi.cpp.trc.h"
#endif
#endif

namespace btstackif {
namespace genivi {

DeviceManagerGenivi::DeviceManagerGenivi() :
_callback(0),
_controlIf(0),
_objectPathManagerIf(0),
_serviceStatusDevice()
{
   _serviceStatusDevice.setEnabled(true); // only used if any device connection is created
   _serviceStatusDevice.setName("Device");
   _serviceStatusDevice.setInterface((BTSCommonEnumClass)BTS_GEN_DBUS_SERVICE_DEVICE);
}

DeviceManagerGenivi::~DeviceManagerGenivi()
{
   _callback = 0;
   _controlIf = 0;
   _objectPathManagerIf = 0;
}

void DeviceManagerGenivi::reset(void)
{
   _serviceStatusDevice.removeAllServiceInfos();

   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);
   _objectPathManagerIf->removeAllAddress2ObjectPathMappings();
}

void DeviceManagerGenivi::setCallback(IN IDeviceManagerCallback* callback)
{
   _callback = callback;

   FW_NORMAL_ASSERT(0 != _callback);
}

void DeviceManagerGenivi::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_IF_NULL_PTR_RETURN(_controlIf);

   _serviceStatusDevice.setIpc2BtsSendIf(_controlIf);
}

void DeviceManagerGenivi::disconnect(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const bool pauseBtStreaming)
{
   (void)(bts2AppMsgList);

   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);
   FW_IF_NULL_PTR_RETURN(_callback);

   // all necessary checks are done before

   BTSObjectPath objPath;
   if(false == _objectPathManagerIf->getObjectPath4Address(objPath, address))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // get data entry
   DeviceManagerData& data = _callback->getEntry(address);

   // if requested value already set, then ignore
   if(pauseBtStreaming != data.enableAvpPause)
   {
      createSetEnableAvpPauseMsg(bts2IpcMsgList, address, objPath, pauseBtStreaming);
   }

   createDeviceDisconnectMsg(bts2IpcMsgList, address, objPath);
}

void DeviceManagerGenivi::create(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address)
{
   (void)(bts2AppMsgList);

   // all necessary checks are done before

   createDeviceCreateMsg(bts2IpcMsgList, address);
}

void DeviceManagerGenivi::remove(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address)
{
   (void)(bts2AppMsgList);

   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   // all necessary checks are done before

   BTSObjectPath objPath;
   if(false == _objectPathManagerIf->getObjectPath4Address(objPath, address))
   {
      // should never happen
      // FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   createDeviceRemoveMsg(bts2IpcMsgList, address, objPath);
}

void DeviceManagerGenivi::sendVirtualDeviceConnectedUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const bool connected, IN const BTSIpcCommonErrorCode errorCode)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);

   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);
   FW_IF_NULL_PTR_RETURN(_controlIf);

   // all necessary checks are done before

   BTSObjectPath objPath;
   if(false == _objectPathManagerIf->getObjectPath4Address(objPath, address))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   Ipc2Bts_DeviceConnectedUpdate* msg = ptrNew_Ipc2Bts_DeviceConnectedUpdate();
   if(0 != msg)
   {
      msg->setConnected(connected);
      msg->setDevice(objPath);
      msg->setIpcCommonErrorCode(errorCode);
   }
   _controlIf->sendInternalIpc2BtsMessage(msg);
}

void DeviceManagerGenivi::setObjectPathManagerIf(IN IObjectPathManagerGenivi* objectPathManager)
{
   _objectPathManagerIf = objectPathManager;

   FW_NORMAL_ASSERT(0 != _objectPathManagerIf);
}

void DeviceManagerGenivi::deviceAdded(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSObjectPath& device)
{
   ETG_TRACE_USR3((" deviceAdded: address=%s", address.c_str()));

   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   // device object path was created
   // given: address, object path
   // needed: address, object path
   _objectPathManagerIf->setAddress2ObjectPath(address, device);

   // indicate that device was added
   const BTSDeviceName name;
   const BTSCod cod(0);
   deviceAdded(bts2IpcMsgList, bts2AppMsgList, messageItem, address, name, cod, false);

   // DBUS service information has to be added
   _serviceStatusDevice.addServiceInfo(address, device);
   // set to waiting because proxy was created
   _serviceStatusDevice.setAvailability(address, BTS_DBUS_SERVICE_WAITING);
}

void DeviceManagerGenivi::deviceAdded(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDbusPropertyList& properties)
{
   ETG_TRACE_USR3((" deviceAdded"));

   // is triggered by a signal
   BTSBDAddress address;
   BTSDeviceName name;
   bool connected(false);
   BTSCod cod(0);

   // store information
   for(size_t i = 0; i < properties.size(); i++)
   {
      const ::ccdbusif::evolution::DeviceProperty property = (::ccdbusif::evolution::DeviceProperty)properties[i].propEnum;

      switch(property)
      {
         case ::ccdbusif::evolution::DEVICE_ADDRESS:
         {
            const BTSBDAddress& receivedAddress = properties[i].propData.getString();
            if(false == receivedAddress.empty())
            {
               convertBdAddress2InternalValue(address, receivedAddress);
            }
            break;
         }
         case ::ccdbusif::evolution::DEVICE_CLASS:
         {
            cod = properties[i].propData.getUInt32();
            break;
         }
         case ::ccdbusif::evolution::DEVICE_CONNECTED:
         {
            connected = properties[i].propData.getBool();
            break;
         }
         case ::ccdbusif::evolution::DEVICE_ALIAS:
         {
            name = properties[i].propData.getString();
            break;
         }
         default:
         {
            break;
         }
      }
   }

   if(true == ::fw::isValidBdAddress(address))
   {
      deviceAdded(bts2IpcMsgList, bts2AppMsgList, messageItem, address, name, cod, connected);
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }
}

void DeviceManagerGenivi::deviceAdded(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSDeviceName& name, IN const BTSCod cod, IN const bool connected)
{
   ETG_TRACE_USR3((" deviceAdded: address=%s", address.c_str()));

   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   // consider that this function can be called more than once
   // doubled calls are handled within deviceAdded()
   _callback->deviceAdded(bts2IpcMsgList, bts2AppMsgList, messageItem, address);

   // check name
   if(false == name.empty())
   {
      _callback->updateName(bts2IpcMsgList, bts2AppMsgList, messageItem, address, name);
   }

   // check cod
   if(0 != cod)
   {
      _callback->updateCod(address, cod);
   }


   // check connected
   if(true == connected)
   {
      _callback->updateAclStatus(bts2IpcMsgList, bts2AppMsgList, messageItem, address, BTS_CONN_CONNECTED, BTS_DISCONNECT_REASON_LAST);
   }

   // check if all data is complete
   checkForDataComplete(bts2IpcMsgList, bts2AppMsgList, messageItem, *_callback, address);
}

void DeviceManagerGenivi::deviceRemoved(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device)
{
   ETG_TRACE_USR3((" deviceRemoved: device=%s", device.c_str()));

   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   // device object path was removed
   BTSBDAddress address;
   if(true == _objectPathManagerIf->getAddress4ObjectPath(address, device))
   {
      // DBUS service information has to be removed
      _serviceStatusDevice.removeServiceInfo(address);
      // remove related entry in device manager
      _callback->deviceRemoved(bts2IpcMsgList, bts2AppMsgList, messageItem, address);
   }

   _objectPathManagerIf->removeAddress2ObjectPath(device);
}

void DeviceManagerGenivi::updateAclStatus(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const bool connected, IN const BTSIpcCommonErrorCode errorCode, IN const bool response)
{
   (void)(response);

   ETG_TRACE_USR3((" updateAclStatus: device=%s", device.c_str()));

   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   if(BTS_IPC_SUCCESS != errorCode)
   {
      // we have to ignore
      return;
   }

   // TODO: double check device object path removal
   BTSBDAddress address;
   if(false == _objectPathManagerIf->getAddress4ObjectPath(address, device))
   {
      // should never happen
      return;
   }

   BTSConnectionStatus status(BTS_CONN_DISCONNECTED);
   if(true == connected)
   {
      status = BTS_CONN_CONNECTED;
   }

   _callback->updateAclStatus(bts2IpcMsgList, bts2AppMsgList, messageItem, address, status, BTS_DISCONNECT_REASON_LAST);

   // check if all data is complete
   checkForDataComplete(bts2IpcMsgList, bts2AppMsgList, messageItem, *_callback, address);
}

void DeviceManagerGenivi::updateName(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSDeviceName& name, IN const BTSIpcCommonErrorCode errorCode, IN const bool response)
{
   (void)(response);

   ETG_TRACE_USR3((" updateName: device=%s", device.c_str()));

   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   if(BTS_IPC_SUCCESS != errorCode)
   {
      // we have to ignore
      return;
   }

   BTSBDAddress address;
   if(false == _objectPathManagerIf->getAddress4ObjectPath(address, device))
   {
      // should never happen
      return;
   }

   _callback->updateName(bts2IpcMsgList, bts2AppMsgList, messageItem, address, name);

   // check if all data is complete
   checkForDataComplete(bts2IpcMsgList, bts2AppMsgList, messageItem, *_callback, address);
}

void DeviceManagerGenivi::updateRole(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const bool master, IN const BTSIpcCommonErrorCode errorCode, IN const bool response)
{
   (void)(response);

   ETG_TRACE_USR3((" updateRole: device=%s", device.c_str()));

   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   if(BTS_IPC_SUCCESS != errorCode)
   {
      // we have to ignore
      return;
   }

   BTSBDAddress address;
   if(false == _objectPathManagerIf->getAddress4ObjectPath(address, device))
   {
      // should never happen
      return;
   }

   _callback->updateRole(address, master);

   // check if all data is complete
   checkForDataComplete(bts2IpcMsgList, bts2AppMsgList, messageItem, *_callback, address);
}

void DeviceManagerGenivi::updateEnableAvpPause(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const bool enableAvpPause, IN const BTSIpcCommonErrorCode errorCode, IN const bool response)
{
   (void)(response);

   ETG_TRACE_USR3((" updateEnableAvpPause: device=%s", device.c_str()));

   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   if(BTS_IPC_SUCCESS != errorCode)
   {
      // we have to ignore
      return;
   }

   BTSBDAddress address;
   if(false == _objectPathManagerIf->getAddress4ObjectPath(address, device))
   {
      // should never happen
      return;
   }

   _callback->updateEnableAvpPause(address, enableAvpPause);

   // check if all data is complete
   checkForDataComplete(bts2IpcMsgList, bts2AppMsgList, messageItem, *_callback, address);
}

void DeviceManagerGenivi::updatePowerMode(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSBtPowerMode mode, IN const BTSIpcCommonErrorCode errorCode, IN const bool response)
{
   (void)(response);

   ETG_TRACE_USR3((" updatePowerMode: device=%s", device.c_str()));

   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   if(BTS_IPC_SUCCESS != errorCode)
   {
      // we have to ignore
      return;
   }

   BTSBDAddress address;
   if(false == _objectPathManagerIf->getAddress4ObjectPath(address, device))
   {
      // should never happen
      return;
   }

   _callback->updatePowerMode(address, mode);

   // check if all data is complete
   checkForDataComplete(bts2IpcMsgList, bts2AppMsgList, messageItem, *_callback, address);
}

void DeviceManagerGenivi::updateDbusServiceAvailability(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSDbusServiceAvailability availability)
{
   ETG_TRACE_USR3((" updateDbusServiceAvailability: device=%s", device.c_str()));

   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   BTSBDAddress address;
   if(false == _objectPathManagerIf->getAddress4ObjectPath(address, device))
   {
      // device object path is created before proxy is created => therefore this error should never happen
      return;
   }

   // set availability
   _serviceStatusDevice.setAvailability(address, availability);
   const bool available(BTS_DBUS_SERVICE_AVAILABLE == availability);

   // TODO: check for BTS_DBUS_SERVICE_NOT_AVAILABLE needed

   // get secondary device data information and set information
   DeviceManagerData& data = _callback->getEntry(address);
   setDbusServiceAvailability(data.secondaryInfo, available);

   // check if all data is complete
   checkForDataComplete(bts2IpcMsgList, bts2AppMsgList, messageItem, *_callback, address);

   // request major data in case of service is available
   if(true == available)
   {
      createDeviceGetDataMsg(bts2IpcMsgList, address, device);
   }
}

void DeviceManagerGenivi::handleDeviceDisconnectResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSIpcCommonErrorCode errorCode)
{
   ETG_TRACE_USR3((" handleDeviceDisconnectResult: device=%s", device.c_str()));

   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   // TODO: double check device object path removal
   BTSBDAddress address;
   if(false == _objectPathManagerIf->getAddress4ObjectPath(address, device))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   BTSRequestResult result(BTS_REQ_FAILED);

   if((BTS_IPC_SUCCESS == errorCode) || (BTS_IPC_NOT_CONNECTED == errorCode))
   {
      // disconnect request is successful
      result = BTS_REQ_SUCCESS;
   }

   _callback->disconnectResult(bts2IpcMsgList, bts2AppMsgList, messageItem, address, result);
}

void DeviceManagerGenivi::handleCreateDeviceResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSObjectPath& device, IN const BTSIpcCommonErrorCode errorCode)
{
   (void)(device);

   ETG_TRACE_USR3((" handleCreateDeviceResult: address=%s", address.c_str()));

   FW_IF_NULL_PTR_RETURN(_callback);
   FW_IF_NULL_PTR_RETURN(_objectPathManagerIf);

   BTSRequestResult result(BTS_REQ_FAILED);

   if((BTS_IPC_SUCCESS == errorCode) || (BTS_IPC_ALREADY_CREATED == errorCode))
   {
      // create request is successful
      result = BTS_REQ_SUCCESS;
   }

   _callback->createResult(bts2IpcMsgList, bts2AppMsgList, messageItem, address, result);
}

void DeviceManagerGenivi::handleRemoveDeviceResult(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode errorCode)
{
   ETG_TRACE_USR3((" handleRemoveDeviceResult"));

   FW_IF_NULL_PTR_RETURN(_callback);

   BTSRequestResult result(BTS_REQ_FAILED);

   if(BTS_IPC_SUCCESS == errorCode)
   {
      // remove request is successful
      result = BTS_REQ_SUCCESS;
   }

   _callback->removeResult(bts2IpcMsgList, bts2AppMsgList, messageItem, result);
}

App2Bts_BaseMessage* DeviceManagerGenivi::getApp2BtsWorkingMessage(IN const BTSBDAddress& address)
{
   FW_IF_NULL_PTR_RETURN_NULL(_callback);

   return _callback->getApp2BtsWorkingMessage(address);
}

void DeviceManagerGenivi::createDeviceDisconnectMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSObjectPath& device) const
{
   Bts2Ipc_DeviceDisconnect* msg = ptrNew_Bts2Ipc_DeviceDisconnect();
   if(0 != msg)
   {
      msg->setBDAddress(address);
      msg->setDevice(device);

      bts2IpcMsgList.push_back(msg);
   }
}

void DeviceManagerGenivi::createDeviceCreateMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& address) const
{
   Bts2Ipc_CreateDevice* msg = ptrNew_Bts2Ipc_CreateDevice();
   if(0 != msg)
   {
      msg->setBDAddress(address);
      msg->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg);
   }
}

void DeviceManagerGenivi::createDeviceRemoveMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSObjectPath& device) const
{
   Bts2Ipc_RemoveDevice* msg = ptrNew_Bts2Ipc_RemoveDevice();
   if(0 != msg)
   {
      msg->setBDAddress(address);
      msg->setDevice(device);
      msg->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg);
   }
}

void DeviceManagerGenivi::createSetEnableAvpPauseMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSObjectPath& device, IN const bool enableAvpPause) const
{
   Bts2Ipc_SetDeviceEnableAvpPause* msg = ptrNew_Bts2Ipc_SetDeviceEnableAvpPause();
   if(0 != msg)
   {
      msg->setBDAddress(address);
      msg->setDevice(device);
      msg->setEnableAvpPause(enableAvpPause);
      msg->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg);
   }
}

void DeviceManagerGenivi::createDeviceGetDataMsg(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSObjectPath& device) const
{
   Bts2Ipc_GetDeviceAlias* msg1 = ptrNew_Bts2Ipc_GetDeviceAlias();
   if(0 != msg1)
   {
      msg1->setBDAddress(address);
      msg1->setDevice(device);
      msg1->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg1);
   }

   Bts2Ipc_GetDeviceConnected* msg2 = ptrNew_Bts2Ipc_GetDeviceConnected();
   if(0 != msg2)
   {
      msg2->setBDAddress(address);
      msg2->setDevice(device);
      msg2->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg2);
   }

   Bts2Ipc_GetDeviceMode* msg3 = ptrNew_Bts2Ipc_GetDeviceMode();
   if(0 != msg3)
   {
      msg3->setBDAddress(address);
      msg3->setDevice(device);
      msg3->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg3);
   }

   Bts2Ipc_GetDeviceRole* msg4 = ptrNew_Bts2Ipc_GetDeviceRole();
   if(0 != msg4)
   {
      msg4->setBDAddress(address);
      msg4->setDevice(device);
      msg4->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg4);
   }

   Bts2Ipc_GetDeviceEnableAvpPause* msg5 = ptrNew_Bts2Ipc_GetDeviceEnableAvpPause();
   if(0 != msg5)
   {
      msg5->setBDAddress(address);
      msg5->setDevice(device);
      msg5->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg5);
   }
}

void DeviceManagerGenivi::setDbusServiceAvailability(OUT unsigned int& secondaryData, IN const bool enable) const
{
   if(true == enable)
   {
      secondaryData = 1;
   }
   else
   {
      secondaryData = 0;
   }
}

bool DeviceManagerGenivi::getDbusServiceAvailability(IN const unsigned int secondaryData) const
{
   return (0 != secondaryData);
}

void DeviceManagerGenivi::checkForDataComplete(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN IDeviceManagerCallback& callback, IN const BTSBDAddress& address) const
{
   // get data entry
   DeviceManagerData& data = callback.getEntry(address);

   // get stored availability information
   const bool oldAvailable(data.info.getBit(DeviceManagerData::DEVICE_AVAILABLE));

   // get new availability information
   bool newAvailable(true);

   if(false == data.info.getBit(DeviceManagerData::DEVICE_ADDED))
   {
      newAvailable = false;
   }
   else if(false == getDbusServiceAvailability(data.secondaryInfo))
   {
      newAvailable = false;
   }
   else if(false == data.info.getBit(DeviceManagerData::ACL_STATUS_UPDATED))
   {
      newAvailable = false;
   }
   else if(false == data.info.getBit(DeviceManagerData::NAME_UPDATED))
   {
      newAvailable = false;
   }
   else if(false == data.info.getBit(DeviceManagerData::ROLE_UPDATED))
   {
      newAvailable = false;
   }
   else if(false == data.info.getBit(DeviceManagerData::MODE_UPDATED))
   {
      newAvailable = false;
   }
   else if(false == data.info.getBit(DeviceManagerData::ENABLEAVPPAUSE_UPDATED))
   {
      newAvailable = false;
   }

   // indicate only changed states
   if(oldAvailable != newAvailable)
   {
      if(true == newAvailable)
      {
         callback.deviceAvailable(bts2IpcMsgList, bts2AppMsgList, messageItem, address);
      }
      else
      {
         callback.deviceUnavailable(bts2IpcMsgList, bts2AppMsgList, messageItem, address);
      }
   }
}

} //genivi
} //btstackif
