/**
 * @file BasicConfigurationGenivi.cpp
 *
 * @par SW-Component
 * State machine for basic configuration
 *
 * @brief Implementation of Genivi basic configuration state machine.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of Genivi basic configuration state machine.
 */

#include "BasicConfigurationGenivi.h"
#include "IBasicConfigurationCallback.h"
#include "IBasicControl.h"
#include "Bts2Ipc_MessageWrapper_GEN.h"
#include "Ipc2Bts_MessageWrapper_GEN.h"
#include "cc_dbus_if/EvolutionGeniviDbusParser.h"
#include "EvolutionGeniviUtils.h"

#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/BasicConfigurationGenivi.cpp.trc.h"
#endif
#endif

namespace btstackif {
namespace genivi {

BasicConfigurationGenivi::BasicConfigurationGenivi() :
_callback(0),
_controlIf(0)
{
}

BasicConfigurationGenivi::~BasicConfigurationGenivi()
{
   _callback = 0;
   _controlIf = 0;
}

void BasicConfigurationGenivi::reset(void)
{
}

void BasicConfigurationGenivi::setCallback(IN IBasicConfigurationCallback* callback)
{
   _callback = callback;

   FW_NORMAL_ASSERT(0 != _callback);
}

void BasicConfigurationGenivi::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_NORMAL_ASSERT(0 != _controlIf);
}

void BasicConfigurationGenivi::getConfigurationParameter(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSConfigurationParameter configurationParameter)
{
   (void)(bts2AppMsgList);

   switch(configurationParameter)
   {
      case BTS_CONFIG_PARAM_LOCAL_SERVICES:
         createGetAdapterUuidsMsg(bts2IpcMsgList);
         break;
      case BTS_CONFIG_PARAM_PAGE_TIMEOUT:
      case BTS_CONFIG_PARAM_LAST:
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }
}

void BasicConfigurationGenivi::setConfigurationParameter(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSConfigurationParameter configurationParameter, IN const uint64_t data)
{
   (void)(bts2AppMsgList);

   switch(configurationParameter)
   {
      case BTS_CONFIG_PARAM_LOCAL_SERVICES:
         createSetAdapterUuidsMsg(bts2IpcMsgList, data);
         break;
      case BTS_CONFIG_PARAM_PAGE_TIMEOUT:
      case BTS_CONFIG_PARAM_LAST:
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }
}

void BasicConfigurationGenivi::setConfigurationParameter(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSConfigurationParameter configurationParameter, IN const ::std::string& data)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(configurationParameter);
   (void)(data);

   FW_NORMAL_ASSERT_ALWAYS();
}

void BasicConfigurationGenivi::sendVirtualConfigurationParameterUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSConfigurationParameter configurationParameter, IN const uint64_t data, IN const BTSIpcCommonErrorCode errorCode)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);

   FW_IF_NULL_PTR_RETURN(_controlIf);

   Ipc2Bts_BaseMessage* msg(0);

   switch(configurationParameter)
   {
      case BTS_CONFIG_PARAM_LOCAL_SERVICES:
         msg = createVirtualAdapterUuidsUpdateMsg(data, errorCode);
         break;
      case BTS_CONFIG_PARAM_PAGE_TIMEOUT:
      case BTS_CONFIG_PARAM_LAST:
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }

   if(0 != msg)
   {
      _controlIf->sendInternalIpc2BtsMessage(msg);
   }
}

void BasicConfigurationGenivi::sendVirtualConfigurationParameterUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSConfigurationParameter configurationParameter, IN const ::std::string& data, IN const BTSIpcCommonErrorCode errorCode)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(configurationParameter);
   (void)(data);
   (void)(errorCode);

   FW_NORMAL_ASSERT_ALWAYS();
}

IBasicConfigurationRequest* BasicConfigurationGenivi::getRequestIf(void)
{
   return this;
}

void BasicConfigurationGenivi::handleAdapterUuidsUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSUuidList& uuidList, IN const BTSIpcCommonErrorCode errorCode, IN const bool response)
{
   (void)(response);

   FW_IF_NULL_PTR_RETURN(_callback);

   unsigned short int data(0);
   convertUuids2Services(data, uuidList);

   _callback->updateConfigurationParameter(bts2IpcMsgList, bts2AppMsgList, messageItem, BTS_CONFIG_PARAM_LOCAL_SERVICES, data, (BTS_IPC_SUCCESS == errorCode));
}

void BasicConfigurationGenivi::createGetAdapterUuidsMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList) const
{
   Bts2Ipc_GetAdapterUuids* msg = ptrNew_Bts2Ipc_GetAdapterUuids();
   if(0 != msg)
   {
      msg->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg);
   }
}

void BasicConfigurationGenivi::createSetAdapterUuidsMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const uint64_t data) const
{
   Bts2Ipc_SetAdapterUuids* msg = ptrNew_Bts2Ipc_SetAdapterUuids();
   if(0 != msg)
   {
      convertServices2Uuids(msg->getUuidsMutable(), (unsigned short int)data);
      msg->setResponseMessageFlag(true);

      bts2IpcMsgList.push_back(msg);
   }
}

Ipc2Bts_BaseMessage* BasicConfigurationGenivi::createVirtualAdapterUuidsUpdateMsg(IN const uint64_t data, IN const BTSIpcCommonErrorCode errorCode) const
{
   Ipc2Bts_AdapterUuidsUpdate* msg = ptrNew_Ipc2Bts_AdapterUuidsUpdate();
   if(0 != msg)
   {
      convertServices2Uuids(msg->getUuidsMutable(), (unsigned short int)data);
      msg->setIpcCommonErrorCode(errorCode);
   }

   return msg;
}

void BasicConfigurationGenivi::convertServices2Uuids(OUT BTSUuidList& uuids, IN const unsigned short int data) const
{
   // convert to UUIDs for local side
   BTSSupportedServices services;
   services.setData(data);
   ::ccdbusif::evolution::EvolutionGeniviDbusParser parser;

   if(true == services.getBit(BTS_SUPP_SRV_HFP))
   {
      uuids.push_back(parser.getUuid2String(::ccdbusif::evolution::UUID_HANDSFREE));
   }

   if((true == services.getBit(BTS_SUPP_SRV_A2DP)) || (true == services.getBit(BTS_SUPP_SRV_AVRCP)))
   {
      uuids.push_back(parser.getUuid2String(::ccdbusif::evolution::UUID_AUDIOSINK));
      uuids.push_back(parser.getUuid2String(::ccdbusif::evolution::UUID_A_V_REMOTECONTROL));
   }

   if(true == services.getBit(BTS_SUPP_SRV_PBAP))
   {
      uuids.push_back(parser.getUuid2String(::ccdbusif::evolution::UUID_PHONEBOOK_ACCESS_PCE));
   }

   if(true == services.getBit(BTS_SUPP_SRV_MAP))
   {
      uuids.push_back(parser.getUuid2String(::ccdbusif::evolution::UUID_MESSAGE_NOTIFICATION_SERVER));
   }

   if(true == services.getBit(BTS_SUPP_SRV_DUN))
   {
      uuids.push_back(parser.getUuid2String(::ccdbusif::evolution::UUID_DIALUPNETWORKING));
   }

   if(true == services.getBit(BTS_SUPP_SRV_PAN))
   {
      uuids.push_back(parser.getUuid2String(::ccdbusif::evolution::UUID_PANU));
   }

   if(true == services.getBit(BTS_SUPP_SRV_SPP))
   {
      uuids.push_back(parser.getUuid2String(::ccdbusif::evolution::UUID_SERIALPORT));
   }

   if(true == services.getBit(BTS_SUPP_SRV_DID))
   {
      uuids.push_back(parser.getUuid2String(::ccdbusif::evolution::UUID_PNPINFORMATION));
   }
}

void BasicConfigurationGenivi::convertUuids2Services(OUT unsigned short int& data, IN const BTSUuidList& uuids) const
{
   BTSSupportedServices serviceList;
   ::std::string convertedUuid;

   for(size_t i = 0; i < uuids.size(); i++)
   {
      BTSSupportedServicesBit supportedServicesBit(convertUuid2SupportedService(convertedUuid, uuids[i]));

      if(BTS_SUPP_SRV_LAST != supportedServicesBit)
      {
         serviceList.setBit(supportedServicesBit);
      }

      convertedUuid.clear();
   }

   data = serviceList.getData();
}

} //genivi
} //btstackif
