/**
 * @file StartupGenivi.h
 *
 * @par SW-Component
 * State machine for startup
 *
 * @brief Implementation of Genivi startup state machine.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of Genivi startup state machine.
 */

#ifndef _STARTUP_GENIVI_H_
#define _STARTUP_GENIVI_H_

#include "IStartupRequest.h"
#include "IStartupGenivi.h"
#include "IExtendedTimeoutHandler.h"
#include "EvolutionGeniviStackTypes.h"
#include "DbusServiceAvailability.h"
#include "ExtendedTimerEntry.h"

namespace btstackif {
namespace genivi {

/**
 * Startup class.
 */
class StartupGenivi : public IStartupRequest, public IStartupGenivi, public IExtendedTimeoutHandler
{
public:
   /**
    * Default constructor.
    */
   StartupGenivi();

   /**
    * Destructor.
    */
   virtual ~StartupGenivi();

   virtual void reset(void);

   virtual void setCallback(IN IStartupCallback* callback);

   virtual void setControlIf(IN IBasicControl* control);

   virtual void setTimerPoolIf(IN ITimerPool* timerPool);

   virtual void setAgentCapability(IN const BTSAgentCapability capability);

   virtual void checkAvailableServices(void);

   virtual void getIntrospections(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList);

   virtual void setHfpDeviceCapabilitiesGeniviIf(IN IHfpDeviceCapabilitiesGenivi* hfpDeviceCapabilitiesGeniviIf);

   virtual IStartupRequest* getRequestIf(void);

   virtual void addToServiceGroup(IN const BTSUserMode userMode, IN DbusServiceStatus& status, IN const bool enabled);

   virtual void addInterfaces(OUT ::std::vector< BTSDbusInterfaceItem >& dbusInterfaces, IN const BTSUserMode userMode);

   virtual bool updateDbusServiceAvailability(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem,
             IN const BTSGenDbusServiceInterface interface, IN const BTSDbusServiceAvailability availability, IN const BTSCommonEnumClass busType, IN const BTSBusName& busName, IN const BTSObjectPath& objPath);

   virtual void checkNewGroupAvailability(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool checkGroupAvailability);

   virtual void handleAdapterProperties(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDbusPropertyList& properties, IN const BTSIpcCommonErrorCode errorCode);

   virtual void handleAdapterInitialization(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem);

   virtual void handleRegisterAgentResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode errorCode);

   virtual void handleExtendedTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSTimerId timerId);

private:
   enum
   {
      TIMEOUT_GET_ALL_ADAPTER = 500, /**< timeout for retry of requesting GetAll for adapter interface */
      TIMEOUT_INIT_ADAPTER = 90, /**< timeout for waiting for AdapterInitialization signal */
      TIMEOUT_REPEAT_REGISTER_AGENT = 90, /**< timeout for retry of register agent request */
      MAX_NUMBER_REGISTER_AGENT_REQUESTS = 5 /**< limitation for register agent requests */
   };

   IStartupCallback* _callback; /**< callback interface */
   IBasicControl* _controlIf; /**< basic control interface */
   ITimerPool* _timerPoolIf; /**< timer pool interface */
   BTSAgentCapability _capability; /**< agent capability */
   IHfpDeviceCapabilitiesGenivi* _hfpDeviceCapabilitiesGeniviIf; /**< HFP device capabilities interface */

   bool _getAllReceived; /**< flag for data received */
   bool _getAllTimerUsed; /**< flag for timer usage */
   bool _getAllSent; /**< flag for sent GetAll request */
   ExtendedTimerEntry _timerGetAll; /**< timer for requesting GetAll */

   bool _initReceived; /**< flag for adapter initialization received */
   bool _initTimerUsed; /**< flag for timer usage */
   ExtendedTimerEntry _timerInit; /**< timer for adapter initialization */

   bool _registerAgentSent; /**< flag for sent register agent request */
   ExtendedTimerEntry _registerAgentTimer; /**< timer for register agent */
   unsigned int _nmbRegisterAgentRequests; /**< number of register agent request sent to stack */

   // group service
   DbusServiceGroupStatus _serviceGroupGeneral; /**< service group */
   // special service to handle disabled BT stack communication
   DbusServiceStatus _serviceStatusDisabledBtStackComm; /**< service to handle disabled BT stack communication */
   // general services
   DbusServiceStatus _serviceStatusObjectManagerSystem; /**< service for object manager system bus */
   DbusServiceStatus _serviceStatusObjectManagerSession; /**< service for object manager session bus */
   DbusServiceStatus _serviceStatusIntrospectable; /**< service for introspectable functionality */
   DbusServiceStatus _serviceStatusPropertiesGetAll; /**< service for properties */
   DbusServiceStatus _serviceStatusGenInitialization; /**< service for general initialization */
   DbusServiceStatus _serviceStatusHliInitialization; /**< service for Bluetooth stack initialization */
   // specific services
   DbusServiceStatus _serviceStatusAdapter; /**< service for adapter interface */
   DbusServiceStatus _serviceStatusAgent; /**< service for agent interface */
   DbusServiceStatus _serviceStatusAgentManager; /**< service for agent manager interface */
   DbusServiceStatus _serviceStatusDM; /**< service for DM interface */
   DbusServiceStatus _serviceStatusTrace; /**< service for trace interface */
   DbusServiceStatus _serviceStatusGeniviTrace; /**< service for Genivi trace interface */
   DbusServiceStatus _serviceStatusDID; /**< service for DID interface */
   DbusServiceStatus _serviceStatusObexClient; /**< service for obex client interface */

   bool _testSimulateDbusStartupError; /**< flag for simulating DBus startup error TODO: test interface */
   bool _testSimulateEvolutionStartupError; /**< flag for simulating Evolution startup error TODO: test interface */

   void setDataReceived(IN const bool enable);

   bool getDataReceived(void) const;

   void setAdapterInitDone(IN const bool enable);

   bool getAdapterInitDone(void) const;

   void startGetAllTimer(void);

   void stopGetAllTimer(void);

   void checkSendingGetAll(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const bool objectManagerAvailable, IN const bool adapterAvailable);

   void startInitTimer(void);

   void stopInitTimer(void);

   void checkSendingRegisterAgent(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const bool agentManagerAvailable, IN const bool genInitializationAvailable, IN const bool hliInitializationAvailable);

   void startRepeatTimer(void);

   void createGetAllPropertiesConnectionMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList) const;

   void createRegisterAgentMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList) const;

   void createGetIntrospectionConnectionMsg(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList) const;

   void startTimer(IN ExtendedTimerEntry& timer, IN const BTSTimeValue timeout);

   void stopTimer(IN ExtendedTimerEntry& timer) const;

   void releaseTimer(IN ExtendedTimerEntry& timer) const;
};

} //genivi
} //btstackif

#endif //_STARTUP_GENIVI_H_
