/**
 * @file ServiceSearchGenivi.h
 *
 * @par SW-Component
 * State machine for service search
 *
 * @brief Implementation of Genivi service search state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of Genivi service search state machine.
 */

#ifndef _SERVICE_SEARCH_GENIVI_H_
#define _SERVICE_SEARCH_GENIVI_H_

#include "IServiceSearchRequest.h"
#include "IServiceSearchGenivi.h"
#include "EvolutionGeniviStackTypes.h"

namespace btstackif {
namespace genivi {

// class forward declarations
class IObjectPathManagerGenivi;

/**
 * ServiceSearch class.
 */
class ServiceSearchGenivi : public IServiceSearchRequest, public IServiceSearchGenivi
{
public:
   /**
    * Default constructor.
    */
   ServiceSearchGenivi();

   /**
    * Destructor.
    */
   virtual ~ServiceSearchGenivi();

   virtual void reset(void);

   virtual void setCallback(IN IServiceSearchCallback* callback);

   virtual void setControlIf(IN IBasicControl* control);

   virtual void startSearch(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSSearchType searchType);

   virtual void cancelSearch(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address);

   virtual void sendVirtualFailedResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address);

   virtual void getDeviceUuids(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address);

   virtual void getServiceVersion(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol);

   virtual void getDeviceVendorId(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address);

   virtual void getDeviceVendorIdSource(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address);

   virtual void getDeviceProductId(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address);

   virtual void getDeviceVersion(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address);

   virtual void getPbapInfo(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address);

   virtual void getMapInfo(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address);

   virtual void setObjectPathManagerIf(IN IObjectPathManagerGenivi* objectPathManager);

   virtual void handleDiscoverServicesResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, OUT bool& connectFailed, OUT BTSBDAddress& address, IN const BTSObjectPath& device, IN const BTSIpcCommonErrorCode errorCode);

   virtual void handleCancelDiscoveryResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const BTSIpcCommonErrorCode errorCode);

   virtual void setSdpSearchActiveAfterPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const bool enable, IN const bool success);

   virtual bool startSdpSearchAfterPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device, IN const bool success);

   virtual bool startSdpSearchAfterPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSObjectPath& device);

   virtual void setPairingStatus(IN const BTSObjectPath& device, IN const bool success);

   virtual void updateDeviceUuids(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSObjectPath& device, IN const BTSSupportedServices& supportedServices, IN const BTSUuidList& uuidList);

   virtual void updateServiceVersion(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSObjectPath& device, IN const BTSProfileVersion version);

   virtual void updateDeviceVendorId(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSObjectPath& device, IN const BTSVendorId vendorId);

   virtual void updateDeviceVendorIdSource(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSObjectPath& device, IN const BTSVendorIdSource vendorIdSource);

   virtual void updateDeviceProductId(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSObjectPath& device, IN const BTSProductId productId);

   virtual void updateDeviceVersion(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSObjectPath& device, IN const BTSVersion version);

   virtual void updatePbapInfo(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSObjectPath& device, IN const BTSPbapSdpRecord& record, IN const bool valid);

   virtual void updateMapInfo(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSIpcCommonErrorCode result, IN const bool response, IN const BTSObjectPath& device, IN const BTSMapSdpRecordList& recordList, IN const bool valid);

private:
   IServiceSearchCallback* _callback; /**< callback interface */
   IBasicControl* _controlIf; /**< basic control interface */
   IObjectPathManagerGenivi* _objectPathManagerIf; /**< object path manager interface */

   void createDiscoverServicesRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSSearchType searchType, IN const BTSObjectPath& device, IN const bool responseFlag = false) const;

   void createCancelDiscoveryRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSObjectPath& device, IN const bool responseFlag = false) const;

   void createGetUuidRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSObjectPath& device, IN const bool responseFlag = false) const;

   void createGetServiceVersionRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSObjectPath& service, IN const bool responseFlag = false) const;

   void createGetDeviceVendorIdRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSObjectPath& device, IN const bool responseFlag = false) const;

   void createGetDeviceVendorIdSourceRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSObjectPath& device, IN const bool responseFlag = false) const;

   void createGetDeviceProductIdRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSObjectPath& device, IN const bool responseFlag = false) const;

   void createGetDeviceVersionRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSObjectPath& device, IN const bool responseFlag = false) const;

   void createGetPbapInfoRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSObjectPath& device, IN const bool responseFlag = false) const;

   void createGetMapInfoRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, IN const BTSBDAddress& address, IN const BTSObjectPath& device, IN const bool responseFlag = false) const;
};

} //genivi
} //btstackif

#endif //_SERVICE_SEARCH_GENIVI_H_
